import "package:fade_indexed_stack/fade_indexed_stack.dart";
import "package:flutter/foundation.dart";
import "package:flutter/material.dart";
import "package:flutter_animate/flutter_animate.dart";
import "package:logging/logging.dart";
import "package:photos/generated/l10n.dart";
import "package:photos/models/search/album_search_result.dart";
import "package:photos/models/search/generic_search_result.dart";
import "package:photos/models/search/index_of_indexed_stack.dart";
import "package:photos/models/search/search_result.dart";
import "package:photos/models/search/search_types.dart";
import "package:photos/service_locator.dart";
import "package:photos/services/wrapped/wrapped_service.dart";
import "package:photos/states/all_sections_examples_state.dart";
import "package:photos/theme/ente_theme.dart";
import "package:photos/ui/activity/activity_banner.dart";
import "package:photos/ui/common/loading_widget.dart";
import "package:photos/ui/viewer/search/result/no_result_widget.dart";
import "package:photos/ui/viewer/search/search_suggestions.dart";
import "package:photos/ui/viewer/search/tab_empty_state.dart";
import "package:photos/ui/viewer/search_tab/albums_section.dart";
import "package:photos/ui/viewer/search_tab/file_type_section.dart";
import "package:photos/ui/viewer/search_tab/locations_section.dart";
import "package:photos/ui/viewer/search_tab/magic_section.dart";
import "package:photos/ui/viewer/search_tab/people_section.dart";
import "package:photos/ui/wrapped/wrapped_discovery_section.dart";

class SearchTab extends StatefulWidget {
  const SearchTab({super.key});

  @override
  State<SearchTab> createState() => _SearchTabState();
}

class _SearchTabState extends State<SearchTab> {
  late int index;
  final indexOfStackNotifier = IndexOfStackNotifier();

  @override
  void initState() {
    super.initState();
    index = indexOfStackNotifier.index;
    indexOfStackNotifier.addListener(indexNotifierListener);
  }

  void indexNotifierListener() {
    setState(() {
      index = indexOfStackNotifier.index;
    });
  }

  @override
  void dispose() {
    indexOfStackNotifier.removeListener(indexNotifierListener);
    indexOfStackNotifier.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return AllSectionsExamplesProvider(
      child: FadeIndexedStack(
        lazy: false,
        duration: const Duration(milliseconds: 150),
        index: index,
        children: const [
          AllSearchSections(),
          SearchSuggestionsWidget(),
          NoResultWidget(),
        ],
      ),
    );
  }
}

class AllSearchSections extends StatefulWidget {
  const AllSearchSections({super.key});

  @override
  State<AllSearchSections> createState() => _AllSearchSectionsState();
}

class _AllSearchSectionsState extends State<AllSearchSections> {
  final Logger _logger = Logger('_AllSearchSectionsState');

  @override
  Widget build(BuildContext context) {
    final searchTypes = SectionType.values.toList(growable: true);

    return Padding(
      padding: const EdgeInsets.only(top: 8),
      child: Stack(
        children: [
          FutureBuilder<List<List<SearchResult>>>(
            future: InheritedAllSectionsExamples.of(context)
                .allSectionsExamplesFuture,
            builder: (context, snapshot) {
              if (snapshot.hasData && snapshot.data!.isNotEmpty) {
                if (snapshot.data!.every((element) => element.isEmpty)) {
                  return const Padding(
                    padding: EdgeInsets.only(bottom: 72),
                    child: SearchTabEmptyState(),
                  );
                }
                if (snapshot.data!.length != searchTypes.length) {
                  return Padding(
                    padding: const EdgeInsets.only(bottom: 72),
                    child: Text(
                      AppLocalizations.of(context).searchSectionsLengthMismatch(
                        snapshotLength: snapshot.data!.length,
                        searchLength: searchTypes.length,
                      ),
                    ),
                  );
                }
                return ListView.builder(
                  padding: const EdgeInsets.only(bottom: 180),
                  physics: const BouncingScrollPhysics(),
                  itemCount: searchTypes.length,
                  itemBuilder: (context, index) {
                    final sectionType = searchTypes[index];
                    switch (sectionType) {
                      case SectionType.face:
                        if (!flagService.hasGrantedMLConsent) {
                          return const SizedBox.shrink();
                        }
                        return PeopleSection(
                          examples: snapshot.data!.elementAt(index)
                              as List<GenericSearchResult>,
                        );
                      case SectionType.album:
                        return AlbumsSection(
                          snapshot.data!.elementAt(index)
                              as List<AlbumSearchResult>,
                        );
                      case SectionType.ritual:
                        return const _RitualsDiscoverySection();
                      case SectionType.wrapped:
                        return ValueListenableBuilder<WrappedEntryState>(
                          valueListenable: wrappedService.stateListenable,
                          builder: (
                            BuildContext context,
                            WrappedEntryState state,
                            Widget? child,
                          ) {
                            if (!wrappedService.shouldShowDiscoveryEntry) {
                              return const SizedBox.shrink();
                            }
                            return WrappedDiscoverySection(state: state);
                          },
                        );
                      case SectionType.location:
                        return LocationsSection(
                          snapshot.data!.elementAt(index)
                              as List<GenericSearchResult>,
                        );
                      case SectionType.contacts:
                        return const SizedBox.shrink();
                      case SectionType.fileTypesAndExtension:
                        return FileTypeSection(
                          snapshot.data!.elementAt(index)
                              as List<GenericSearchResult>,
                        );
                      case SectionType.magic:
                        return MagicSection(
                          snapshot.data!.elementAt(index)
                              as List<GenericSearchResult>,
                        );
                    }
                  },
                )
                    .animate(
                      delay: const Duration(milliseconds: 150),
                    )
                    .slide(
                      begin: const Offset(0, -0.015),
                      end: const Offset(0, 0),
                      duration: const Duration(milliseconds: 300),
                      curve: Curves.easeOut,
                    )
                    .fadeIn(
                      duration: const Duration(milliseconds: 150),
                      curve: Curves.easeOut,
                    );
              } else if (snapshot.hasError) {
                _logger.severe(
                  'Failed to load sections: ',
                  snapshot.error,
                  snapshot.stackTrace,
                );
                if (kDebugMode) {
                  return Padding(
                    padding: const EdgeInsets.only(bottom: 72),
                    child: Text(
                      AppLocalizations.of(context).error +
                          ': ${snapshot.error}',
                    ),
                  );
                }
                return const Padding(
                  padding: EdgeInsets.only(bottom: 72),
                  child: EnteLoadingWidget(),
                );
              } else {
                return const Padding(
                  padding: EdgeInsets.only(bottom: 72),
                  child: EnteLoadingWidget(),
                );
              }
            },
          ),
          ValueListenableBuilder(
            valueListenable:
                InheritedAllSectionsExamples.of(context).isDebouncingNotifier,
            builder: (context, value, _) {
              return value
                  ? const EnteLoadingWidget(
                      alignment: Alignment.topRight,
                    )
                  : const SizedBox.shrink();
            },
          ),
        ],
      ),
    );
  }
}

class _RitualsDiscoverySection extends StatelessWidget {
  const _RitualsDiscoverySection();

  @override
  Widget build(BuildContext context) {
    if (!flagService.ritualsFlag) {
      return const SizedBox.shrink();
    }
    final textTheme = getEnteTextTheme(context);
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Padding(
          padding: const EdgeInsets.fromLTRB(12, 8, 12, 0),
          child: Text(
            AppLocalizations.of(context).ritualsTitle,
            style: textTheme.largeBold,
          ),
        ),
        const SizedBox(height: 4),
        const ActivityBanner(),
      ],
    );
  }
}
