import 'package:emoji_picker_flutter/emoji_picker_flutter.dart';
import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';

// Use for golden tests, helpful in debugging
// await expectLater(
//   find.byType(MaterialApp),
//   matchesGoldenFile('overlay.png'),
// );

void main() {
  group('EmojiPicker Tests', () {
    testWidgets('Should allow user to select an emoji',
        (WidgetTester tester) async {
      final _controller = TextEditingController();
      Emoji? _emojiSelected;
      Category? _categorySelected;

      // Build our app and trigger a frame.
      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: EmojiPicker(
              textEditingController: _controller,
              onEmojiSelected: (category, emoji) {
                _emojiSelected = emoji;
                _categorySelected = category;
              },
              config: const Config(
                  height: 256,
                  categoryViewConfig: CategoryViewConfig(
                    recentTabBehavior: RecentTabBehavior.NONE,
                  )),
            ),
          ),
        ),
      );

      // Wait for the emojis to load if they are being loaded asynchronously
      await tester.pumpAndSettle();

      // Find an emoji in the picker
      final emoji = find.text('🙂').hitTestable();

      // Verify if we can find the emoji
      expect(emoji, findsOneWidget);

      // Tap on the emoji, this should trigger the selection action
      await tester.tap(emoji);

      // Call pumpAndSettle in case the UI needs to settle after an interaction
      await tester.pumpAndSettle();

      // Check if the emoji is added to the text controller
      expect(_controller.text, contains('🙂'));

      // Check if the emoji been passed to the 'onEmojiSelected' callback
      expect(
          _emojiSelected, equals(const Emoji('🙂', 'Slightly Smiling Face')));

      // Check if the category been passed to the 'onEmojiSelected' callback
      expect(_categorySelected, equals(Category.SMILEYS));
    });

    testWidgets('Should allow to select an emoji with skintone on longPress',
        (WidgetTester tester) async {
      final _controller = TextEditingController();
      final _utils = EmojiPickerUtils();
      final emoji = const Emoji('👍', 'Thumbs Up', hasSkinTone: true);
      Emoji? _emojiSelected;
      Category? _categorySelected;

      // Build our app and trigger a frame.
      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: Padding(
              padding: const EdgeInsets.only(top: 64.0),
              child: EmojiPicker(
                textEditingController: _controller,
                onEmojiSelected: (category, emoji) {
                  _emojiSelected = emoji;
                  _categorySelected = category;
                },
                config: const Config(
                  height: 500,
                  categoryViewConfig: CategoryViewConfig(
                    recentTabBehavior: RecentTabBehavior.NONE,
                  ),
                ),
              ),
            ),
          ),
        ),
      );

      // Wait for the emojis to load if they are being loaded asynchronously
      await tester.pumpAndSettle();

      // Find an emoji in the picker
      final emojiToFind = find.text(emoji.emoji);

      // Scroll until the emoji to be found appears.
      await tester.dragUntilVisible(
        emojiToFind,
        find.byKey(const Key('emojiScrollView')),
        const Offset(0, -300),
      );

      // Verify if we can find the emoji
      expect(emojiToFind, findsOneWidget);

      // Tap on the emoji, this should trigger the skintone overlay
      await tester.longPress(emojiToFind);

      // Call pumpAndSettle in case the UI needs to settle after an interaction
      await tester.pumpAndSettle();

      /// Check if all skin tones are rendered in overlay
      Finder? skinToneVariantToFind;
      for (var i = 0; i < SkinTone.values.length; i++) {
        skinToneVariantToFind =
            find.text(_utils.applySkinTone(emoji, SkinTone.values[i]).emoji);
        // Verify if we can find the skintone variant
        expect(skinToneVariantToFind, findsOneWidget);
      }

      // Tap on the emoji, this should trigger the selection action
      await tester.tap(skinToneVariantToFind!);

      // Check if the emoji is added to the text controller
      expect(_controller.text, contains('👍🏿'));

      // Check if the emoji been passed to the 'onEmojiSelected' callback
      expect(_emojiSelected?.emoji, equals('👍🏿'));
      expect(_emojiSelected?.name, equals('Thumbs Up'));
      expect(_emojiSelected?.hasSkinTone, equals(true));

      // Check if the category been passed to the 'onEmojiSelected' callback
      expect(_categorySelected, equals(Category.SMILEYS));
    });
  });
}
