/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package info.zamojski.soft.towercollector.files.formatters.kml;

import android.text.TextUtils;

import org.stockchart.utils.DoubleUtils;

import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

import info.zamojski.soft.towercollector.model.Cell;
import info.zamojski.soft.towercollector.model.Measurement;
import info.zamojski.soft.towercollector.providers.GeneralCellUtils;
import info.zamojski.soft.towercollector.providers.ICellUtils;
import info.zamojski.soft.towercollector.utils.ApkUtils;

public class KmlExportFormatter implements IKmlFormatter {

    private static final Locale LOCALE = Locale.ENGLISH;

    private static final NumberFormat coordsDoubleFormatter;
    private static final NumberFormat gpsDoubleFormatter;
    private static final SimpleDateFormat exportDateFormatter;

    protected static final NumberFormat intFormatter;

    private static final ICellUtils cellUtils;

    static {
        coordsDoubleFormatter = NumberFormat.getNumberInstance(LOCALE);
        coordsDoubleFormatter.setGroupingUsed(false);
        coordsDoubleFormatter.setMinimumFractionDigits(8);
        coordsDoubleFormatter.setMaximumFractionDigits(12);

        gpsDoubleFormatter = NumberFormat.getNumberInstance(LOCALE);
        gpsDoubleFormatter.setGroupingUsed(false);
        gpsDoubleFormatter.setMinimumFractionDigits(0);
        gpsDoubleFormatter.setMaximumFractionDigits(2);

        intFormatter = NumberFormat.getNumberInstance(LOCALE);
        intFormatter.setParseIntegerOnly(true);
        intFormatter.setGroupingUsed(false);

        exportDateFormatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'", LOCALE);
        exportDateFormatter.setTimeZone(TimeZone.getTimeZone("UTC"));

        cellUtils = new GeneralCellUtils();
    }

    @Override
    public String formatHeader(long firstMeasurementTimestamp, long lastMeasurementTimestamp) {
        String firstMeasurementDateString = formatDate(firstMeasurementTimestamp);
        String lastMeasurementDateString = formatDate(lastMeasurementTimestamp);
        String trackName = String.format(LOCALE, "<name>Sessions taken between %s and %s</name>\r\n", firstMeasurementDateString, lastMeasurementDateString);

        return "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n" +
                "<kml xmlns=\"http://www.opengis.net/kml/2.2\" xmlns:gx=\"http://www.google.com/kml/ext/2.2\" xmlns:kml=\"http://www.opengis.net/kml/2.2\" xmlns:atom=\"http://www.w3.org/2005/Atom\">\r\n" +
                "<Document>\r\n" +
                "  " + trackName +
                String.format(LOCALE, "  <description>File generated by Tower Collector %s on %s</description>\r\n", ApkUtils.getApkVersionName(), ApkUtils.getDeviceName()) +
                "  <Style id=\"defaultIcon\">\r\n" +
                "    <LabelStyle>\r\n" +
                "      <scale>0</scale>\r\n" +
                "    </LabelStyle>\r\n" +
                "    <IconStyle>\r\n" +
                "      <scale>0.5</scale>\r\n" +
                "    </IconStyle>\r\n" +
                "  </Style>\r\n" +
                "  <Style id=\"hoverIcon\">\r\n" +
                "    <IconStyle>\r\n" +
                "      <scale>1.2</scale>\r\n" +
                "    </IconStyle>\r\n" +
                "  </Style>\r\n" +
                "  <Style id=\"lineStyle\">\r\n" +
                "    <LineStyle>\r\n" +
                "      <width>6</width>\r\n" +
                "    </LineStyle>\r\n" +
                "  </Style>\r\n" +
                "  <StyleMap id=\"defaultStyle\">\r\n" +
                "    <Pair>\r\n" +
                "      <key>normal</key>\r\n" +
                "      <styleUrl>#defaultIcon</styleUrl>\r\n" +
                "    </Pair>\r\n" +
                "    <Pair>\r\n" +
                "      <key>highlight</key>\r\n" +
                "      <styleUrl>#hoverIcon</styleUrl>\r\n" +
                "    </Pair>\r\n" +
                "  </StyleMap>\r\n" +
                "  <Folder>\r\n" +
                "    <name>Tracks</name>\r\n" +
                "    <Folder>\r\n" +
                "      " + trackName +
                "      <TimeSpan>\r\n" +
                "        <begin>" + firstMeasurementDateString + "</begin>\r\n" +
                "        <end>" + lastMeasurementDateString + "</end>\r\n" +
                "      </TimeSpan>\r\n";
    }

    @Override
    public String formatEntry(Measurement m) {
        StringBuilder sb = new StringBuilder();
        sb.append("      <Placemark>\r\n");
        sb.append("        <name>");
        for (Cell c : m.getCells()) {
            int mcc = c.getMcc();
            int psc = c.getPsc();
            sb.append(String.format(LOCALE, "(%sMNC %s, LAC %s, CID %s, %sSignal strength %s dBm, Network %s%s), ",
                    (mcc != Cell.UNKNOWN_CID ? String.format(LOCALE, "MCC %s, ", formatInt(mcc)) : ""),
                    formatInt(c.getMnc()),
                    formatInt(c.getLac()),
                    formatLong(c.getCid()),
                    (psc != Cell.UNKNOWN_CID ? String.format(LOCALE, "PSC %s, ", formatInt(psc)) : ""),
                    formatDbmSignal(c.getDbm()),
                    cellUtils.getSystemType(c.getNetworkType()),
                    c.isNeighboring() ? ", neighboring" : ""));
        }
        sb.append(String.format(LOCALE, "(GPS accuracy %s m)</name>\r\n", formatGpsValue(m.getGpsAccuracy())));
        boolean speedAvailable = DoubleUtils.greaterThan(m.getGpsSpeed(), Measurement.GPS_VALUE_NOT_AVAILABLE);
        boolean bearingAvailable = DoubleUtils.greaterThan(m.getGpsBearing(), Measurement.GPS_VALUE_NOT_AVAILABLE);
        if (speedAvailable || bearingAvailable) {
            sb.append("        <description><![CDATA[<table>\r\n");
            if (speedAvailable)
                sb.append(String.format(LOCALE, "          <tr><td>Speed: %s m/s</td></tr>\r\n", formatGpsValue(m.getGpsSpeed())));
            if (bearingAvailable)
                sb.append(String.format(LOCALE, "          <tr><td>Heading: %s</td></tr>\r\n", formatGpsValue(m.getGpsBearing())));
            sb.append("        </table>]]></description>\r\n");
        }
        sb.append("        <TimeStamp>\r\n");
        sb.append(String.format(LOCALE, "          <when>%s</when>\r\n", formatDate(m.getMeasuredAt())));
        sb.append("        </TimeStamp>\r\n");
        sb.append("        <styleUrl>#defaultStyle</styleUrl>\r\n");
        sb.append("        <Point>\r\n");
        boolean altitudeAvailable = !DoubleUtils.equals(m.getGpsAltitude(), Measurement.GPS_VALUE_NOT_AVAILABLE);
        sb.append(String.format(LOCALE, "          <coordinates>%s,%s%s</coordinates>\r\n",
                formatCoordinate(m.getLongitude()),
                formatCoordinate(m.getLatitude()),
                altitudeAvailable ? "," + formatGpsValue(m.getGpsAltitude()) : ""));
        sb.append("        </Point>\r\n");
        sb.append("      </Placemark>\r\n");
        return sb.toString();
    }

    @Override
    public String formatNewSegment(int segmentId) {
        return "      </Folder>\r\n"
                + "      <Folder>\r\n"
                + String.format(LOCALE, "        <name>Segment %s</name>\r\n", formatInt(segmentId));
    }

    @Override
    public String formatLineEntry(Measurement m) {
        boolean altitudeAvailable = !DoubleUtils.equals(m.getGpsAltitude(), Measurement.GPS_VALUE_NOT_AVAILABLE);
        return String.format(LOCALE, "%s,%s%s",
                formatCoordinate(m.getLongitude()),
                formatCoordinate(m.getLatitude()),
                altitudeAvailable ? "," + formatGpsValue(m.getGpsAltitude()) : "");
    }

    @Override
    public String formatLineSegment(List<String> entries) {
        String lineString = TextUtils.join(" ", entries);
        return "          <LineString>\r\n" +
                "            <tessellate>1</tessellate>\r\n" +
                "            <coordinates>\r\n" +
                String.format(LOCALE, "              %s\r\n", lineString) +
                "            </coordinates>\r\n" +
                "          </LineString>\r\n";
    }

    @Override
    public String formatFooter(List<String> segments) {
        return "      <Placemark>\r\n" +
                "        <name>Path</name>\r\n" +
                "        <styleUrl>#lineStyle</styleUrl>\r\n" +
                "        <MultiGeometry>\r\n" +
                TextUtils.join("", segments) +
                "        </MultiGeometry>\r\n" +
                "      </Placemark>\r\n" +
                "    </Folder>\r\n" +
                "  </Folder>\r\n" +
                "</Document>\r\n" +
                "</kml>\r\n";
    }

    private String formatCoordinate(double value) {
        return coordsDoubleFormatter.format(value);
    }

    private String formatGpsValue(double value) {
        return gpsDoubleFormatter.format(value);
    }

    private String formatDbmSignal(int dbm) {
        if (dbm != Cell.UNKNOWN_SIGNAL)
            return String.valueOf(dbm);
        else
            return "";
    }

    private String formatDate(long timestamp) {
        return exportDateFormatter.format(new Date(timestamp));
    }

    private String formatInt(int value) {
        return intFormatter.format(value);
    }

    private String formatLong(long value) {
        return intFormatter.format(value);
    }
}
