/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package info.zamojski.soft.towercollector.files.formatters.gpx;

import info.zamojski.soft.towercollector.files.formatters.gpx.model.HeaderData;
import info.zamojski.soft.towercollector.model.Cell;
import info.zamojski.soft.towercollector.model.Measurement;
import info.zamojski.soft.towercollector.providers.GeneralCellUtils;
import info.zamojski.soft.towercollector.providers.ICellUtils;
import info.zamojski.soft.towercollector.utils.ApkUtils;

import org.stockchart.utils.DoubleUtils;

import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

public class GpxExportFormatter implements IGpxFormatter {

    private static final Locale LOCALE = Locale.ENGLISH;

    private static final NumberFormat coordsDoubleFormatter;
    private static final NumberFormat gpsDoubleFormatter;
    private static final SimpleDateFormat exportDateFormatter;

    protected static final NumberFormat intFormatter;

    private static final ICellUtils cellUtils;

    static {
        coordsDoubleFormatter = NumberFormat.getNumberInstance(LOCALE);
        coordsDoubleFormatter.setGroupingUsed(false);
        coordsDoubleFormatter.setMinimumFractionDigits(8);
        coordsDoubleFormatter.setMaximumFractionDigits(12);

        gpsDoubleFormatter = NumberFormat.getNumberInstance(LOCALE);
        gpsDoubleFormatter.setGroupingUsed(false);
        gpsDoubleFormatter.setMinimumFractionDigits(0);
        gpsDoubleFormatter.setMaximumFractionDigits(2);

        intFormatter = NumberFormat.getNumberInstance(LOCALE);
        intFormatter.setParseIntegerOnly(true);
        intFormatter.setGroupingUsed(false);

        exportDateFormatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'", LOCALE);
        exportDateFormatter.setTimeZone(TimeZone.getTimeZone("UTC"));

        cellUtils = new GeneralCellUtils();
    }

    @Override
    public String formatHeader(HeaderData h) {
        String headerString = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n";
        headerString += String.format(LOCALE, "<gpx version=\"1.1\" creator=\"Tower Collector %s\" xmlns=\"http://www.topografix.com/GPX/1/1\" xmlns:gpxtpx=\"http://www.garmin.com/xmlschemas/TrackPointExtension/v2\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd http://www.garmin.com/xmlschemas/TrackPointExtension/v2 http://www.garmin.com/xmlschemas/TrackPointExtensionv2.xsd\">\r\n",
                ApkUtils.getApkVersionName());
        headerString += "  <metadata>\r\n";
        String trackName = String.format(LOCALE, "Sessions taken between %s and %s",
                formatDate(h.FirstMeasurementTimestamp),
                formatDate(h.LastMeasurementTimestamp));
        headerString += String.format(LOCALE, "    <name>%s</name>\r\n",
                trackName);
        headerString += String.format(LOCALE, "    <desc>File generated by Tower Collector %s on %s</desc>\r\n", ApkUtils.getApkVersionName(), ApkUtils.getDeviceName());
        headerString += String.format(LOCALE, "    <time>%s</time>\r\n",
                formatDate(System.currentTimeMillis()));
        headerString += String.format(LOCALE, "    <bounds minlat=\"%s\" minlon=\"%s\" maxlat=\"%s\" maxlon=\"%s\" />\r\n",
                formatCoordinate(h.Boundaries.getMinLat()),
                formatCoordinate(h.Boundaries.getMinLon()),
                formatCoordinate(h.Boundaries.getMaxLat()),
                formatCoordinate(h.Boundaries.getMaxLon()));
        headerString += "  </metadata>\r\n";
        headerString += "  <trk>\r\n";
        headerString += String.format(LOCALE, "    <name>%s</name>\r\n",
                trackName);
        headerString += "    <trkseg>\r\n";
        return headerString;
    }

    @Override
    public String formatEntry(Measurement m) {
        StringBuilder sb = new StringBuilder();

        // begin trackpoint
        sb.append(String.format(LOCALE, "      <trkpt lat=\"%s\" lon=\"%s\">\r\n", formatCoordinate(m.getLatitude()), formatCoordinate(m.getLongitude())));

        //set standard properties
        boolean altitudeAvailable = !DoubleUtils.equals(m.getGpsAltitude(), Measurement.GPS_VALUE_NOT_AVAILABLE);
        if (altitudeAvailable) {
            sb.append(String.format(LOCALE, "        <ele>%s</ele>\r\n", formatGpsValue(m.getGpsAltitude())));
        }
        sb.append(String.format(LOCALE, "        <time>%s</time>\r\n", formatDate(m.getMeasuredAt())));
        sb.append("        <name><![CDATA[");
        for (Cell c : m.getCells()) {
            int mcc = c.getMcc();
            int psc = c.getPsc();
            sb.append(String.format(LOCALE, "(%sMNC %s, LAC %s, CID %s, %sSignal strength %s dBm, Network %s%s), ",
                    (mcc != Cell.UNKNOWN_CID ? String.format(LOCALE, "MCC %s, ", formatInt(mcc)) : ""),
                    formatInt(c.getMnc()),
                    formatInt(c.getLac()),
                    formatLong(c.getCid()),
                    (psc != Cell.UNKNOWN_CID ? String.format(LOCALE, "PSC %s, ", formatInt(psc)) : ""),
                    formatDbmSignal(c.getDbm()),
                    cellUtils.getSystemType(c.getNetworkType()),
                    c.isNeighboring() ? ", neighboring" : ""));
        }
        sb.append(String.format(LOCALE, "(GPS accuracy %s m)]]></name>\r\n", formatGpsValue(m.getGpsAccuracy())));

        // set extended properties
        boolean speedAvailable = DoubleUtils.greaterThan(m.getGpsSpeed(), Measurement.GPS_VALUE_NOT_AVAILABLE);
        boolean bearingAvailable = DoubleUtils.greaterThan(m.getGpsBearing(), Measurement.GPS_VALUE_NOT_AVAILABLE);
        if (speedAvailable || bearingAvailable) {
            sb.append("        <extensions>\r\n"
                    + "          <gpxtpx:TrackPointExtension>\r\n");
            if (speedAvailable)
                sb.append(String.format(LOCALE, "            <gpxtpx:speed>%s</gpxtpx:speed>\r\n",
                        formatGpsValue(m.getGpsSpeed())));
            if (bearingAvailable)
                sb.append(String.format(LOCALE, "            <gpxtpx:bearing>%s</gpxtpx:bearing>\r\n",
                        formatGpsValue(m.getGpsBearing())));
            sb.append("          </gpxtpx:TrackPointExtension>\r\n"
                    + "        </extensions>\r\n");
        }

        // end trackpoint
        sb.append("      </trkpt>\r\n");

        return sb.toString();
    }

    @Override
    public String formatNewSegment() {
        return "    </trkseg>\r\n"
                + "    <trkseg>\r\n";
    }

    @Override
    public String formatFooter() {
        return "    </trkseg>\r\n"
                + "  </trk>\r\n"
                + "</gpx>\r\n";
    }

    private String formatCoordinate(double value) {
        return coordsDoubleFormatter.format(value);
    }

    private String formatGpsValue(double value) {
        return gpsDoubleFormatter.format(value);
    }

    private String formatDbmSignal(int dbm) {
        if (dbm != Cell.UNKNOWN_SIGNAL)
            return String.valueOf(dbm);
        else
            return "";
    }

    private String formatDate(long timestamp) {
        return exportDateFormatter.format(new Date(timestamp));
    }

    private String formatInt(int value) {
        return intFormatter.format(value);
    }

    private String formatLong(long value) {
        return intFormatter.format(value);
    }
}
