(ns status-im.contexts.wallet.wallet-connect.modals.session-proposal.view
  (:require
    [clojure.string :as string]
    [quo.context]
    [quo.core :as quo]
    [react-native.core :as rn]
    [status-im.common.floating-button-page.view :as floating-button-page]
    [status-im.contexts.wallet.wallet-connect.modals.common.list-info-box.view :as list-info-box]
    [status-im.contexts.wallet.wallet-connect.modals.session-proposal.style :as style]
    [status-im.contexts.wallet.wallet-connect.utils.data-store :as data-store]
    [utils.i18n :as i18n]
    [utils.re-frame :as rf]
    [utils.string]))

(defn- dapp-metadata
  []
  (let [proposer                 (rf/sub [:wallet-connect/session-proposer])
        {:keys [icons name url]} (:metadata proposer)
        first-icon               (first icons)
        dapp-name                (data-store/compute-dapp-name name url)
        profile-picture          (data-store/compute-dapp-icon-path first-icon url)]
    [:<>
     [rn/view {:style style/dapp-avatar}
      [quo/user-avatar
       {:profile-picture profile-picture
        :size            :big
        :full-name       dapp-name}]]
     [quo/page-top
      {:title       dapp-name
       :description :context-tag
       :context-tag {:type    :icon
                     :size    32
                     :icon    :i/link
                     :context url}}]]))

(defn- approval-note
  []
  (let [dapp-name (rf/sub [:wallet-connect/session-proposer-name])]
    [list-info-box/view
     {:dapp-name       dapp-name
      :container-style {:margin-horizontal 20}}]))

(defn- format-network-name
  [network]
  (-> network :network-name name string/capitalize))

(defn- set-current-proposal-address
  [acc]
  (fn []
    (rf/dispatch [:wallet-connect/set-current-proposal-address (:address acc)])
    (rf/dispatch [:hide-bottom-sheet])))

(defn- accounts-list
  []
  (let [accounts         (rf/sub [:wallet/operable-accounts])
        selected-address (rf/sub [:wallet-connect/current-proposal-address])]
    [rn/view {:style style/account-switcher-list}
     (for [{:keys [address] :as account} accounts]
       ^{:key (str address)}
       [quo/account-item
        {:type          :default
         :state         (if (= address selected-address)
                          :selected
                          :default)
         :account-props account
         :on-press      (set-current-proposal-address account)}])]))

(defn- account-switcher-sheet
  []
  [:<>
   [rn/view {:style style/account-switcher-title}
    [quo/text
     {:size                :heading-2
      :weight              :semi-bold
      :accessibility-label "select-account-title"}
     (i18n/label :t/select-account)]]
   [accounts-list]])

(defn- show-account-switcher-bottom-sheet
  []
  (rf/dispatch
   [:show-bottom-sheet
    {:content account-switcher-sheet}]))

(defn- connection-category
  []
  (let [{:keys [session-networks
                all-networks-in-session?]}       (rf/sub
                                                  [:wallet-connect/session-proposal-network-details])
        address                                  (rf/sub [:wallet-connect/current-proposal-address])
        {:keys [name customization-color emoji]} (rf/sub [:wallet-connect/account-details-by-address
                                                          address])
        network-names                            (->> session-networks
                                                      (map format-network-name)
                                                      (string/join ", "))
        network-images                           (mapv :source session-networks)
        data-item-common-props                   {:blur?  false
                                                  :card?  false
                                                  :status :default
                                                  :size   :large}
        account-data-item-props                  (assoc data-item-common-props
                                                        :right-content {:type :accounts
                                                                        :size :size-32
                                                                        :data [{:emoji emoji
                                                                                :customization-color
                                                                                customization-color}]}
                                                        :on-press      show-account-switcher-bottom-sheet
                                                        :title         (i18n/label :t/account-title)
                                                        :subtitle      name
                                                        :right-icon    :i/chevron-right)
        networks-data-item-props                 (assoc data-item-common-props
                                                        :right-content {:type :network
                                                                        :data network-images}
                                                        :title         (i18n/label :t/networks)
                                                        :subtitle      (if all-networks-in-session?
                                                                         (i18n/label :t/all-networks)
                                                                         network-names))]
    [quo/category
     {:blur?     false
      :list-type :data-item
      :data      [account-data-item-props
                  networks-data-item-props]}]))

(defn- footer
  []
  (let [customization-color (rf/sub [:profile/customization-color])]
    [quo/bottom-actions
     {:actions                 :two-actions
      :buttons-container-style style/footer-buttons-container
      :button-two-label        (i18n/label :t/decline)
      :button-two-props        {:type                :grey
                                :accessibility-label :wc-deny-connection
                                :on-press            #(rf/dispatch
                                                       [:dismiss-modal
                                                        :screen/wallet.wallet-connect-session-proposal])}
      :button-one-label        (i18n/label :t/connect)
      :button-one-props        {:customization-color customization-color
                                :type                :primary
                                :accessibility-label :wc-connect
                                :on-press            #(rf/dispatch
                                                       [:wallet-connect/approve-session])}}]))

(defn- header
  []
  [quo/page-nav
   {:type                :no-title
    :background          :blur
    :icon-name           :i/close
    :on-press            (rn/use-callback
                          #(rf/dispatch [:dismiss-modal :screen/wallet.wallet-connect-session-proposal]))
    :accessibility-label :wc-session-proposal-top-bar}])

(defn view
  []
  (rn/use-unmount #(rf/dispatch [:wallet-connect/reject-session-proposal]))
  [floating-button-page/view
   {:footer-container-padding 0
    :header                   [header]
    :footer                   [footer]}
   [rn/view
    [dapp-metadata]
    [connection-category]
    [approval-note]]])
