(ns status-im.contexts.wallet.events
  (:require
    [camel-snake-kebab.extras :as cske]
    [cljs-time.coerce :as time-coerce]
    [clojure.set]
    [clojure.string :as string]
    [react-native.platform :as platform]
    [status-im.constants :as constants]
    [status-im.contexts.network.db :as network.db]
    [status-im.contexts.profile.db :as profile.db]
    [status-im.contexts.settings.wallet.effects]
    [status-im.contexts.settings.wallet.events]
    [status-im.contexts.wallet.account.db :as account.db]
    [status-im.contexts.wallet.common.activity-tab.events]
    [status-im.contexts.wallet.common.transaction-settings.core :as transaction-settings]
    [status-im.contexts.wallet.common.utils :as utils]
    [status-im.contexts.wallet.data-store :as data-store]
    [status-im.contexts.wallet.db :as db]
    [status-im.contexts.wallet.db-path :as db-path]
    [status-im.contexts.wallet.item-types :as item-types]
    [status-im.contexts.wallet.networks.core :as networks]
    [status-im.contexts.wallet.networks.db :as networks.db]
    status-im.contexts.wallet.networks.events
    [status-im.contexts.wallet.send.utils :as send-utils]
    [status-im.contexts.wallet.sheets.network-selection.view :as network-selection]
    [status-im.contexts.wallet.tokens.events]
    [status-im.feature-flags :as ff]
    [taoensso.timbre :as log]
    [utils.collection]
    [utils.ethereum.eip.eip55 :as eip55]
    [utils.i18n :as i18n]
    [utils.number]
    [utils.re-frame :as rf]
    [utils.security.core :as security]
    [utils.transforms :as transforms]))

(rf/reg-event-fx :wallet/show-account-created-toast
 (fn [{:keys [db]} [address]]
   (let [account-name (get-in db [:wallet :accounts address :name])]
     {:db (update db :wallet dissoc :navigate-to-account :new-account?)
      :fx [[:dispatch
            [:toasts/upsert
             {:id   :new-wallet-account-created
              :type :positive
              :text (i18n/label :t/account-created {:name account-name})}]]]})))

(rf/reg-event-fx :wallet/navigate-to-account
 (fn [{:keys [db]} [address]]
   {:db (assoc-in db [:wallet :current-viewing-account-address] address)
    :fx [[:dispatch [:navigate-to :screen/wallet.accounts]]]}))

(rf/reg-event-fx :wallet/navigate-to-account-within-stack
 (fn [{:keys [db]} [address]]
   {:db (assoc-in db [:wallet :current-viewing-account-address] address)
    :fx [[:dispatch [:navigate-to-within-stack [:screen/wallet.accounts :screen/shell-stack]]]]}))

(rf/reg-event-fx :wallet/navigate-to-new-account
 (fn [{:keys [db]} [address]]
   {:db (assoc-in db [:wallet :current-viewing-account-address] address)
    :fx [[:dispatch [:hide-bottom-sheet]]
         [:dispatch [:navigate-to :screen/wallet.accounts]]
         [:dispatch [:wallet/show-account-created-toast address]]]}))

(rf/reg-event-fx :wallet/select-account-tab
 (fn [{:keys [db]} [tab]]
   {:db (assoc-in db [:wallet :ui :account-page :active-tab] tab)
    :fx [(if (= tab :activity)
           [:dispatch [:wallet/fetch-activities-for-current-account]]
           [:dispatch [:wallet/stop-activity-filter-session]])]}))

(rf/reg-event-fx :wallet/select-home-tab
 (fn [{:keys [db]} [tab]]
   {:db (assoc-in db [:wallet :ui :active-tab] tab)}))

(rf/reg-event-fx :wallet/clear-account-tab
 (fn [{:keys [db]}]
   {:db (assoc-in db [:wallet :ui :account-page :active-tab] nil)
    :fx [[:dispatch [:wallet/stop-activity-filter-session]]]}))

(rf/reg-event-fx :wallet/switch-current-viewing-account
 (fn [{:keys [db]} [address]]
   (let [{:keys [tx-type]} (db/send db)
         bridge-tx?        (= tx-type :tx/bridge)]
     {:db (cond-> db
            :always    (assoc-in [:wallet :current-viewing-account-address] address)
            bridge-tx? (update-in db-path/send assoc :to-address address))})))

(rf/reg-event-fx :wallet/clean-current-viewing-account
 (fn [{:keys [db]} [ignore-just-completed-transaction?]]
   (let [{:keys [entry-point just-completed-transaction?
                 collectible-multiple-owners?]} (db/send db)
         entry-point-wallet-home?               (= entry-point :screen/wallet-stack)]
     {:db (cond-> db
            (and (not entry-point)
                 (not ignore-just-completed-transaction?)
                 (not just-completed-transaction?))
            (update :wallet dissoc :current-viewing-account-address)

            (and entry-point-wallet-home? (not collectible-multiple-owners?))
            (update-in db-path/send dissoc :entry-point)

            (and entry-point-wallet-home?
                 (not just-completed-transaction?))
            (update :wallet dissoc :current-viewing-account-address))})))

(rf/reg-event-fx :wallet/close-account-page
 (fn [{:keys [db]}]
   (let [just-completed-transaction? (get-in db (conj db-path/send :just-completed-transaction?))]
     {:db (update db :wallet dissoc :current-viewing-account-address)
      :fx [(when-not just-completed-transaction?
             [:dispatch [:wallet/clear-account-tab]])
           [:dispatch [:wallet/stop-activity-filter-session]]]})))

(defn log-rpc-error
  [_ [{:keys [event params]} error]]
  (log/warn (str "[wallet] Failed to " event)
            {:params params
             :error  error}))

(rf/reg-event-fx :wallet/log-rpc-error log-rpc-error)

(def refresh-accounts-fx-dispatches
  [[:dispatch [:wallet/get-wallet-token-for-all-accounts]]
   [:dispatch [:wallet/request-collectibles-for-all-accounts {:new-request? true}]]])

(rf/reg-event-fx
 :wallet/fetch-assets-for-address
 (fn [_ [address]]
   {:fx [[:dispatch [:wallet/get-wallet-token-for-accounts [address]]]
         [:dispatch [:wallet/request-collectibles-for-account address]]]}))

(defn- reconcile-accounts
  [db-accounts-by-address new-accounts]
  (reduce
   (fn [res {:keys [address] :as account}]
     ;; Because we add extra fields (tokens and collectibles) into the RPC
     ;; response from accounts_getAccounts, if we are offline we want to keep
     ;; the old balances in the accounts, thus we merge the up-to-date account
     ;; from status-go into the cached accounts. We also merge when online
     ;; because we will re-fetch balances anyway.
     ;;
     ;; Refactor improvement: don't augment entities from status-go, store
     ;; tokens and collectibles in separate keys in the app-db indexed by
     ;; account address.
     (assoc res
            address
            (-> (get db-accounts-by-address address)
                (merge account)
                ;; These should not be cached, otherwise when going
                ;; offline->online collectibles won't be fetched.
                (dissoc :current-collectible-idx :has-more-collectibles?))))
   {}
   new-accounts))

(rf/reg-event-fx :wallet/get-accounts-success
 (fn [{:keys [db]} [accounts]]
   (let [wallet-accounts     (data-store/rpc->accounts accounts)
         wallet-db           (get db :wallet)
         new-account?        (:new-account? wallet-db)
         navigate-to-account (:navigate-to-account wallet-db)]
     {:db (update-in db [:wallet :accounts] reconcile-accounts wallet-accounts)
      :fx (concat (when (or (data-store/tokens-never-loaded? db)
                            (network.db/online? db))
                    refresh-accounts-fx-dispatches)
                  [(when new-account?
                     [:dispatch [:wallet/navigate-to-new-account navigate-to-account]])])})))

(rf/reg-event-fx :wallet/get-accounts
 (fn []
   {:fx [[:json-rpc/call
          [{:method     "accounts_getAccounts"
            :on-success [:wallet/get-accounts-success]
            :on-error   [:wallet/log-rpc-error {:event :wallet/get-accounts}]}]]]}))

(rf/reg-event-fx
 :wallet/save-account
 (fn [_ [{:keys [account on-success]}]]
   {:fx [[:json-rpc/call
          [{:method     "accounts_saveAccount"
            :params     [(data-store/<-account account)]
            :on-success (fn []
                          (rf/dispatch [:wallet/get-accounts])
                          (when (fn? on-success)
                            (on-success)))
            :on-error   [:wallet/log-rpc-error {:event :wallet/save-account}]}]]]}))

(rf/reg-event-fx
 :wallet/show-account-deleted-toast
 (fn [_ [toast-message]]
   {:fx [[:dispatch [:toasts/upsert {:type :positive :text toast-message}]]]}))

(rf/reg-event-fx
 :wallet/remove-account-success
 (fn [_ [toast-message _]]
   {:fx [[:dispatch [:wallet/clean-current-viewing-account]]
         [:dispatch [:wallet/get-accounts]]
         [:dispatch [:wallet/get-keypairs]]
         [:dispatch-later
          {:ms       100
           :dispatch [:hide-bottom-sheet]}]
         [:dispatch-later
          {:ms       100
           :dispatch [:pop-to-root :screen/shell-stack]}]
         [:dispatch-later
          {:ms       100
           :dispatch [:wallet/show-account-deleted-toast toast-message]}]]}))

(rf/reg-event-fx
 :wallet/remove-account
 (fn [_ [{:keys [address toast-message]}]]
   {:fx [[:json-rpc/call
          [{:method     "accounts_deleteAccount"
            :params     [address]
            :on-success [:wallet/remove-account-success toast-message]
            :on-error   [:wallet/log-rpc-error {:event :wallet/remove-account}]}]]]}))

(rf/reg-event-fx :wallet/get-wallet-token-for-all-accounts
 (fn [{:keys [db]}]
   (let [addresses (->> (get-in db [:wallet :accounts])
                        (vals)
                        (keep :address)
                        (vec))]
     {:fx [[:dispatch [:wallet/get-wallet-token-for-accounts addresses]]]})))

(rf/reg-event-fx :wallet/get-wallet-token-for-accounts
 (fn [{:keys [db]} [addresses]]
   {:db (reduce
         (fn [db address]
           (assoc-in db [:wallet :ui :tokens-loading address] true))
         db
         addresses)
    :fx [[:json-rpc/call
          [{:method     "wallet_fetchOrGetCachedWalletBalances"
            :params     [addresses true]
            :on-success [:wallet/store-wallet-token addresses]
            :on-error   [:wallet/get-wallet-token-for-accounts-failed addresses]}]]]}))

(rf/reg-event-fx :wallet/reload-cached-balances
 (fn [{:keys [db]}]
   (let [addresses (account.db/get-accounts-addresses db)]
     {:fx [[:json-rpc/call
            [{:method     "wallet_fetchOrGetCachedWalletBalances"
              :params     [addresses false]
              :on-success [:wallet/store-wallet-token addresses]
              :on-error   [:wallet/get-wallet-token-for-accounts-failed addresses]}]]]})))

(rf/reg-event-fx
 :wallet/get-wallet-token-for-accounts-failed
 (fn [{:keys [db]} [addresses error]]
   (log/info "failed to get wallet token "
             {:error  error
              :event  :wallet/get-wallet-token-for-accounts
              :params addresses})
   {:fx [[:dispatch [:wallet/get-last-wallet-token-update-if-needed]]]
    :db (reduce
         (fn [db address]
           (assoc-in db [:wallet :ui :tokens-loading address] false))
         db
         addresses)}))

(rf/reg-event-fx
 :wallet/set-all-tokens-loading
 (fn [{:keys [db]}]
   {:db (reduce (fn [db address]
                  (assoc-in db [:wallet :ui :tokens-loading address] true))
                db
                (account.db/get-accounts-addresses db))}))

(rf/reg-event-fx
 :wallet/reset-accounts-tokens
 (fn [{:keys [db]}]
   (let [reset-tokens (fn [stored-accounts]
                        (reduce-kv
                         (fn [accounts address _]
                           (update accounts address assoc :tokens []))
                         stored-accounts
                         stored-accounts))]
     {:db (update-in db [:wallet :accounts] reset-tokens)
      :fx [[:dispatch [:wallet/set-all-tokens-loading]]]})))

(rf/reg-event-fx
 :wallet/store-wallet-token
 (fn [{:keys [db]} [addresses raw-tokens-data]]
   (let [supported-chains-by-token-symbol (get-in db [:wallet :tokens :supported-chains-by-symbol])
         profile-currency                 (get-in db [:profile/profile :currency])
         tokens                           (data-store/rpc->tokens raw-tokens-data
                                                                  supported-chains-by-token-symbol)
         add-tokens                       (fn [stored-accounts tokens-per-account]
                                            (reduce-kv
                                             (fn [accounts address tokens-data]
                                               (if (contains? accounts address)
                                                 (update accounts address assoc :tokens tokens-data)
                                                 accounts))
                                             stored-accounts
                                             tokens-per-account))
         symbols                          (reduce-kv
                                           (fn [acc _ tokens-data]
                                             (into acc (map :symbol tokens-data)))
                                           #{}
                                           tokens)]
     {:db (-> db
              (update-in [:wallet :accounts] add-tokens tokens)
              ((fn [db]
                 (reduce (fn [db address]
                           (assoc-in db [:wallet :ui :tokens-loading address] false))
                         db
                         addresses))))
      :fx [[:dispatch [:wallet/get-last-wallet-token-update-if-needed]]
           [:effects.wallet.tokens/fetch-market-values
            {:symbols    symbols
             :currency   profile-currency
             :on-success [:wallet.tokens/store-market-values]
             :on-error   [:wallet.tokens/fetch-market-values-failed]}]
           [:effects.wallet.tokens/fetch-prices
            {:symbols    symbols
             :currencies [constants/profile-default-currency profile-currency]
             :on-success [:wallet.tokens/store-prices]
             :on-error   [:wallet.tokens/fetch-prices-failed]}]]})))

(rf/reg-event-fx
 :wallet/get-last-wallet-token-update-if-needed
 (fn [{:keys [db]}]
   (let [all-tokens-loaded? (->> (get-in db [:wallet :ui :tokens-loading])
                                 vals
                                 (every? false?))]
     (when all-tokens-loaded?
       {:fx [[:json-rpc/call
              [{:method     "wallet_getLastWalletTokenUpdate"
                :params     []
                :on-success [:wallet/get-last-wallet-token-update-success]
                :on-error   [:wallet/log-rpc-error
                             {:event :wallet/get-last-wallet-token-update-if-needed}]}]]]}))))

(rf/reg-event-fx
 :wallet/get-last-wallet-token-update-success
 (fn [{:keys [db]} [data]]
   (let [last-updates (reduce (fn [acc [k v]]
                                (assoc acc k (time-coerce/from-long (* 1000 v))))
                              {}
                              data)]
     {:db (assoc-in db [:wallet :ui :last-updates-per-address] last-updates)})))

(rf/defn scan-address-success
  {:events [:wallet/scan-address-success]}
  [{:keys [db]} address]
  {:db (assoc-in db [:wallet :ui :scanned-address] address)})

(rf/defn clean-scanned-address
  {:events [:wallet/clean-scanned-address]}
  [{:keys [db]}]
  {:db (update-in db [:wallet :ui] dissoc :scanned-address)})

(rf/reg-event-fx :wallet/add-account-success
 (fn [{:keys [db]} [address]]
   {:db (-> db
            (assoc-in [:wallet :navigate-to-account] address)
            (assoc-in [:wallet :new-account?] true))
    :fx [[:dispatch [:wallet/get-accounts]]
         [:dispatch [:wallet/get-keypairs]]
         [:dispatch [:wallet/clear-create-account]]]}))

(rf/reg-event-fx :wallet/add-account
 (fn [_
      [{:keys [key-uid password account-name emoji color type] :or {type :generated}}
       {:keys [public-key address path]}]]
   (let [lowercase-address (some-> address
                                   string/lower-case)
         account-config    {:key-uid    (when (= type :generated) key-uid)
                            :wallet     false
                            :chat       false
                            :type       type
                            :name       account-name
                            :emoji      emoji
                            :path       path
                            :address    lowercase-address
                            :public-key public-key
                            :colorID    color}]
     {:fx [[:json-rpc/call
            [{:method     "accounts_addAccount"
              :params     [(when (= type :generated)
                             (security/safe-unmask-data password))
                           account-config]
              :on-success [:wallet/add-account-success lowercase-address]
              :on-error   [:wallet/log-rpc-error
                           {:event  :wallet/add-account
                            :params account-config}]}]]]})))

(defn get-keypairs
  [_]
  {:fx [[:json-rpc/call
         [{:method     "accounts_getKeypairs"
           :params     []
           :on-success [:wallet/get-keypairs-success]
           :on-error   [:wallet/log-rpc-error {:event :wallet/get-keypairs}]}]]]})

(rf/reg-event-fx :wallet/get-keypairs get-keypairs)

(defn- remove-unsupported-bridge-networks-from-token
  [token]
  (-> token
      (update :networks
              (fn [networks]
                (filter networks/bridge-supported-network? networks)))
      (update :balances-per-chain
              (fn [balances]
                (into {}
                      (filter (fn [[_ balance]]
                                (networks/bridge-supported-network? balance))
                              balances))))))

(rf/reg-event-fx :wallet/bridge-select-token
 (fn [{:keys [db]}
      [{:keys [token token-symbol stack-id network owners networks start-flow?] :as params}]]
   (let [{:keys [wallet]}             db
         bridge-supported-networks    (networks/filter-bridge-supported-networks networks)
         unique-owner                 (when (= (count owners) 1)
                                        (first owners))
         token                        (if (and unique-owner (nil? token))
                                        (let [token (utils/get-token-from-account db
                                                                                  token-symbol
                                                                                  unique-owner)]
                                          (utils/token-with-balance token bridge-supported-networks))
                                        token)
         token                        (remove-unsupported-bridge-networks-from-token token)
         missing-recipient?           (-> db db/send :to-address nil?)
         to-address                   (or unique-owner
                                          (-> db :wallet :current-viewing-account-address)
                                          (utils/get-default-account (-> db :wallet :accounts)))
         view-id                      (:view-id db)
         root-screen?                 (or (= view-id :screen/wallet-stack) (nil? view-id))
         multi-account-balance?       (-> (utils/get-accounts-with-token-balance (:accounts wallet)
                                                                                 token)
                                          (count)
                                          (> 1))
         account-not-defined?         (and (not unique-owner) multi-account-balance?)
         networks-with-balance        (when (and token (:balances-per-chain token))
                                        (filter #(not= (:balance %) "0")
                                                (vals (:balances-per-chain token))))
         balance-in-only-one-network? (when networks-with-balance (= (count networks-with-balance) 1))
         networks                     (networks.db/get-active-networks db)
         network                      (if balance-in-only-one-network?
                                        (first (filter #(= (:chain-id %)
                                                           (:chain-id (first networks-with-balance)))
                                                       networks))
                                        network)]
     {:db (cond-> db
            :always            (update-in db-path/send assoc :tx-type :tx/bridge)
            token              (update-in db-path/send assoc :token token)
            token-symbol       (update-in db-path/send assoc :token-symbol token-symbol)
            network            (update-in db-path/send assoc :network network)
            missing-recipient? (update-in db-path/send assoc :to-address to-address))
      :fx (cond
            ;; If the token has a balance in more than one account and this was dispatched from the
            ;; general wallet screen, open the account selection screen.
            (and account-not-defined? root-screen?)
            [[:dispatch [:open-modal :screen/wallet.select-from]]]

            ;; If the token has a balance in only one account (or this was dispatched from the
            ;; account screen) and the network is already set, navigate forward in the bridge flow.
            (some? network)
            [(when to-address [:dispatch [:wallet/switch-current-viewing-account to-address]])
             [:dispatch
              [:wallet/wizard-navigate-forward
               {:current-screen stack-id
                :start-flow?    start-flow?
                :flow-id        :wallet-bridge-flow}]]]

            ;; If we know which account to bridge the token from but the network is not set yet,
            ;; show the network selection drawer.
            :else
            [[:dispatch [:wallet/switch-current-viewing-account to-address]]
             [:dispatch
              [:show-bottom-sheet
               {:content (fn []
                           [network-selection/view
                            {:title             (i18n/label :t/select-network-to-bridge-from)
                             :token-symbol      (or token-symbol (:symbol token))
                             :source            :bridge
                             :on-select-network (fn [network]
                                                  (rf/dispatch [:hide-bottom-sheet])
                                                  (rf/dispatch
                                                   [:wallet/bridge-select-token
                                                    (assoc params :network network)]))}])}]]])})))

(rf/reg-event-fx
 :wallet/start-bridge
 (fn [{:keys [db]}]
   {:db (update-in db db-path/send assoc :tx-type :tx/bridge)
    :fx [[:dispatch
          [:wallet/wizard-navigate-forward
           {:start-flow? true
            :flow-id     :wallet-bridge-flow}]]]}))

(rf/reg-event-fx
 :wallet/set-send-tx-type
 (fn [{:keys [db]} [type]]
   {:db (update-in db db-path/send assoc :tx-type type)}))

(rf/reg-event-fx :wallet/select-bridge-network
 (fn [{:keys [db]} [{:keys [network-chain-id stack-id]}]]
   {:db (update-in db db-path/send assoc :bridge-to-chain-id network-chain-id)
    :fx [[:dispatch
          [:wallet/wizard-navigate-forward
           {:current-screen stack-id
            :flow-id        :wallet-bridge-flow}]]]}))

(rf/reg-event-fx
 :wallet/find-ens
 (fn [{:keys [db]} [input contacts on-error-fn]]
   (let [result (if (empty? input)
                  []
                  (filter #(string/starts-with? (or (:ens-name %) "") input) contacts))]
     (if (and input (empty? result))
       {:fx [[:dispatch [:wallet/search-ens input on-error-fn ".stateofus.eth"]]]}
       {:db (-> db
                (assoc-in [:wallet :ui :search-address :local-suggestions]
                          (map #(assoc % :type item-types/saved-address) result))
                (assoc-in [:wallet :ui :search-address :valid-ens-or-address?]
                          (not-empty result)))}))))

(rf/reg-event-fx
 :wallet/search-ens
 (fn [{db :db} [input on-error-fn domain]]
   (let [ens      (if (string/includes? input ".")
                    input
                    (str input domain))
         chain-id (networks.db/get-chain-id db :mainnet)]
     {:fx [[:json-rpc/call
            [{:method     "ens_addressOf"
              :params     [chain-id ens]
              :on-success #(rf/dispatch [:wallet/set-ens-address % ens])
              :on-error   (fn []
                            (if (= domain ".stateofus.eth")
                              (rf/dispatch [:wallet/search-ens input on-error-fn ".eth"])
                              (do
                                (rf/dispatch [:wallet/set-ens-address nil ens])
                                (on-error-fn))))}]]]})))

(rf/reg-event-fx
 :wallet/set-ens-address
 (fn [{:keys [db]} [result ens]]
   (let [suggestion (if result
                      [{:type         item-types/address
                        :ens          ens
                        :address      (eip55/address->checksum result)
                        :full-address (eip55/address->checksum result)}]
                      [])]
     {:db (-> db
              (assoc-in [:wallet :ui :search-address :local-suggestions] suggestion)
              (assoc-in [:wallet :ui :search-address :valid-ens-or-address?] (boolean result)))})))

(rf/reg-event-fx :wallet/address-validation-success
 (fn [{:keys [db]}]
   {:db (update-in db
                   [:wallet :ui :search-address]
                   assoc
                   :valid-ens-or-address? true
                   :loading?              false)}))

(rf/reg-event-fx :wallet/address-validation-failed
 (fn [{:keys [db]}]
   {:db (update-in db
                   [:wallet :ui :search-address]
                   assoc
                   :valid-ens-or-address? false
                   :loading?              false)}))

(rf/reg-event-fx :wallet/clean-local-suggestions
 (fn [{:keys [db]}]
   {:db (-> db
            (assoc-in [:wallet :ui :search-address :local-suggestions] [])
            (assoc-in [:wallet :ui :search-address :valid-ens-or-address?] false))}))

(rf/reg-event-fx :wallet/searching-address
 (fn [{:keys [db]}]
   {:db (assoc-in db [:wallet :ui :search-address :loading?] true)}))

(rf/reg-event-fx :wallet/reload
 (fn [{:keys [db]}]
   (let [supported-chains-by-symbol (get-in db [:wallet :tokens :supported-chains-by-symbol])
         symbols                    (mapv name (keys supported-chains-by-symbol))
         profile-currency           (get-in db [:profile/profile :currency])]
     {:fx [[:dispatch [:wallet/get-wallet-token-for-all-accounts]]
           [:effects.wallet.tokens/fetch-prices
            {:symbols    symbols
             :currencies [constants/profile-default-currency profile-currency]
             :on-success [:wallet.tokens/store-prices]
             :on-error   [:wallet.tokens/fetch-prices-failed]}]]
      :db (assoc-in db [:wallet :ui :loading :prices] true)})))

(rf/reg-event-fx :wallet/start-wallet
 (fn [_]
   {:fx [[:json-rpc/call
          [{:method   "wallet_startWallet"
            :on-error [:wallet/log-rpc-error {:event :wallet/start-wallet}]}]]]}))

(rf/reg-event-fx :wallet/initialize
 (fn []
   {:fx [[:dispatch [:wallet/start-wallet]]
         [:dispatch [:wallet/get-networks]]
         [:dispatch [:wallet/get-accounts]]
         [:dispatch [:wallet/get-keypairs]]
         [:dispatch [:wallet/get-saved-addresses]]
         (when (ff/enabled? ::ff/wallet.wallet-connect)
           [:dispatch-later [{:ms 500 :dispatch [:wallet-connect/init]}]])]}))

(rf/reg-event-fx :wallet/share-account
 (fn [_ [{:keys [content title]}]]
   {:fx [[:effects.share/open
          {:options (if platform/ios?
                      {:activityItemSources
                       [{:placeholderItem {:type    :text
                                           :content content}
                         :item            {:default {:type    :text
                                                     :content content}}
                         :linkMetadata    {:title title}}]}
                      {:title   title
                       :subject title
                       :message content})}]]}))

(rf/reg-event-fx
 :wallet/blockchain-status-changed
 (fn [{:keys [db]} [{:keys [message]}]]
   (let [chains                 (-> (transforms/json->clj message)
                                    (update-keys (comp utils.number/parse-int name)))
         down-chain-ids         (-> (select-keys chains
                                                 (for [[k v] chains :when (= v "down")] k))
                                    keys)
         test-networks-enabled? (profile.db/testnet? db)
         chain-ids              (networks.db/get-active-chain-ids db)
         chains-filtered        (remove #(not
                                          (contains? chain-ids %))
                                        down-chain-ids)
         chains-down?           (and (network.db/online? db) (seq chains-filtered))
         chain-names            (when chains-down?
                                  (->> (map (partial networks.db/get-network-details db)
                                            chains-filtered)
                                       (map :full-name)
                                       distinct
                                       (string/join ", ")))]
     (when (seq down-chain-ids)
       (log/info "[wallet] Chain(s) down: " down-chain-ids)
       (log/info "[wallet] Chain name(s) down: " chain-names)
       (log/info "[wallet] Test network enabled: " (boolean test-networks-enabled?)))

     ;; NOTE <shivekkhurana>: We used to show an error toast, but disabled it because the down
     ;; signal is sent randomly. Needs to be investigated and enabled again !
     ;; Context: https://github.com/status-im/status-mobile/issues/21054

     {:db (assoc-in db [:wallet :statuses :blockchains] chains)})))

(rf/reg-event-fx
 :wallet/get-crypto-on-ramps-success
 (fn [{:keys [db]} [data]]
   (let [crypto-on-ramps (cske/transform-keys transforms/->kebab-case-keyword data)]
     {:db (assoc-in db
           [:wallet :crypto-on-ramps]
           {:one-time  (remove #(string/blank? (:site-url %)) crypto-on-ramps)
            :recurrent (remove #(string/blank? (:recurrent-site-url %)) crypto-on-ramps)})})))

(rf/reg-event-fx
 :wallet/get-crypto-on-ramps
 (fn [_]
   {:fx [[:json-rpc/call
          [{:method     "wallet_getCryptoOnRamps"
            :on-success [:wallet/get-crypto-on-ramps-success]
            :on-error   [:wallet/log-rpc-error {:event :wallet/get-crypto-on-ramps}]}]]]}))

(rf/reg-event-fx
 :wallet/resolve-ens
 (fn [{db :db} [{:keys [ens on-success on-error]}]]
   (let [chain-id (networks.db/get-chain-id db :mainnet)]
     {:fx [[:json-rpc/call
            [{:method     "ens_addressOf"
              :params     [chain-id ens]
              :on-success on-success
              :on-error   on-error}]]]})))

(rf/reg-event-fx
 :wallet/process-keypair-from-backup
 (fn [_ [{:keys [backedUpKeypair]}]]
   {:fx [[:dispatch [:wallet/reconcile-keypairs [backedUpKeypair]]]]}))

(rf/reg-event-fx
 :wallet/process-watch-only-account-from-backup
 (fn [_ [{:keys [backedUpWatchOnlyAccount]}]]
   {:fx [[:dispatch [:wallet/reconcile-watch-only-accounts [backedUpWatchOnlyAccount]]]]}))

(defn reconcile-watch-only-accounts
  [{:keys [db]} [watch-only-accounts]]
  (let [existing-accounts-by-address (get-in db [:wallet :accounts])
        group-label                  #(if % :removed-accounts :updated-accounts)
        {:keys [removed-accounts
                updated-accounts]}   (->> watch-only-accounts
                                          (map data-store/rpc->account)
                                          (group-by (comp group-label :removed)))
        existing-account-addresses   (set (keys existing-accounts-by-address))
        removed-account-addresses    (set (map :address removed-accounts))
        updated-account-addresses    (set (map :address updated-accounts))
        new-account-addresses        (clojure.set/difference updated-account-addresses
                                                             existing-account-addresses)]
    (cond-> {:db (update-in db
                            [:wallet :accounts]
                            (fn [existing-accounts]
                              (merge-with merge
                                          (apply dissoc existing-accounts removed-account-addresses)
                                          (utils.collection/index-by :address updated-accounts))))}

      (seq new-account-addresses)
      (assoc :fx
             (mapv (fn [address] [:dispatch [:wallet/fetch-assets-for-address address]])
                   new-account-addresses)))))

(rf/reg-event-fx :wallet/reconcile-watch-only-accounts reconcile-watch-only-accounts)

(defn reconcile-keypairs
  [{:keys [db]} [keypairs]]
  (let [existing-keypairs-by-id               (get-in db [:wallet :keypairs])
        existing-accounts-by-address          (get-in db [:wallet :accounts])
        {:keys [removed-keypair-ids
                removed-account-addresses
                updated-keypairs-by-id
                updated-accounts-by-address]} (data-store/reconcile-keypairs keypairs)
        updated-keypair-ids                   (set (keys updated-keypairs-by-id))
        updated-account-addresses             (set (keys updated-accounts-by-address))
        existing-account-addresses            (set (keys existing-accounts-by-address))
        new-account-addresses                 (clojure.set/difference updated-account-addresses
                                                                      existing-account-addresses)
        old-account-addresses                 (->> (vals existing-accounts-by-address)
                                                   (filter (fn [{:keys [address key-uid]}]
                                                             (and (contains? updated-keypair-ids key-uid)
                                                                  (not (contains?
                                                                        updated-accounts-by-address
                                                                        address)))))
                                                   (map :address))]
    (cond-> {:db (-> db
                     (assoc-in [:wallet :keypairs]
                               (-> (apply dissoc existing-keypairs-by-id removed-keypair-ids)
                                   (merge updated-keypairs-by-id)))
                     (assoc-in [:wallet :accounts]
                               (merge-with merge
                                           (apply dissoc
                                                  existing-accounts-by-address
                                                  (into removed-account-addresses
                                                        old-account-addresses))
                                           updated-accounts-by-address)))}

      (seq new-account-addresses)
      (assoc :fx
             (mapv (fn [address] [:dispatch [:wallet/fetch-assets-for-address address]])
                   new-account-addresses)))))

(rf/reg-event-fx :wallet/reconcile-keypairs reconcile-keypairs)

(rf/reg-event-fx
 :wallet/blockchain-health-changed
 (fn [{:keys [db]} [{:keys [message]}]]
   (let [full-status (cske/transform-keys message transforms/->kebab-case-keyword)]
     {:db (assoc-in db [:wallet :blockchain] full-status)})))

(rf/reg-event-fx
 :wallet/sign-transactions-signal-received
 (fn [{:keys [db]} [{send-details :sendDetails :as data}]]
   (let [type           (if (or (= (:fromToken send-details)
                                   (:toToken send-details))
                                (string/blank? (:toToken send-details)))
                          :send
                          :swap)
         callback-fx    (get-in db [:wallet :ui type :sign-transactions-callback-fx])
         error-response (:errorResponse send-details)]
     {:fx [(when (and callback-fx (not error-response))
             callback-fx)]
      :db (-> db
              (assoc-in [:wallet :ui type :sign-transactions-callback-fx] nil)
              (assoc-in [:wallet :ui type :error-response] error-response)
              (assoc-in [:wallet :ui type :transaction-for-signing] data))})))

(rf/reg-event-fx
 :wallet/transactions-sent-signal-received
 (fn [{:keys [db]}
      [{sent-transactions :sentTransactions
        send-details      :sendDetails}]]
   (let [swap? (-> db
                   (get-in db-path/swap)
                   seq)]
     {:fx [[:dispatch
            (if-let [error-response (:errorResponse send-details)]
              [(if swap?
                 :wallet.swap/transaction-failure
                 :wallet/transaction-failure)
               error-response]
              [(if swap?
                 :wallet.swap/transaction-success
                 :wallet/transaction-success)
               sent-transactions])]]})))

(rf/reg-event-fx
 :wallet/set-max-base-fee
 (fn [{db :db} [value]]
   {:db (assoc-in db [:wallet :ui :user-tx-settings :max-base-fee] value)}))

(rf/reg-event-fx
 :wallet/set-priority-fee
 (fn [{db :db} [value]]
   {:db (assoc-in db [:wallet :ui :user-tx-settings :priority-fee] value)}))

(rf/reg-event-fx
 :wallet/set-max-gas-amount
 (fn [{db :db} [value]]
   {:db (assoc-in db [:wallet :ui :user-tx-settings :gas-amount] value)}))

(rf/reg-event-fx
 :wallet/set-gas-price
 (fn [{db :db} [value]]
   {:db (assoc-in db [:wallet :ui :user-tx-settings :gas-price] value)}))

(rf/reg-event-fx
 :wallet/set-nonce
 (fn [{db :db} [value]]
   {:db (assoc-in db [:wallet :ui :user-tx-settings :nonce] value)}))

(defn set-fee-mode-effect
  [path-tx-identity gas-rate]
  (let [params [path-tx-identity gas-rate]]
    [:json-rpc/call
     [{:method   "wallet_setFeeMode"
       :params   params
       :on-error (fn [error]
                   (log/error "failed to set quick transaction settings"
                              {:event  :wallet/quick-fee-mode-confirmed
                               :error  (:message error)
                               :params params}))}]]))

(rf/reg-event-fx :wallet/quick-fee-mode-confirmed
 (fn [{db :db} [fee-mode]]
   (let [gas-rate        (transaction-settings/tx-fee-mode->gas-rate fee-mode)
         tx-type         (get-in db [:wallet :ui :send :tx-type])
         route           (if tx-type
                           (first (get-in db [:wallet :ui :send :route]))
                           (get-in db [:wallet :ui :swap :swap-proposal]))
         ;; bridge consist from 2 transactions - approval and send, so we need to apply
         ;; setting to both of them by making 2 calls
         set-fee-effects (if (= tx-type :tx/bridge)
                           [(set-fee-mode-effect (send-utils/path-identity route true) gas-rate)
                            (set-fee-mode-effect (send-utils/path-identity route false) gas-rate)]
                           [(set-fee-mode-effect (send-utils/path-identity route) gas-rate)])]
     {:db (assoc-in db [:wallet :ui :user-fee-mode] fee-mode)
      :fx (conj set-fee-effects
                [:dispatch [:wallet/mark-user-tx-settings-for-deletion]])})))

(defn set-custom-tx-effect
  [path-tx-identity custom-tx-params]
  (let [params [path-tx-identity custom-tx-params]]
    [:json-rpc/call
     [{:method   "wallet_setCustomTxDetails"
       :params   params
       :on-error (fn [error]
                   (log/error "failed to set custom tx settings"
                              {:event  :wallet/custom-transaction-settings-confirmed
                               :error  (:message error)
                               :params params}))}]]))

(rf/reg-event-fx :wallet/custom-transaction-settings-confirmed
 (fn [{db :db}]
   (let [tx-type                   (get-in db [:wallet :ui :send :tx-type])
         route                     (if tx-type
                                     (first (get-in db [:wallet :ui :send :route]))
                                     (get-in db [:wallet :ui :swap :swap-proposal]))
         user-tx-settings          (get-in db [:wallet :ui :user-tx-settings])
         custom-tx-params          (send-utils/path-tx-custom-params user-tx-settings route)
         ;; bridge consist from 2 transactions - approval and send, so we need to apply
         ;; setting to both of them by making 2 calls
         custom-tx-details-effects (if (= tx-type :tx/bridge)
                                     [(set-custom-tx-effect (send-utils/path-identity route true)
                                                            custom-tx-params)
                                      (set-custom-tx-effect (send-utils/path-identity route false)
                                                            custom-tx-params)]
                                     [(set-custom-tx-effect (send-utils/path-identity route)
                                                            custom-tx-params)])]
     {:db (assoc-in db [:wallet :ui :user-fee-mode] :tx-fee-mode/custom)
      :fx (conj custom-tx-details-effects
                [:dispatch [:wallet/mark-user-tx-settings-for-deletion]])})))

;; There is a delay between the moment when user selected
;; custom settings and the moment when new route arrived
;; with those settings applied. During this delay
;; we should keep user settings for ui. After new route
;; arrived we should clean the settings.
(rf/reg-event-fx :wallet/mark-user-tx-settings-for-deletion
 (fn [{db :db}]
   {:db (assoc-in db [:wallet :ui :user-tx-settings :delete-on-routes-update?] true)}))

