(ns status-im.contexts.chat.messenger.messages.delete-message.events
  (:require
    [status-im.contexts.chat.messenger.messages.list.events :as message-list]
    [utils.datetime :as datetime]
    [utils.i18n :as i18n]
    [utils.re-frame :as rf]))

(defn- update-db-clear-undo-timer
  [db chat-id message-id]
  (when (get-in db [:messages chat-id message-id])
    (update-in db
               [:messages chat-id message-id]
               dissoc
               :deleted-undoable-till)))

(defn- update-db-delete-locally
  "Delete message in re-frame db and set the undo timelimit"
  [db chat-id message-id undo-time-limit-ms deleted-by]
  (when (get-in db [:messages chat-id message-id])
    (update-in db
               [:messages chat-id message-id]
               assoc
               :deleted?              true
               :deleted-by            deleted-by
               :deleted-undoable-till (+ (datetime/timestamp) undo-time-limit-ms))))

(defn- update-db-undo-locally
  "Restore deleted message if called within timelimit"
  [db chat-id message-id]
  (let [{:keys [deleted? deleted-undoable-till]}
        (get-in db [:messages chat-id message-id])]
    (if (and deleted?
             (> deleted-undoable-till (datetime/timestamp)))
      (update-in db
                 [:messages chat-id message-id]
                 dissoc
                 :deleted?
                 :deleted-by
                 :deleted-undoable-till)
      (update-db-clear-undo-timer db chat-id message-id))))

(defn- should-and-able-to-unpin-to-be-deleted-message
  "check
  1. if the to-be deleted message is pinned
  2. if user has the permission to unpin message in current chat"
  [db {:keys [chat-id message-id]}]
  (let [{:keys [group-chat chat-id public? community-id
                admins]}    (get-in db [:chats chat-id])
        community           (get-in db [:communities community-id])
        community-admin?    (get community :admin false)
        pub-key             (get-in db [:profile/profile :public-key])
        group-admin?        (get admins pub-key)
        message-pin-enabled (and (not public?)
                                 (or (not group-chat)
                                     (and group-chat
                                          (or group-admin?
                                              community-admin?))))
        pinned              (get-in db
                                    [:pin-messages chat-id
                                     message-id])]
    (and pinned message-pin-enabled)))

(defn toast-undo-delete-message-for-everyone
  []
  (rf/dispatch [:chat.ui/undo-all-delete-message])
  (rf/dispatch [:toasts/close
                :delete-message-for-everyone]))

(defn delete
  "Delete message now locally and broadcast after undo time limit timeout"
  [{:keys [db]} [{:keys [chat-id message-id]} undo-time-limit-ms]]
  (when-let [message (get-in db [:messages chat-id message-id])]
    ;; all delete message toast are the same toast with id :delete-message-for-everyone new delete
    ;; operation will reset prev pending deletes' undo timelimit undo will undo all pending deletes
    ;; all pending deletes are stored in toast
    (let [unpin? (should-and-able-to-unpin-to-be-deleted-message
                  db
                  {:chat-id chat-id :message-id message-id})

          pub-key (get-in db [:profile/profile :public-key])

          ;; compute deleted by
          message-from (get message :from)
          deleted-by (when (not= pub-key message-from) pub-key)

          existing-undo-toast (get-in db [:toasts :toasts :delete-message-for-everyone])
          toast-count (inc (get existing-undo-toast :message-deleted-for-everyone-count 0))
          existing-undos
          (-> existing-undo-toast
              (get :message-deleted-for-everyone-undos [])
              (conj {:message-id message-id :chat-id chat-id :should-pin-back? unpin?}))]
      (assoc
       (message-list/rebuild-message-list
        {:db (reduce
              ;; sync all pending deletes' undo timelimit, extend to the latest one
              (fn [db-acc {:keys [chat-id message-id]}]
                (update-db-delete-locally db-acc chat-id message-id undo-time-limit-ms deleted-by))
              db
              existing-undos)}
        chat-id)

       :dispatch-n
       [[:toasts/close :delete-message-for-everyone]
        [:toasts/upsert
         {:id                                 :delete-message-for-everyone
          :type                               :negative
          :message-deleted-for-everyone-count toast-count
          :message-deleted-for-everyone-undos existing-undos
          :text                               (i18n/label-pluralize
                                               toast-count
                                               :t/message-deleted-for-everyone-count)
          :duration                           undo-time-limit-ms
          :undo-duration                      (/ undo-time-limit-ms 1000)
          :undo-on-press                      toast-undo-delete-message-for-everyone}]
        (when unpin?
          [:pin-message/send-pin-message-locally
           {:chat-id chat-id :message-id message-id :pinned false}])]
       :utils/dispatch-later (mapv (fn [{:keys [chat-id message-id]}]
                                     {:dispatch [:chat.ui/delete-message-and-send
                                                 {:chat-id chat-id :message-id message-id}]
                                      :ms       undo-time-limit-ms})
                                   existing-undos)))))

(rf/reg-event-fx :chat.ui/delete-message delete)

(defn undo
  [{:keys [db]} [{:keys [chat-id message-id should-pin-back?]}]]
  (when (get-in db [:messages chat-id message-id])
    (let [effects (message-list/rebuild-message-list
                   {:db (update-db-undo-locally db chat-id message-id)}
                   chat-id)
          message (get-in effects [:db :messages chat-id message-id])]
      (cond-> effects
        should-pin-back?
        (assoc :dispatch
               [:pin-message/send-pin-message-locally (assoc message :pinned true)])))))

(rf/reg-event-fx :chat.ui/undo-delete-message undo)

(defn undo-all
  [{:keys [db]}]
  (when-let [pending-undos (get-in db
                                   [:toasts :toasts :delete-message-for-everyone
                                    :message-deleted-for-everyone-undos])]
    {:dispatch-n (mapv #(vector :chat.ui/undo-delete-message %) pending-undos)}))

(rf/reg-event-fx :chat.ui/undo-all-delete-message undo-all)

(defn- check-before-delete-and-send
  "make sure message alredy deleted? locally and undo timelimit has passed"
  [db chat-id message-id]
  (let [message                                  (get-in db [:messages chat-id message-id])
        {:keys [deleted? deleted-undoable-till]} message]
    (and deleted?
         deleted-undoable-till
         (>= (datetime/timestamp) deleted-undoable-till))))

(defn delete-and-send
  [{:keys [db]} [{:keys [message-id chat-id]} force?]]
  (when (get-in db [:messages chat-id message-id])
    (when (or force? (check-before-delete-and-send db chat-id message-id))
      (let [unpin-locally?
            ;; this only check against local client data generally msg is already unpinned at delete
            ;; locally phase when user has unpin permission
            ;;
            ;; will be true only if
            ;; 1. admin delete an unpinned msg
            ;; 2. another admin pin the msg within the undo time limit
            ;; 3. msg finally deleted
            (should-and-able-to-unpin-to-be-deleted-message db
                                                            {:chat-id    chat-id
                                                             :message-id message-id})]
        {:db (update-db-clear-undo-timer db chat-id message-id)
         :fx [[:json-rpc/call
               [{:method      "wakuext_deleteMessageAndSend"
                 :params      [message-id]
                 :js-response true
                 :on-error    [:chat/delete-and-send-error message-id]
                 :on-success  [:sanitize-messages-and-process-response]}]]
              [:dispatch
               [:pin-message/send-pin-message
                {:chat-id      chat-id
                 :message-id   message-id
                 :pinned       false
                 :remote-only? (not unpin-locally?)}]]]}))))

(rf/reg-event-fx :chat.ui/delete-message-and-send delete-and-send)

(defn delete-and-send-error
  [_ [message-id error]]
  {:fx [[:effects.log/error
         ["failed to delete message"
          {:message-id message-id
           :error      error}]]]})

(rf/reg-event-fx :chat/delete-and-send-error delete-and-send-error)

(defn- filter-pending-send-messages
  "traverse all messages find not yet synced deleted? messages"
  [acc chat-id messages]
  (->> messages
       (filter (fn [[_ {:keys [deleted? deleted-undoable-till]}]] (and deleted? deleted-undoable-till)))
       (map (fn [message] {:chat-id chat-id :message-id (first message)}))
       (concat acc)))

(defn send-all
  "Get all deleted messages that not yet synced with status-go and send them"
  [{:keys [db] :as cofx}]
  (let [pending-send-messages (reduce-kv filter-pending-send-messages [] (:messages db))
        pending-effects       (map (fn [message]
                                     (fn [cofx]
                                       (delete-and-send cofx [message true])))
                                   pending-send-messages)]
    (apply rf/merge cofx pending-effects)))

(rf/reg-event-fx :chat.ui/send-all-deleted-messages send-all)
