(ns react-native.audio-toolkit
  (:require
    ["@react-native-community/audio-toolkit" :refer
     (Player Recorder MediaStates PlaybackCategories)]))

;; get mediastates from react module
(def PLAYING (.-PLAYING ^js MediaStates))
(def PAUSED (.-PAUSED ^js MediaStates))
(def RECORDING (.-RECORDING ^js MediaStates))
(def IDLE (.-IDLE ^js MediaStates))
(def DESTROYED (.-DESTROYED ^js MediaStates))

;; get PlaybackCategories from react module
(def PLAYBACK (.-Playback ^js PlaybackCategories))

(def default-recorder-options
  {:filename         "recording.aac"
   :bitrate          32000
   :channels         1
   :sampleRate       22050
   :quality          "medium" ; ios only
   :meteringInterval 50
   :category         PLAYBACK})

(defn get-state
  [player-recorder]
  (when player-recorder
    (.-state ^js player-recorder)))

(defn new-recorder
  [options on-meter on-ended]
  (let [recorder (new ^js Recorder
                      (:filename options)
                      (clj->js options))]
    (when on-meter
      (.on ^js recorder "meter" on-meter))
    (when on-ended
      (.on ^js recorder "ended" on-ended))
    recorder))

(defn new-player
  [audio options on-ended]
  (let [player (new ^js Player
                    audio
                    (clj->js options))]
    (when on-ended
      (.on ^js player "ended" on-ended))
    player))

(defn prepare-player
  [player on-prepared on-error]
  (when (and player (.-canPrepare ^js player))
    (.prepare ^js player
              (fn [^js err]
                (if err
                  (on-error {:error (.-err err) :message (.-message err)})
                  (on-prepared))))))

(defn start-recording
  [recorder on-start on-error]
  (when (and recorder
             (or
              (.-canRecord ^js recorder)
              (.-canPrepare ^js recorder)))
    (.record ^js recorder
             (fn [^js err]
               (if err
                 (on-error {:error (.-err err) :message (.-message err)})
                 (on-start))))))

(defn stop-recording
  [recorder on-stop on-error]
  (if (and recorder (#{RECORDING PAUSED} (get-state recorder)))
    (.stop ^js recorder
           (fn [^js err]
             (if err
               (on-error {:error (.-err err) :message (.-message err)})
               (on-stop))))
    (on-stop)))

(defn stop-playing
  [player on-stop on-error]
  (if (and player (.-isPlaying ^js player))
    (.stop ^js player
           (fn [^js err]
             (if err
               (on-error {:error (.-err err) :message (.-message err)})
               (on-stop))))
    (on-stop)))

(defn get-recorder-file-path
  [recorder]
  (when recorder
    (.-fsPath ^js recorder)))

(defn get-player-duration
  [player]
  (when (and player (.-canPlay ^js player))
    (.-duration ^js player)))

(defn get-player-current-time
  [player]
  (when (and player (.-canPlay ^js player))
    (.-currentTime ^js player)))

(defn set-player-wake-lock
  [player wake-lock?]
  (when player
    (set! (.-wakeLock player) wake-lock?)))

(defn toggle-playpause-player
  [player on-play on-pause on-error]
  (when (and player (.-canPlay ^js player))
    (.playPause ^js player
                (fn [^js error pause?]
                  (if error
                    (on-error {:error (.-err error) :message (.-message error)})
                    (if pause?
                      (on-pause)
                      (on-play)))))))

(defn seek-player
  [player value on-seek on-error]
  (when (and player (.-canPlay ^js player))
    (.seek ^js player
           value
           (fn [^js err]
             (if err
               (on-error {:error (.-err err) :message (.-message err)})
               (on-seek))))))

(defn destroy-recorder
  [recorder]
  (stop-recording recorder
                  #(when (and recorder (.-destroy ^js recorder) (not= (get-state recorder) DESTROYED))
                     (.destroy ^js recorder))
                  #()))

(defn destroy-player
  [player]
  (stop-playing player
                #(when (and player (.-destroy ^js player) (not= (get-state player) IDLE))
                   (.destroy ^js player))
                #()))
