(ns legacy.status-im.events
  (:require
    clojure.set
    legacy.status-im.browser.core
    legacy.status-im.browser.permissions
    legacy.status-im.chat.models.loading
    legacy.status-im.data-store.chats
    legacy.status-im.data-store.visibility-status-updates
    legacy.status-im.ens.core
    legacy.status-im.fleet.core
    legacy.status-im.group-chats.core
    legacy.status-im.log-level.core
    legacy.status-im.multiaccounts.login.core
    [legacy.status-im.multiaccounts.model :as multiaccounts.model]
    legacy.status-im.multiaccounts.update.core
    legacy.status-im.pairing.core
    legacy.status-im.profile.core
    legacy.status-im.search.core
    legacy.status-im.stickers.core
    legacy.status-im.ui.components.invite.events
    [legacy.status-im.ui.components.react :as react]
    [legacy.status-im.utils.dimensions :as dimensions]
    legacy.status-im.utils.logging.core
    [legacy.status-im.utils.utils :as utils]
    legacy.status-im.visibility-status-popover.core
    legacy.status-im.visibility-status-updates.core
    legacy.status-im.waku.core
    [native-module.core :as native-module]
    [re-frame.core :as re-frame]
    [react-native.core :as rn]
    [react-native.permissions :as permissions]
    [react-native.platform :as platform]
    status-im.common.serialization
    status-im.common.standard-authentication.events
    [status-im.common.theme.core :as theme]
    [status-im.common.universal-links :as universal-links]
    [status-im.constants :as constants]
    status-im.contexts.chat.home.events
    status-im.contexts.communities.home.events
    status-im.contexts.onboarding.events
    status-im.contexts.shell.activity-center.events
    status-im.contexts.shell.activity-center.notification.contact-requests.events
    [utils.i18n :as i18n]
    [utils.re-frame :as rf]))

(re-frame/reg-fx
 :dismiss-keyboard
 (fn []
   (react/dismiss-keyboard!)))

(re-frame/reg-fx
 :request-permissions-fx
 (fn [options]
   (permissions/request-permissions options)))

(re-frame/reg-fx
 :request-notifications-fx
 (fn [options]
   (permissions/request-notifications options)))

(re-frame/reg-fx
 :ui/show-confirmation
 (fn [options]
   (utils/show-confirmation options)))

(re-frame/reg-fx
 ::app-state-change-fx
 (fn [state]
   (when (and platform/ios? (= state "active"))
     ;; Change the app theme if the ios device theme was updated when the app was in the background
     ;; https://github.com/status-im/status-mobile/issues/15708
     (theme/change-device-theme (rn/get-color-scheme)))
   (native-module/app-state-change state)))

(rf/defn dismiss-keyboard
  {:events [:dismiss-keyboard]}
  [_]
  {:dismiss-keyboard nil})

(rf/defn gfycat-generated
  {:events [:gfycat-generated]}
  [{:keys [db]} path gfycat]
  {:db (assoc-in db path gfycat)})

(rf/reg-event-fx
 :system-theme-mode-changed
 (fn [{:keys [db]}]
   (let [appearance-type (-> db :profile/profile :appearance)]
     (when (and (multiaccounts.model/logged-in? db)
                (= appearance-type status-im.constants/appearance-type-system))
       {:fx [[:dispatch [:theme/switch]]]}))))

(defn- on-biometric-auth-fail
  [{:keys [code]}]
  (if (= code "USER_FALLBACK")
    (re-frame/dispatch [:profile/logout])
    (utils/show-confirmation
     {:title               (i18n/label :t/biometric-auth-confirm-title)
      :content             (i18n/label :t/biometric-auth-confirm-message)
      :confirm-button-text (i18n/label :t/biometric-auth-confirm-try-again)
      :cancel-button-text  (i18n/label :t/biometric-auth-confirm-logout)
      :on-accept           #(rf/dispatch [:biometric/authenticate {:on-fail on-biometric-auth-fail}])
      :on-cancel           #(re-frame/dispatch [:profile/logout])})))

(rf/defn on-return-from-background
  [{:keys [db now] :as cofx}]
  (let [new-account?            (get db :onboarding/new-account?)
        app-in-background-since (get db :app-in-background-since)
        signed-up?              (get-in db [:profile/profile :signed-up?])
        notifications-settings? (= (:view-id db) :screen/settings.notifications)
        requires-bio-auth       (and
                                 signed-up?
                                 (= (:auth-method db) "biometric")
                                 (some? app-in-background-since)
                                 (>= (- now app-in-background-since)
                                     constants/ms-in-bg-for-require-bioauth))]
    (rf/merge cofx
              {:db                                   (dissoc db :app-in-background-since)
               :effects.biometric/get-supported-type nil}
              (when-not new-account?
                (universal-links/process-stored-event))
              #(when-let [chat-id (:current-chat-id db)]
                 {:dispatch [:chat/mark-all-as-read chat-id]})
              #(when requires-bio-auth
                 {:dispatch [:biometric/authenticate {:on-fail on-biometric-auth-fail}]})
              #(when notifications-settings?
                 {:dispatch [:notifications/check-notifications-blocked]}))))

(rf/defn on-going-in-background
  [{:keys [db now]}]
  {:db (-> db
           (assoc :app-in-background-since now)
           (dissoc :universal-links/handling))})
   ;; event not implemented
   ;; :dispatch-n [[:audio-recorder/on-background] [:audio-message/on-background]]


(rf/defn app-state-change
  {:events [:app-state-change]}
  [{:keys [db] :as cofx} state]
  (let [app-coming-from-background? (= state "active")
        app-going-in-background?    (= state "background")]
    (rf/merge cofx
              {::app-state-change-fx state
               :db                   (assoc db :app-state state)}
              #(when app-coming-from-background?
                 (on-return-from-background %))
              #(when app-going-in-background?
                 (on-going-in-background %)))))

(rf/defn request-permissions
  {:events [:request-permissions]}
  [_ options]
  {:request-permissions-fx options})

(rf/reg-event-fx
 :request-notifications
 (fn [_ [options]]
   (prn options)
   {:request-notifications-fx options}))

(rf/defn update-window-dimensions
  {:events [:update-window-dimensions]}
  [{:keys [db]} dimensions]
  {:db (assoc db :dimensions/window (dimensions/window dimensions))})

(rf/defn on-will-focus
  {:events [:screens/on-will-focus]}
  [{:keys [db] :as cofx} view-id]
  (rf/merge cofx
            (cond
              (= :screen/chat view-id)
              {:effects.async-storage/set {:chat-id (get-in cofx [:db :current-chat-id])
                                           :key-uid (get-in cofx [:db :profile/profile :key-uid])}
               :db                        (assoc db :screens/was-focused-once? true)
               :dispatch                  [:alert-banners/unhide]}

              (not (get db :screens/was-focused-once?))
              {:db (assoc db :screens/was-focused-once? true)})))


;;TODO :replace by named events
(rf/defn set-event
  {:events [:set]}
  [{:keys [db]} k v]
  {:db (assoc db k v)})

;;TODO :replace by named events
(rf/defn set-once-event
  {:events [:set-once]}
  [{:keys [db]} k v]
  (when-not (get db k)
    {:db (assoc db k v)}))

;;TODO :replace by named events
(rf/defn set-in-event
  {:events [:set-in]}
  [{:keys [db]} path v]
  {:db (assoc-in db path v)})

(defn on-ramp<-rpc
  [on-ramp]
  (clojure.set/rename-keys on-ramp
                           {:logoUrl :logo-url
                            :siteUrl :site-url}))

(rf/defn crypto-loaded-event
  {:events [::crypto-loaded]}
  [{:keys [db]} on-ramps]
  {:db (assoc
        db
        :buy-crypto/on-ramps
        (map on-ramp<-rpc on-ramps))})

(rf/defn buy-crypto-ui-loaded
  {:events [:buy-crypto.ui/loaded]}
  [_]
  {:json-rpc/call [{:method     "wallet_getCryptoOnRamps"
                    :params     []
                    :on-success (fn [on-ramps]
                                  (re-frame/dispatch [::crypto-loaded on-ramps]))}]})
