package godau.fynn.moodledirect.view

import androidx.annotation.StringRes
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import godau.fynn.moodledirect.OfflineException
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.util.ExceptionHandler.networkErrorMessage
import java.io.IOException

@Composable
fun BoxScope.NoDataView(error: ErrorState) = Column(Modifier.align(Alignment.Center).width(IntrinsicSize.Max).padding(32.dp)) {
    when (error) {
        is NotAvailableOffline -> R.drawable.ic_cloud_off
        is ExceptionOccurred -> when (error.e) {
            is OfflineException -> R.drawable.ic_cloud_off
            else -> R.drawable.ic_error
        }
        is NoResults -> null
    }?.let {
        Image(
            painter = painterResource(id = it),
            contentDescription = null,
            modifier = Modifier.fillMaxWidth()
        )

        Spacer(modifier = Modifier.height(12.dp))
    }

    val errorText = when (error) {
        is NotAvailableOffline -> error.text
        is NoResults -> error.text
        is ExceptionOccurred -> when (error.e) {
            is IOException -> error.e.networkErrorMessage()
            is OfflineException -> R.string.offline_not_available
            else -> R.string.error_generic
        }
    }.let { stringResource(id = it) }
    Text(text = errorText, modifier = Modifier.fillMaxWidth(), style = MaterialTheme.typography.body1)

}

sealed class ErrorState
data class NotAvailableOffline(@StringRes val text: Int = R.string.offline_not_available) :
    ErrorState()

data class ExceptionOccurred(val e: Exception) : ErrorState()
data class NoResults(@StringRes val text: Int) : ErrorState()