package godau.fynn.moodledirect.view

import android.content.res.Resources
import android.util.Log
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.material.Divider
import androidx.compose.material.Icon
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.material.ripple.rememberRipple
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import coil.decode.SvgDecoder
import coil.request.ImageRequest
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.model.ResourceType
import godau.fynn.moodledirect.model.api.file.File
import godau.fynn.moodledirect.model.database.CourseSection
import godau.fynn.moodledirect.model.database.Module
import godau.fynn.moodledirect.module.FileManager.DownloadStatus
import godau.fynn.moodledirect.module.link.ModuleLink.Companion.getIcon

const val HIERARCHY_INTENSITY = 3

@Composable
fun ModuleList(
    modules: List<Module>,
    fileStates: Map<File, DownloadStatus>,
    modifier: Modifier = Modifier,
    moduleClick: (Module) -> Unit
) = LazyColumn(modifier) {
    itemsIndexed(modules) { index: Int, module: Module ->

        ModuleRow(module, fileStates) { moduleClick(module) }

        val isLast = index >= modules.size - 1
        if (!isLast) {
            val next: Module = modules[index + 1]

            // Show dividers above course sections and labels
            var needsDivider =
                ((next is CourseSection) or (next.getModuleType() == ResourceType.LABEL)) and
                        // Show no dividers below course sections and labels (don't separate them from their content)
                        (module !is CourseSection && module.getModuleType() != ResourceType.LABEL)

            needsDivider = needsDivider or (next.hierarchyDepth != module.hierarchyDepth)

            val dividerDepth = next.hierarchyDepth

            if (needsDivider) Divider(Modifier.padding(start = (dividerDepth * HIERARCHY_INTENSITY * 16).dp))
        }
    }
}

@Composable
fun ModuleRow(module: Module, fileStates: Map<File, DownloadStatus>, moduleClick: () -> Unit) =
    if (module is CourseSection) {
        CourseSectionRow(module)
    } else if (module.getModuleType() == ResourceType.LABEL) {
        LabelModuleRow(module)
    } else {
        ItemModuleRow(module, fileStates, moduleClick)
    }

@Composable
fun CourseSectionRow(module: CourseSection) {
    Column(Modifier.fillMaxWidth().padding(start = (module.hierarchyDepth * HIERARCHY_INTENSITY * 16).dp)) {
        Text(
            text = module.name,
            modifier = Modifier
                .padding(top = 16.dp, start = 16.dp, end = 16.dp)
                .alpha(
                    if (module.available) 1f
                    else 0.5f
                ),
            fontWeight = FontWeight.Bold,
            style = MaterialTheme.typography.h5
        )
        ModuleDescriptionText(
            module = module,
            modifier = Modifier
                .padding(horizontal = 16.dp, vertical = 8.dp)
                .alpha(
                    if (module.available) 1f
                    else 0.5f
                )
        )

        if (!module.available && module.notAvailableReason != null) {
            RestrictionRow(module)
        }
    }
    // TODO: expandable text
}

@Composable
fun LabelModuleRow(module: Module) {
    ModuleDescriptionText(
        module = module, modifier = Modifier
            .padding(start = ((1 + module.hierarchyDepth * HIERARCHY_INTENSITY) * 16).dp, end = 16.dp, top = 8.dp, bottom = 8.dp)
            .fillMaxWidth()
    )
    // TODO: expandable text
}

@Composable
fun ItemModuleRow(
    module: Module,
    fileStates: Map<File, DownloadStatus>,
    moduleClick: () -> Unit
) {
    if (module.available || module.notAvailableReason == null) {
        ItemModuleRowContent(module, fileStates, moduleClick)
    } else {
        Column {
            ItemModuleRowContent(module, fileStates, moduleClick)
            RestrictionRow(module)
        }
    }
}

@Composable
fun ItemModuleRowContent(
    module: Module,
    fileStates: Map<File, DownloadStatus>,
    moduleClick: () -> Unit
) {
    // TODO: expandable text
    val downloadStatus = if (module.isDownloadable && module.getModuleType() == ResourceType.FILE) {
        // Per Module.isDownloadable(), FILEs have exactly one downloadable file
        fileStates[module.fileList[0]]
    } else null

    Row(
        Modifier
            .fillMaxWidth()
            .clickable(
                interactionSource = remember { MutableInteractionSource() },
                indication = rememberRipple(),
                enabled = module.available
            ) {
                moduleClick()
            }
            .padding(start = ((1 + module.hierarchyDepth * HIERARCHY_INTENSITY) * 16).dp, end = 16.dp, top = 16.dp, bottom = 16.dp)
            .let {
                if (module.available) it
                else it.alpha(0.5f)
            }
    ) {
        val resourceIcon: Int = module.moduleIcon + getIcon(module)
        val iconModifier = Modifier
            .size(32.dp)
            .align(Alignment.Top)
        val resourceIconView = @Composable { iconModifier: Modifier ->
            if (resourceIcon != Resources.ID_NULL) {
                Icon(
                    painter = painterResource(resourceIcon),
                    contentDescription = null,
                    modifier = iconModifier,
                    tint = Color.Unspecified
                )
            } else {
                val defaultIcon = painterResource(id = R.drawable.ic_file_generic)
                AsyncImage(
                    model = ImageRequest.Builder(LocalContext.current)
                        .data(module.moduleIconUrl)
                        .decoderFactory(SvgDecoder.Factory())
                        .build(),
                    contentDescription = null,
                    modifier = iconModifier,
                    placeholder = defaultIcon,
                    error = defaultIcon,
                    onError = {
                        Log.w("ModuleRow", it.result.throwable)
                    }
                )
            }
        }

        if (downloadStatus == null || downloadStatus == DownloadStatus.DOWNLOADED) {
            resourceIconView(iconModifier)
        } else {
            Box(iconModifier) {
                resourceIconView(iconModifier.alpha(0.25f))

                if (downloadStatus == DownloadStatus.DOWNLOADING) {
                    CircularProgressIndicator(iconModifier, colorResource(id = R.color.colorAccent))
                }

                if (downloadStatus != DownloadStatus.DOWNLOADING) {
                    Icon(
                        painter = painterResource(
                            id = when (downloadStatus) {
                                DownloadStatus.NOT_DOWNLOADED -> R.drawable.ic_download
                                DownloadStatus.UPDATE_AVAILABLE -> R.drawable.ic_update
                                DownloadStatus.FAILED -> R.drawable.ic_error
                                DownloadStatus.DOWNLOADING, DownloadStatus.DOWNLOADED -> throw IllegalStateException()
                            }
                        ),
                        contentDescription = null,
                        modifier = Modifier
                            .size(16.dp)
                            .align(Alignment.BottomEnd)
                    )
                }
            }
        }

        Column(
            Modifier
                .padding(horizontal = 16.dp)
                .align(Alignment.CenterVertically)
        ) {
            Text(text = module.name, style = MaterialTheme.typography.body1)

            ModuleDescriptionText(module = module, Modifier.padding(top = 8.dp))
        }


    }

}