package godau.fynn.moodledirect.view

import android.view.View
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.material.ripple.rememberRipple
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import coil.compose.AsyncImage
import coil.request.ImageRequest
import godau.fynn.moodledirect.data.persistence.PreferenceHelper
import godau.fynn.moodledirect.model.database.Course
import godau.fynn.moodledirect.util.TextUtil

typealias Token = String

const val WIDTH_HEADER_IMAGE = 120

@Composable
fun CourseRow(
    course: Course,
    token: Token,
    appearance: PreferenceHelper.CourseRowAppearance = PreferenceHelper.CourseRowAppearance(),
    onClick: () -> Unit
) = Row(
    Modifier
        .height(IntrinsicSize.Min)
        .fillMaxWidth()
        .clickable(
            interactionSource = remember { MutableInteractionSource() },
            indication = rememberRipple(),
            onClick = onClick,
        )
        .background(MaterialTheme.colors.background)
) {
    Spacer(modifier = Modifier.width(16.dp))
    Column(Modifier.weight(1f)) {
        Spacer(modifier = Modifier.height(16.dp))
        // Set category text
        when (appearance.categoryAppearance) {
            "none" -> null
            "top" -> course.topCategoryName

            "both" -> if (course.topCategoryName != null && course.categoryName != course.topCategoryName) {
                "${course.topCategoryName} / ${course.categoryName}"
            } else {
                course.categoryName
            }

            "bottom" -> course.categoryName
            else -> course.categoryName
        }?.let { categoryText ->
            Text(
                text = categoryText.uppercase(),
                letterSpacing = 1.5.sp,
                fontWeight = FontWeight.Bold,
                modifier = Modifier.alpha(0.5f).padding(bottom = 2.dp)
            )
        }

        // Course name
        Text(text = course.name, style = MaterialTheme.typography.body1.copy(lineHeight = 22.sp))


        // Set course summary text
        if (!appearance.showDescription || course.summary.trim { it <= ' ' }.isEmpty()) {
            null
        } else {
            val charSequence: CharSequence = TextUtil.fromHtml(course.summary, LocalContext.current)
            if (TextUtil.hasContent(charSequence)) {
                TextUtil.crop(charSequence, 120)
            } else {
                null
            }
        }?.let { descriptionText ->
            Text(
                text = descriptionText.toString(),
                Modifier
                    .alpha(0.5f)
                    .padding(top = 0.dp)
            )
        }

        Spacer(modifier = Modifier.height(16.dp))
    }

    Spacer(modifier = Modifier.width(16.dp))

    // Header image
    if (appearance.showHeaderImage) course.headerImage?.let { header ->
        Box(
            Modifier
                .width(120.dp)
                .fillMaxHeight()
        ) {

            AsyncImage(
                model = ImageRequest.Builder(LocalContext.current)
                    .data("${header.url}?token=$token")
                    .crossfade(200)
                    .build(),
                contentDescription = null,
                contentScale = ContentScale.Crop,
                modifier = Modifier
                    .width(WIDTH_HEADER_IMAGE.dp)
                    .fillMaxHeight()
            )

            val backgroundColor = MaterialTheme.colors.background
            val ltr = LocalConfiguration.current.layoutDirection == View.LAYOUT_DIRECTION_LTR
            Canvas(
                modifier = Modifier
                    .width(WIDTH_HEADER_IMAGE.dp)
                    .fillMaxHeight()
            ) {
                drawRect(
                    Brush.horizontalGradient(
                        colors = listOf(backgroundColor, Color.Transparent),
                        if (ltr) 0f else WIDTH_HEADER_IMAGE.dp.toPx(),
                        if (ltr) 18.dp.toPx() else (WIDTH_HEADER_IMAGE - 18).dp.toPx()
                    )
                )
            }
        }
    }
}