package godau.fynn.moodledirect.util;

import android.app.Application;
import android.content.Context;
import android.graphics.drawable.Drawable;
import android.net.ConnectivityManager;
import android.net.NetworkRequest;
import android.net.Uri;
import android.os.Build;
import android.widget.ImageView;

import com.mikepenz.materialdrawer.util.AbstractDrawerImageLoader;
import com.mikepenz.materialdrawer.util.DrawerImageLoader;
import com.squareup.picasso.NetworkPolicy;
import com.squareup.picasso.Picasso;
import com.squareup.picasso.RequestCreator;

import godau.fynn.moodledirect.data.persistence.MoodleDatabase;
import godau.fynn.moodledirect.data.persistence.PreferenceHelper;
import godau.fynn.moodledirect.data.persistence.UserAccount;
import godau.fynn.moodledirect.network.NetworkStateReceiver;

public class MyApplication extends Application {

    private static MyApplication instance;
    private MoodleDatabase moodle;
    private boolean isDarkMode = false;

    public final NetworkStateReceiver networkStateReceiver = new NetworkStateReceiver();

    public static synchronized MyApplication getInstance() {
        return instance;
    }

    public static MoodleDatabase moodle() {
        if (instance.moodle == null) instance.moodle = MoodleDatabase.get(instance);
        return instance.moodle;
    }

    public static void clearMoodle() {
        if (instance.moodle != null) {
            instance.moodle.close();
        }
        instance.moodle = null;
    }

    public static void switchMoodleToAccount(UserAccount newAccount) {
        clearMoodle();
        instance.moodle = MoodleDatabase.get(instance, newAccount.getDatabaseName());
    }

    /**
     * Ensures that the database is closed at least once (to remove temporary files).
     */
    public static void commitDatabase() {
        moodle().close();
        instance.moodle = null;
    }

    @Override
    public void onCreate() {
        super.onCreate();
        instance = this;

        PreferenceHelper preferences = new PreferenceHelper(this);
        if (preferences.isLoggedIn()) {
            Constants.API_URL = preferences.getActiveAccount().getUrl();
        }

        networkStateReceiver.onUpdateForce(preferences.isForceOfflineModeEnabled());

        this.isDarkMode = preferences.isDarkThemeEnabled();

        ConnectivityManager connectivityManager = ((ConnectivityManager) getSystemService(CONNECTIVITY_SERVICE));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            connectivityManager.registerDefaultNetworkCallback(networkStateReceiver);
        } else {
            connectivityManager.registerNetworkCallback(new NetworkRequest.Builder().build(), networkStateReceiver);
        }
        networkStateReceiver.update(connectivityManager);

        DrawerImageLoader.init(new AbstractDrawerImageLoader() {
            @Override
            public Drawable placeholder(Context ctx, String tag) {
                return super.placeholder(ctx, tag);
            }

            @Override
            public void cancel(ImageView imageView) {
                Picasso.get().cancelRequest(imageView);
            }

            @Override
            public void set(ImageView imageView, Uri uri, Drawable placeholder, String tag) {
                // Load avatar, don't attempt to use network if offline
                RequestCreator creator = Picasso.get().load(
                        uri
                );
                if (NetworkStateReceiver.getOfflineStatus()) {
                    creator.networkPolicy(NetworkPolicy.OFFLINE);
                }
                creator.into(imageView);

                super.set(imageView, uri, placeholder, tag);
            }
        });
    }

    public boolean isDarkModeEnabled() {
        return this.isDarkMode;
    }

    public void setDarkModeEnabled(boolean isEnabled) {
        this.isDarkMode = isEnabled;
    }
}