package godau.fynn.moodledirect.util

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.net.Uri
import android.util.Log
import androidx.annotation.UiThread
import androidx.annotation.WorkerThread
import androidx.appcompat.app.AlertDialog
import androidx.documentfile.provider.DocumentFile
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.data.persistence.PreferenceHelper
import godau.fynn.moodledirect.model.api.file.File
import godau.fynn.moodledirect.model.database.Module
import godau.fynn.moodledirect.module.FileManager
import godau.fynn.moodledirect.module.files.HasFiles
import godau.fynn.moodledirect.util.ExceptionHandler.ExceptionableSupplier
import godau.fynn.moodledirect.util.ExceptionHandler.tryAndThenThread
import java.util.function.Consumer
import java.util.stream.Collectors
import java.util.stream.Stream
import kotlin.math.min

/**
 * Convenience method for access to [FileManager]
 */
class FileManagerWrapper(private val fileManager: FileManager, private val context: Context) {
    @WorkerThread
    fun getLocation(file: File?): Uri {
        return fileManager.getLocation(file)
    }

    @WorkerThread
    fun addDownloadStatus(listHasFiles: List<HasFiles?>?) {
        if (listHasFiles == null) return
        val files = listHasFiles.stream()
            .filter { hasFiles: HasFiles? -> if (hasFiles is Module) return@filter hasFiles.isDownloadable else return@filter true }
            .flatMap { hasFiles: HasFiles? ->
                if (hasFiles!!.getFileList() != null) return@flatMap hasFiles.getFileList()
                    .stream() else return@flatMap Stream.empty<File>()
            }
            .collect(Collectors.toList())

        // Avoid too big queries to addDownloadStatus
        var i = 0
        var last = 0
        do {
            i = min(i + 30, files.size)
            fileManager.addDownloadStatus(files.subList(last, i))
            last = i
        } while (i < files.size)
        for (file in files) {
            if (fileThreadMap.containsKey(file)) {
                file.downloadStatus = FileManager.DownloadStatus.DOWNLOADING
            }
        }
    }

    /**
     * Creates intent to open the provided file.
     *
     * @param file         File to be opened
     * @param doesNotExist To be run if desired file does not exist
     * @param context      Context from which to spawn AlertDialog
     */
    @UiThread
    fun openFile(file: File, doesNotExist: Runnable, context: Context) {
        context.tryAndThenThread<Uri>(
            { fileManager.getLocation(file) },
            { location: Uri? ->
                if (location == null) {
                    doesNotExist.run()
                } else {
                    val intent = Intent(Intent.ACTION_VIEW)
                    intent.setDataAndType(location, file.mimetype)
                    intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION or Intent.FLAG_ACTIVITY_NEW_TASK)
                    intent.putExtra(Intent.EXTRA_TITLE, file.filename)
                    try {
                        if (intent.resolveActivity(context.packageManager) == null) throw ActivityNotFoundException()
                        context.startActivity(intent)
                    } catch (e: ActivityNotFoundException) {
                        AlertDialog.Builder(context)
                            .setTitle(R.string.error_activity_not_found_title)
                            .setMessage(
                                context.getString(
                                    R.string.error_activity_not_found_message,
                                    file.mimetype
                                )
                            )
                            .show()
                    }
                }
            },
            { exception: Exception -> exception.printStackTrace()}
        )
    }

    @UiThread @Deprecated(message = "use startDownload(File, String, String, Context, (Uri) -> Unit)")
    fun startDownload(
        file: File,
        courseShortName: String,
        moduleName: String?,
        context: Context,
        then: Consumer<Uri?>
    ): Boolean = startDownload(file, courseShortName, moduleName, context, { state -> file.downloadStatus = state}) {
        then.accept(it)
    }

    @UiThread
    fun startDownload(
        file: File,
        courseShortName: String,
        moduleName: String? = null,
        context: Context,
        setState: (FileManager.DownloadStatus) -> Unit,
        then: ((Uri?) -> Unit)? = null
    ): Boolean {
        Log.d(
            FileManagerWrapper::class.java.getSimpleName(),
            "Attempting to start download for: " + file.url +
                    " (" + file.filename + ") "
        )

        // Download path set?
        if (fileManager.getDownloadUri() == null) {
            AlertDialog.Builder(context)
                .setTitle(R.string.download_path_dialog_title)
                .setMessage(R.string.download_path_dialog_message)
                .setPositiveButton(
                    R.string._continue
                ) { dialog: DialogInterface?, which: Int ->
                    (context as Activity)
                        .startActivityForResult(requestPermissionIntent(), REQUEST_OPEN_DIRECTORY)
                }
                .setNegativeButton(R.string._cancel, null)
                .show()
            Log.d(
                FileManagerWrapper::class.java.getSimpleName(),
                "No storage location configured, can't download."
            )
            return false
        }

        // Download path available?
        val downloadDirectory = DocumentFile.fromTreeUri(context, fileManager.getDownloadUri())
        if (downloadDirectory == null || !downloadDirectory.exists()) {
            AlertDialog.Builder(context)
                .setTitle(R.string.download_path_unavailable_dialog_title)
                .setMessage(R.string.download_path_unavailable_dialog_message)
                .setPositiveButton(
                    R.string.download_path_unavailable_dialog_change
                ) { _, _ ->
                    (context as Activity)
                        .startActivityForResult(requestPermissionIntent(), REQUEST_OPEN_DIRECTORY)
                }
                .setNeutralButton(
                    R.string.download_path_unavailable_dialog_info
                ) { _, _ ->
                    AlertDialog.Builder(context).setMessage(
                        context.getString(
                            R.string.download_path_unavailable_info_dialog_message,
                            fileManager.getDownloadUri().toString()
                        )
                    ).show()
                }
                .setNegativeButton(R.string._cancel, null)
                .show()
            Log.d(
                FileManagerWrapper::class.java.getSimpleName(),
                "Download path is no longer available, can't download."
            )
            return false
        }
        val thread = context.tryAndThenThread(
            { fileManager.download(file, courseShortName, moduleName) },
            { fileUri: Uri? ->
                setState(FileManager.DownloadStatus.DOWNLOADED)
                fileThreadMap.remove(file)
                then?.invoke(fileUri)
                Log.d(
                    FileManagerWrapper::class.java.getSimpleName(),
                    "Download of " + file.url + " successful."
                )
            },
            { otherwise: Exception ->
                setState(FileManager.DownloadStatus.FAILED)
                fileThreadMap.remove(file)
                then?.invoke(null)
                Log.d(
                    FileManagerWrapper::class.java.getSimpleName(),
                    "Download of " + file.url + " failed."
                )
            }
        )
        setState(FileManager.DownloadStatus.DOWNLOADING)
        fileThreadMap[file] = thread
        return true
    }

    companion object {
        const val REQUEST_OPEN_DIRECTORY = 346 // phonecode for DIR

        /**
         * Track which files are currently being downloaded
         */
        private val fileThreadMap: MutableMap<File, Thread> = HashMap()
        @JvmStatic
        fun requestPermissionIntent(): Intent {
            return Intent(Intent.ACTION_OPEN_DOCUMENT_TREE)
        }

        @JvmStatic
        fun onRequestPermissionResult(data: Intent?, context: Context) {
            if (data != null) {
                val uri = data.data
                val takeFlags = (data.flags
                        and (Intent.FLAG_GRANT_READ_URI_PERMISSION
                        or Intent.FLAG_GRANT_WRITE_URI_PERMISSION))
                // Accept offer to persist this permission (noinspection WrongConstant)
                context.contentResolver.takePersistableUriPermission(uri!!, takeFlags)
                PreferenceHelper(context).downloadPath = uri.toString()
            }
        }
    }
}
