package godau.fynn.moodledirect.util

import android.content.Context
import android.content.res.Resources
import android.database.sqlite.SQLiteException
import android.os.Handler
import android.os.Looper
import android.widget.Toast
import androidx.annotation.StringRes
import androidx.annotation.UiThread
import androidx.annotation.WorkerThread
import androidx.appcompat.app.AlertDialog
import com.google.gson.JsonParseException
import com.google.gson.JsonSyntaxException
import godau.fynn.moodledirect.OfflineException
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.model.api.MoodleException
import godau.fynn.moodledirect.module.FileManager.DirectoryCreationException
import godau.fynn.moodledirect.module.FileManager.FileCreationException
import godau.fynn.moodledirect.network.exception.NotOkayException
import godau.fynn.moodledirect.view.dialog.MoodleExceptionDialog
import kotlinx.coroutines.runBlocking
import java.io.FileNotFoundException
import java.io.IOException
import java.io.InterruptedIOException
import java.net.ConnectException
import java.net.NoRouteToHostException
import java.net.UnknownHostException
import java.util.function.Consumer
import javax.net.ssl.SSLException

object ExceptionHandler {
    /**
     * Handles exceptions that occur during execution of `runnable`. In case of exception, displays
     * message to user on main thread. In case of success, passes
     * result to `resultHandler`.
     */
    @WorkerThread
    suspend fun <T> tryAndThen(
        runnable: ExceptionableSupplier<T>,
        resultHandler: Consumer<T>,
        context: Context
    ) {
        tryAndThen(runnable, resultHandler, null, context)
    }

    /**
     * Handles exceptions that occur during execution of `runnable`. In case of exception, displays
     * message to user on main thread and also invokes `alsoOnFailure`. In case of success, passes
     * result to `resultHandler` on main thread.
     */
    @WorkerThread
    suspend fun <T> tryAndThen(
        runnable: ExceptionableSupplier<T>,
        resultHandler: Consumer<T>,
        alsoOnFailure: Consumer<Exception?>?,
        context: Context
    ) {
        try {
            val t = runnable.run()
            Handler(Looper.getMainLooper()).post { resultHandler.accept(t) }
        } catch (e: MoodleException) {
            e.printStackTrace()
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.accept(e)
                if (e.getErrorCode() == MoodleException.ErrorCode.INVALID_TOKEN) {
                    AlertDialog.Builder(context, R.style.LoginTheme_AlertDialog)
                        .setTitle(R.string.moodle_exception_invalid_token)
                        .setMessage(R.string.moodle_exception_invalid_token_message)
                        .setPositiveButton(
                            R.string._continue
                        ) { _, _ ->
                            UserUtils.startLoginActivityForCurrentInstance(
                                context
                            )
                        }
                        .setNegativeButton(R.string._ignore, null)
                        .setCancelable(false).show()
                } else if (e.getErrorCode() == MoodleException.ErrorCode.INVALID_PRIVATE_TOKEN) {
                    Toast.makeText(
                        context,
                        R.string.moodle_exception_invalid_private_token,
                        Toast.LENGTH_LONG
                    ).show()
                } else if (e.getErrorCode() == MoodleException.ErrorCode.AUTO_LOGIN_TIME_LIMITED) {
                    Toast.makeText(
                        context,
                        R.string.moodle_exception_auto_login_time_limited,
                        Toast.LENGTH_LONG
                    ).show()
                    // Fix: update auto login cooldown
                    ConfigDownloadHelper.updateAutoLoginCooldown(context)
                } else {
                    MoodleExceptionDialog(e, context).show()
                }
            }
        } catch (e: FileNotFoundException) {
            e.printStackTrace()
            // Suppress. Users should be informed that the operation will (likely) not succeed ahead of time.
            Handler(Looper.getMainLooper()).post { alsoOnFailure?.accept(e) }
        } catch (e: IOException) {
            @StringRes val errorMessage: Int = e.networkErrorMessage()
            e.printStackTrace()
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.accept(e)
                Toast.makeText(context, errorMessage, Toast.LENGTH_SHORT).show()
            }
        } catch (e: JsonSyntaxException) {
            @StringRes val errorMessage: Int = R.string.error_network_json
            e.printStackTrace()
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.accept(e)
                Toast.makeText(context, errorMessage, Toast.LENGTH_SHORT).show()
            }
        } catch (e: SQLiteException) {
            e.printStackTrace()
            if (e.message != null && e.message!!.contains("too many SQL variables")) {
                Handler(Looper.getMainLooper()).post {
                    alsoOnFailure?.accept(e)
                    Toast.makeText(
                        context,
                        R.string.error_sqlite_too_many_variables,
                        Toast.LENGTH_SHORT
                    ).show()
                }
            } else {
                Handler(Looper.getMainLooper()).post {
                    AlertDialog.Builder(context)
                        .setTitle(R.string.error_sqlite_other)
                        .setMessage(
                            context.getString(
                                R.string.error_sqlite_other_message,
                                e.message
                            )
                        )
                        .show()
                    alsoOnFailure?.accept(e)
                }
            }
        } catch (e: OfflineException) {
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.accept(e)
                if (e.causeMessage != Resources.ID_NULL) AlertDialog.Builder(context)
                    .setMessage(e.causeMessage)
                    .setCancelable(true)
                    .show()
            }
        } catch (e: DirectoryCreationException) {
            Handler(Looper.getMainLooper()).post {
                AlertDialog.Builder(context)
                    .setTitle(R.string.error_directory_creation)
                    .setMessage(context.getString(R.string.error_directory_creation_message))
                    .show()
                alsoOnFailure?.accept(e)
            }
        } catch (e: FileCreationException) {
            Handler(Looper.getMainLooper()).post {
                AlertDialog.Builder(context)
                    .setTitle(R.string.error_file_creation)
                    .setMessage(context.getString(R.string.error_file_creation_message))
                    .show()
                alsoOnFailure?.accept(e)
            }
        }
    }

    @WorkerThread
    suspend fun <T> Context.tryAndThen(
        runnable: suspend () -> T,
        resultHandler: (T) -> Unit,
        alsoOnFailure: ((Exception) -> Unit)? = null
    ) {
        try {
            val t = runnable()
            Handler(Looper.getMainLooper()).post { resultHandler(t) }
        } catch (e: MoodleException) {
            e.printStackTrace()
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.let { it(e) }
                if (e.getErrorCode() == MoodleException.ErrorCode.INVALID_TOKEN) {
                    AlertDialog.Builder(this, R.style.LoginTheme_AlertDialog)
                        .setTitle(R.string.moodle_exception_invalid_token)
                        .setMessage(R.string.moodle_exception_invalid_token_message)
                        .setPositiveButton(
                            R.string._continue
                        ) { _, _ ->
                            UserUtils.startLoginActivityForCurrentInstance(this)
                        }
                        .setNegativeButton(R.string._ignore, null)
                        .setCancelable(false).show()
                } else if (e.getErrorCode() == MoodleException.ErrorCode.INVALID_PRIVATE_TOKEN) {
                    Toast.makeText(
                        this,
                        R.string.moodle_exception_invalid_private_token,
                        Toast.LENGTH_LONG
                    ).show()
                } else if (e.getErrorCode() == MoodleException.ErrorCode.AUTO_LOGIN_TIME_LIMITED) {
                    Toast.makeText(
                        this,
                        R.string.moodle_exception_auto_login_time_limited,
                        Toast.LENGTH_LONG
                    ).show()
                    // Fix: update auto login cooldown
                    ConfigDownloadHelper.updateAutoLoginCooldown(this)
                } else {
                    MoodleExceptionDialog(e, this).show()
                }
            }
        } catch (e: FileNotFoundException) {
            e.printStackTrace()
            // Suppress. Users should be informed that the operation will (likely) not succeed ahead of time.
            Handler(Looper.getMainLooper()).post { alsoOnFailure?.let { it(e) } }
        } catch (e: IOException) {
            @StringRes val errorMessage: Int = e.networkErrorMessage()
            e.printStackTrace()
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.let { it(e) }
                Toast.makeText(this, errorMessage, Toast.LENGTH_SHORT).show()
            }
        } catch (e: JsonSyntaxException) {
            @StringRes val errorMessage: Int = R.string.error_network_json
            e.printStackTrace()
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.let { it(e) }
                Toast.makeText(this, errorMessage, Toast.LENGTH_SHORT).show()
            }
        } catch (e: SQLiteException) {
            e.printStackTrace()
            if (e.message != null && e.message!!.contains("too many SQL variables")) {
                Handler(Looper.getMainLooper()).post {
                    alsoOnFailure?.let { it(e) }
                    Toast.makeText(
                        this,
                        R.string.error_sqlite_too_many_variables,
                        Toast.LENGTH_SHORT
                    ).show()
                }
            } else {
                Handler(Looper.getMainLooper()).post {
                    AlertDialog.Builder(this)
                        .setTitle(R.string.error_sqlite_other)
                        .setMessage(
                            this.getString(
                                R.string.error_sqlite_other_message,
                                e.message
                            )
                        )
                        .show()
                    alsoOnFailure?.let { it(e) }
                }
            }
        } catch (e: OfflineException) {
            Handler(Looper.getMainLooper()).post {
                alsoOnFailure?.let { it(e) }
                if (e.causeMessage != Resources.ID_NULL) AlertDialog.Builder(this)
                    .setMessage(e.causeMessage)
                    .setCancelable(true)
                    .show()
            }
        } catch (e: DirectoryCreationException) {
            Handler(Looper.getMainLooper()).post {
                AlertDialog.Builder(this)
                    .setTitle(R.string.error_directory_creation)
                    .setMessage(this.getString(R.string.error_directory_creation_message))
                    .show()
                alsoOnFailure?.let { it(e) }
            }
        } catch (e: FileCreationException) {
            Handler(Looper.getMainLooper()).post {
                AlertDialog.Builder(this)
                    .setTitle(R.string.error_file_creation)
                    .setMessage(this.getString(R.string.error_file_creation_message))
                    .show()
                alsoOnFailure?.let { it(e) }
            }
        }
    }

    @JvmStatic
    @StringRes
    fun IOException.networkErrorMessage(): Int = when (this) {
        is ConnectException -> R.string.error_network_connect
        is NoRouteToHostException -> R.string.error_network_no_route
        is UnknownHostException -> R.string.error_network_unknown_host
        is InterruptedIOException -> R.string.error_network_timeout
        is SSLException -> R.string.error_network_ssl
        else -> R.string.error_network
    }

    /**
     * Same as [.tryAndThen], but performs the
     * whole operation in a new thread.
     *
     * @return The thread that was created and started.
     */
    @JvmStatic
    @UiThread
    fun <T> tryAndThenThread(
        runnable: ExceptionableSupplier<T>,
        resultHandler: Consumer<T>,
        alsoOnFailure: Consumer<Exception?>?,
        context: Context
    ): Thread {
        val thread = Thread { runBlocking { tryAndThen(runnable, resultHandler, alsoOnFailure, context) } }
        thread.start()
        return thread
    }

    @UiThread
    fun <T> Context.tryAndThenThread(
        runnable: suspend () -> T,
        resultHandler: (T) -> Unit,
        alsoOnFailure: ((Exception) -> Unit)? = null,
    ): Thread {
        val thread = Thread { runBlocking { tryAndThen(runnable, resultHandler, alsoOnFailure) } }
        thread.start()
        return thread
    }

    /**
     * Same as [.tryAndThen], but performs the
     * whole operation in a new thread.
     *
     * @return The thread that was created and started.
     */
    @JvmStatic
    @UiThread
    fun <T> tryAndThenThread(
        runnable: ExceptionableSupplier<T>,
        resultHandler: Consumer<T>,
        context: Context
    ): Thread {
        val thread = Thread { runBlocking { tryAndThen(runnable, resultHandler, context) } }
        thread.start()
        return thread
    }

    fun interface ExceptionableSupplier<T> {
        @Throws(
            MoodleException::class,
            NotOkayException::class,
            JsonParseException::class,
            IOException::class,
            OfflineException::class
        )
        fun run(): T
    }
}
