package godau.fynn.moodledirect.module.link

import android.content.Context
import android.content.res.Resources
import android.net.Uri
import android.util.Log
import androidx.annotation.DrawableRes
import androidx.annotation.StringRes
import androidx.appcompat.app.AlertDialog
import androidx.browser.customtabs.CustomTabsIntent
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.data.persistence.PreferenceHelper
import godau.fynn.moodledirect.data.persistence.UserAccount
import godau.fynn.moodledirect.model.ResourceType
import godau.fynn.moodledirect.model.database.Course
import godau.fynn.moodledirect.model.database.Module
import godau.fynn.moodledirect.util.AutoLoginHelper

/**
 * Module links deal handle events that should open a given module; they instantiate
 * appropriate fragments or similar.
 */
sealed class ModuleLink {
    protected abstract fun onOpen(
        module: Module,
        course: Course,
        context: FragmentActivity,
        setStatus: (String?) -> Unit
    )

    /**
     * @param account Identifies the Moodle instance for which support should be checked.
     * @return False if this module cannot be used in the app with this Moodle instance.
     */
    fun isSupported(account: UserAccount): Boolean {
        val requiresCalls = requiresCalls() ?: return true
        val supportedCalls = account.supportedCalls ?: return true
        // Migration post update to the first version that supported this
        for (call in requiresCalls) {
            val callSupported = supportedCalls.contains(call)
            Log.d(
                "${ModuleLink::class.java.getSimpleName()} required calls",
                "${if (supportedCalls.contains(call)) '✔' else '❌'} $call"
            )
            if (!callSupported) return false
        }
        return true
    }

    /**
     * @return An array of moodle API calls required for this module to function. Module should not
     * be called if one of these functions is not available.
     */
    protected open fun requiresCalls(): Array<String>? {
        return null
    }

    @get:DrawableRes
    abstract val icon: Int

    @get:StringRes
    abstract val name: Int

    @get:StringRes
    open val requirementText: Int?
        get() = null

    protected fun showFragment(activity: FragmentActivity, fragment: Fragment?, tag: String?) {
        val fragmentTransaction = activity.supportFragmentManager.beginTransaction()
            .addToBackStack(null)
            .replace(R.id.course_activity_frame, fragment!!, tag)
        fragmentTransaction.commit()
    }

    companion object {
        private val LINKS: Map<ResourceType, ModuleLink> by lazy {
            mapOf(
                ResourceType.CHOICE to ChoiceLink,
                ResourceType.FOLDER to FolderLink,
                ResourceType.FORUM to ForumLink,
                ResourceType.PAGE to PageLink,
                ResourceType.URL to UrlLink,
                ResourceType.ZOOM to ZoomLink,
                ResourceType.ASSIGNMENT to AssignmentLink
            )

        }
        val links: Collection<ModuleLink> by lazy {
            LINKS.values
        }


        fun openWebsite(context: Context?, url: String?) {
            val intent = CustomTabsIntent.Builder()
                .build()
            intent.launchUrl(context!!, Uri.parse(url))
        }

        @JvmStatic
        fun open(module: Module, course: Course, context: FragmentActivity, setStatus: (String?) -> Unit) {
            val link = LINKS[module.getModuleType()]
            if (link != null && link.isSupported(PreferenceHelper(context).activeAccount)) {
                link.onOpen(module, course, context, setStatus)
            } else if (module.url != null) {
                AutoLoginHelper.openWithAutoLogin(context, setStatus, module.url)
            } else {
                AlertDialog.Builder(context, R.style.LoginTheme_AlertDialog)
                    .setMessage(R.string.error_not_openable)
                    .setCancelable(true).show()

            }
        }

        @JvmStatic
        @DrawableRes
        fun getIcon(module: Module): Int {
            val link = LINKS[module.getModuleType()]
            return link?.icon ?: Resources.ID_NULL
        }
    }
}
