package godau.fynn.moodledirect.data.persistence

import android.content.Context
import android.content.SharedPreferences
import androidx.core.content.edit
import godau.fynn.moodledirect.model.api.SiteInformation

/**
 * Load shared preferences centrally.
 * Preferences are set automatically by associated key through preference fragment.
 */
class PreferenceHelper(private val context: Context) {
    private val preferences: SharedPreferences =
        context.getSharedPreferences(SETTINGS_FILE, Context.MODE_PRIVATE)

    var userAccounts: List<UserAccount>
        get() = preferences.getStringSet(KEY_ACCOUNT_LIST, emptySet<String>())!!
            .map { UserAccount(context, it) }
        set(accounts) = preferences.edit {
            putStringSet(
                KEY_ACCOUNT_LIST,
                accounts.map { getPreferenceFileName(it) }.toSet()
            )
        }

    var activeAccount: UserAccount
        get() {
            val currentAccount = preferences.getString(KEY_CURRENT_ACCOUNT, null)
            return userAccounts.find { getPreferenceFileName(it) == currentAccount } ?: userAccounts.first().also {
                // active account was unset; pick any, but persist it because we are not guaranteed a stable order
                activeAccount = it
            }
        }
        set(active) {
            if (userAccounts.find { it == active } == null) {
                throw IllegalArgumentException("provided account does not exist, cannot switch.")
            }
            preferences.edit(commit = true) {
                putString(KEY_CURRENT_ACCOUNT, getPreferenceFileName(active))
            }
        }

    val isLoggedIn: Boolean
        /**
         * @return True if [.getUserAccount] may be called. Does not indicate whether token is valid.
         */
        get() = userAccounts.isNotEmpty()

    fun logout() {
        val current = activeAccount
        current.logout()
        userAccounts = userAccounts.minus(current)
        preferences.edit {
            // calling [.activeAccount] will now yield any other account, if more accounts are present
            remove(KEY_CURRENT_ACCOUNT)
        }
    }

    fun clearActiveAccount() {
        preferences.edit {
            remove(KEY_CURRENT_ACCOUNT)
        }
    }

    val courseRowAppearance: CourseRowAppearance
        get() = CourseRowAppearance(
            categoryAppearance = preferences.getString("appearance_category", "bottom"),
            showHeaderImage = preferences.getBoolean("appearance_header", true),
            showUnreadCounter = preferences.getBoolean("unread", true),
            showDescription = preferences.getBoolean("appearance_description", true)
    )

    var downloadPath: String?
        get() = preferences.getString("download_path", null)
        set(uri) {
            preferences.edit {
                putString("download_path", uri)
            }
        }
    val isNotificationsEnabled: Boolean
        /**
         * Do not use yet, as notifications have not yet been implemented
         */
        get() = preferences.getBoolean("notifications", false)

    val isForceOfflineModeEnabled: Boolean
        /**
         * @return Whether user has requested offline mode to always be enabled.
         */
        get() = preferences.getBoolean("offline", false)

    val isDarkThemeEnabled: Boolean
        get() = false
        // TODO get() = preferences.getBoolean("dark_theme", false)

    val isAutoLoginEnabled: Boolean
        get() = preferences.getBoolean("web_auto_login", true)

    var lastHelpMenuEntry: String
        get() = preferences.getString("help_entry", "") ?: ""
        set(entry) {
            preferences.edit { putString("help_entry", entry) }
        }

    /**
     * Adds account ONLY to list of accounts. It is the caller's responsibility to
     * also call [SiteInformation.saveAsUserAccount].
     */
    fun addAccount(account: UserAccount) {
        userAccounts += account
    }

    data class CourseRowAppearance(
        @JvmField val categoryAppearance: String? = "bottom",
        @JvmField val showHeaderImage: Boolean = true,
        @JvmField val showUnreadCounter: Boolean = false,
        @JvmField val showDescription: Boolean = true
    )

    companion object {
        const val SETTINGS_FILE = "godau.fynn.moodledirect:preferences"
        private const val KEY_CURRENT_ACCOUNT = "current_account"
        private const val KEY_ACCOUNT_LIST = "accounts"

        internal fun getPreferenceFileName(userAccount: UserAccount): String =
            getPreferenceFileName(userAccount.url!!, userAccount.userId)

        internal fun getPreferenceFileName(accountUrl: String, accountUserId: Int): String {
            val domain = accountUrl.replace("https?://".toRegex(), "")
                .replace("/".toRegex(), ".")
            return domain + ':' + accountUserId

        }
    }
}