package godau.fynn.moodledirect.activity.login;

import android.app.AlertDialog;
import android.app.ProgressDialog;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.util.Base64;
import android.util.Log;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentTransaction;

import godau.fynn.moodledirect.R;
import godau.fynn.moodledirect.activity.help.HelpActivity;
import godau.fynn.moodledirect.activity.MainActivity;
import godau.fynn.moodledirect.activity.login.fragment.ServerFragment;
import godau.fynn.moodledirect.data.persistence.PreferenceHelper;
import godau.fynn.moodledirect.data.persistence.UserAccount;
import godau.fynn.moodledirect.data.persistence.UserAccountKt;
import godau.fynn.moodledirect.model.HelpItem;
import godau.fynn.moodledirect.model.api.UserToken;
import godau.fynn.moodledirect.model.api.base.PublicConfig;
import godau.fynn.moodledirect.module.basic.Login;
import godau.fynn.moodledirect.util.ConfigDownloadHelper;
import godau.fynn.moodledirect.util.Constants;
import godau.fynn.moodledirect.util.ExceptionHandler;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;


public class LoginActivity extends FragmentActivity {

    private PublicConfig config;

    private TextView title;

    private boolean allowSkip;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setTheme(R.style.LoginTheme);

        setContentView(R.layout.activity_login);

        if (savedInstanceState != null && savedInstanceState.get("config") != null) {
            config = (PublicConfig) savedInstanceState.getSerializable("config");
        }

        title = findViewById(R.id.title);

        ImageView helpIcon = findViewById(R.id.help_icon);
        helpIcon.setOnClickListener((v) -> {
            Intent intent = new Intent(LoginActivity.this, HelpActivity.class);
            intent.putExtra(HelpActivity.EXTRA_HELP_ITEM, HelpItem.instance);
            startActivity(intent);
        });

        // Only add new fragments if not restoring old instance state
        if (savedInstanceState == null) {
            FragmentTransaction transaction = getSupportFragmentManager().beginTransaction();
            transaction.setReorderingAllowed(true)
                    .replace(R.id.fragment_container, new ServerFragment())
                    .commitNow();

            // Additionally handle extra intent data
            if (getIntent() != null && getIntent().getData() != null) onNewIntent(getIntent());
        } else {
            setTitle(savedInstanceState.getString("title"));
        }

        PreferenceHelper helper = new PreferenceHelper(this);
        List<UserAccount> accounts = helper.getUserAccounts();
        if (!accounts.isEmpty()) {
            View backIcon = findViewById(R.id.back_icon);
            backIcon.setVisibility(View.VISIBLE);
            backIcon.setOnClickListener((v) -> onBackPressed());
        }
    }

    public AlertDialog getAlertDialog(@StringRes int text) {
        return new AlertDialog.Builder(this, R.style.LoginTheme_AlertDialog)
                .setMessage(text)
                .create();
    }

    public void setPublicConfig(PublicConfig config) {
        this.config = config;
    }

    @Override
    protected void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putSerializable("config", config);
        outState.putString("title", String.valueOf(title.getText()));
    }

    /**
     * Here we need to handle web auth completing. This intent is expected to have one of the following URIs:
     * <ul>
     * <li><code>
     *     moodledirect://token=base64(md5(config.wwwroot + passport):::token[:::privatetoken])
     * </code><p>
     * with {@code passport} as a random number generated by the client, and {@code privatetoken} as an optional
     * parameter that is only sent if the user's authentication towards the web service was "fresh", i.e. the user
     * was not previously logged in.</li>
     *
     * <li><code>
     *     moodlemobile://https://moodle.example.com
     * </code></p></li>
     */
    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);

        Uri uri = intent.getData();

        if (uri == null) // This intent does not require further action
            return;

        String scheme = uri.getScheme();
        if (scheme.equals("moodlemobile") && uri.getHost().startsWith("token="))
            scheme = "moodledirect";

        switch (scheme) {
            case "moodledirect":
                if (config == null) {
                    getAlertDialog(R.string.login_error_not_started).show();
                    return;
                }

                String tokenString = uri.toString();
                Log.d("Intent-Response", tokenString);

                // Prepare String to just get base64
                tokenString = tokenString.replace(uri.getScheme() + "://token=", "");
                byte[] decodeValue = Base64.decode(tokenString, Base64.DEFAULT);
                tokenString = new String(decodeValue);
                String[] splitTokenString = tokenString.split(":::");

                if (splitTokenString.length < 2) {
                    getAlertDialog(R.string.login_invalid_uri).show();
                    return;
                }
                UserToken token = new UserToken();
                token.token = splitTokenString[1];

                if (splitTokenString.length >= 3) {
                    token.privatetoken = splitTokenString[2];
                }

                AtomicReference<Thread> thread = new AtomicReference<>();

                ProgressDialog progressDialog = new ProgressDialog(this, R.style.LoginTheme_AlertDialog);

                // Configure cancellable dialog
                progressDialog.setCanceledOnTouchOutside(false);
                progressDialog.setCancelable(true);
                progressDialog.setOnCancelListener(dialog -> {
                    thread.get().interrupt();
                });

                thread.set(advanceLogin(
                        new Login(this),
                        token,
                        progressDialog,
                        config,
                        null
                ));

                break;

            case "moodlemobile":

                if (uri.getQueryParameterNames().contains("qrlogin")) {
                    thread = new AtomicReference<>();

                    // Spawn progress dialog
                    ProgressDialog progressDialogQr = new ProgressDialog(this, R.style.LoginTheme_AlertDialog);
                    progressDialogQr.setCanceledOnTouchOutside(false);
                    progressDialogQr.setCancelable(true);
                    progressDialogQr.setOnCancelListener(dialog -> {
                        thread.get().interrupt();
                    });

                    progressDialogQr.setMessage(getString(R.string.login_progress_authenticating));
                    progressDialogQr.show();

                    // Log in with provided credentials
                    Login loginService = new Login(this);
                    thread.set(ExceptionHandler.tryAndThenThread(
                            () -> loginService.qrCodeAuth(uri),
                            success -> thread.set(advanceLogin(
                                    loginService, success.second, progressDialogQr, success.first, null
                            )),
                            failure -> {
                                progressDialogQr.dismiss();

                                // Copy server address only
                                Optional<ServerFragment> fragment = getSupportFragmentManager().getFragments().stream()
                                        .filter(instance -> instance instanceof ServerFragment)
                                        .map(instance -> (ServerFragment) instance)
                                        .findFirst();

                                fragment.ifPresent(
                                        serverFragment -> serverFragment.fillUrl(uri.toString().replaceAll("moodlemobile://", ""))
                                );
                            },
                            this
                    ));

                } else {
                    // Copy server address only
                    Optional<ServerFragment> fragment = getSupportFragmentManager().getFragments().stream()
                            .filter(instance -> instance instanceof ServerFragment)
                            .map(instance -> (ServerFragment) instance)
                            .findFirst();

                    if (fragment.isPresent()) {
                        fragment.get().fillUrl(uri.toString().replaceAll("moodlemobile://", ""));
                    } else {
                        // User had already advanced to a later step, unclear how they want this new data to be treated
                        getAlertDialog(R.string.login_data_url_ignored).show();
                    }
                }
        }
    }

    /**
     * Queries user details and finishes.
     */
    public Thread advanceLogin(Login loginService, UserToken token, ProgressDialog progressDialog,
                               PublicConfig config, @Nullable Consumer<Exception> onFailure) {

        progressDialog.setMessage(getString(R.string.login_progress_fetch_detail));

        return ExceptionHandler.tryAndThenThread(
                () -> loginService.getUserDetail(config, token),
                details -> {
                    details.token = token;
                    UserAccount account = UserAccountKt.saveAsUserAccount(details, this);
                    account.setMoodleInstanceName(config.sitename);
                    PreferenceHelper preferences = new PreferenceHelper(this);
                    preferences.addAccount(account);
                    preferences.setActiveAccount(account);
                    Constants.API_URL = details.getSiteUrl();

                    progressDialog.dismiss();

                    if (account.hasPrivateToken()) {
                        ConfigDownloadHelper.updateAutoLoginCooldown(this);
                    }

                    Intent intent = new Intent(this, MainActivity.class);
                    intent.putExtra(MainActivity.INTENT_FLAG_RECREATE_UPON_NEW_INTENT, true);
                    // wow this flag is so cool! it does exactly what we want!!
                    // note: MainActivity's `launchMode` is set to `singleTop`
                    intent.addFlags(Intent.FLAG_ACTIVITY_PREVIOUS_IS_TOP);
                    if (details.getSiteUrl().endsWith("webservice/") || details.getSiteUrl().contains("webservice/pluginfile.php")) {
                        new AlertDialog.Builder(this, R.style.LoginTheme_AlertDialog)
                                .setTitle(R.string.login_confusing_root)
                                .setMessage(R.string.login_confusing_root_details)
                                .setCancelable(false)
                                .setPositiveButton(R.string._continue, (dialog, which) -> {
                                    startActivity(intent);
                                    finish();
                                })
                                .show();
                    } else {
                        startActivity(intent);
                        finish();
                    }
                },
                onFailure,
                this
        );
    }

    @Override
    public void setTitle(CharSequence title) {
        super.setTitle(title);
        this.title.setText(title);
    }
}
