package godau.fynn.moodledirect.activity.fragment

import android.app.Activity
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.annotation.CallSuper
import androidx.fragment.app.Fragment
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.data.persistence.MoodleDatabase.Dispatch
import godau.fynn.moodledirect.network.NetworkStateReceiver
import godau.fynn.moodledirect.util.MyApplication
import godau.fynn.moodledirect.view.NoDataView
import godau.fynn.moodledirect.view.NoDataView.InvokeListener

/**
 * Fragment that provides two features for descendant classes:
 *
 *  * `SwipeRefreshLayout` container
 *  * [NoDataView] in the middle
 *
 */
abstract class SwipeRefreshFragment : Fragment() {
    @JvmField
    protected var swipeRefreshLayout: SwipeRefreshLayout? = null
    @JvmField
    protected var empty: NoDataView? = null

    // Variables are kept when restored from back stack
    private var loaded = false
    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        val frameLayout =
            inflater.inflate(R.layout.fragment_swipe_refresh, container, false) as FrameLayout
        // Do this while we are certain that the IDs are unique (also we need to inflate into the swipeRefreshLayout)
        empty = frameLayout.findViewById(R.id.empty)
        swipeRefreshLayout = frameLayout.findViewById<SwipeRefreshLayout?>(R.id.swipeRefreshLayout).also {
            it.addView(onCreateContentView(inflater, container, savedInstanceState))
        }
        return frameLayout
    }

    protected abstract fun onCreateContentView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View?

    @CallSuper
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        swipeRefreshLayout!!.setOnRefreshListener { loadData(MyApplication.moodle().dispatch) }

        // In offline mode, queries are so quick that the indicator would flicker. Don't show it in the first place.
        if (!loaded && !NetworkStateReceiver.getOfflineStatus()) swipeRefreshLayout!!.isRefreshing = true
        if (!loaded and (savedInstanceState == null)) {
            loadData(MyApplication.moodle().dispatch)
            loaded = true
        } else if (!loaded) {
            // Force offline dispatch if restoring instance state (avoid lag when navigating back)
            loadData(MyApplication.moodle().forceOffline())
            loaded = true
        }
        val activity: Activity = requireActivity()
        if (activity is InvokeListener) {
            empty!!.setOnInvokeListener(activity as InvokeListener)
        }
    }

    /**
     * Called after the fragment is instantiated or if pull-to-refresh was triggered.
     * At this moment, the `SwipeRefreshLayout` is already set to refreshing.
     *
     * @param dispatch Dispatch that should be used to download data
     */
    protected abstract fun loadData(dispatch: Dispatch)
}
