package godau.fynn.moodledirect.activity.fragment

import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.KeyEvent
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.inputmethod.EditorInfo
import android.view.inputmethod.InputMethodManager
import android.widget.EditText
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.material.Divider
import androidx.compose.material.ExperimentalMaterialApi
import androidx.compose.material.pullrefresh.PullRefreshIndicator
import androidx.compose.material.pullrefresh.pullRefresh
import androidx.compose.material.pullrefresh.rememberPullRefreshState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.ComposeView
import androidx.fragment.app.Fragment
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.activity.CourseDetailActivity
import godau.fynn.moodledirect.data.persistence.PreferenceHelper
import godau.fynn.moodledirect.data.persistence.PreferenceHelper.CourseRowAppearance
import godau.fynn.moodledirect.model.database.Course
import godau.fynn.moodledirect.util.Constants
import godau.fynn.moodledirect.util.ExceptionHandler.tryAndThenThread
import godau.fynn.moodledirect.util.MyApplication
import godau.fynn.moodledirect.view.CourseRow
import godau.fynn.moodledirect.view.ErrorState
import godau.fynn.moodledirect.view.ExceptionOccurred
import godau.fynn.moodledirect.view.NoDataView
import godau.fynn.moodledirect.view.NoResults
import godau.fynn.moodledirect.view.Token

class SearchCourseFragment : Fragment() {
    private lateinit var editText: EditText
    private lateinit var searchButton: View

    private var refreshing by mutableStateOf(false)
    private var errorState: ErrorState? by mutableStateOf(null)

    private val courses: MutableList<Course> = mutableStateListOf()

    private var previousSearch = ""
    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        // Inflate the layout for this fragment
        return inflater.inflate(R.layout.fragment_search_course, container, false)
    }

    @OptIn(ExperimentalMaterialApi::class)
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        editText = view.findViewById(R.id.course_search_edit_text)
        searchButton = view.findViewById(R.id.course_search_button)

        val composeLayout = view.findViewById<ComposeView>(R.id.compose)

        val courseRowAppearance = CourseRowAppearance(
            showDescription = true,
            showHeaderImage = true,
            categoryAppearance = "bottom"
        )

        val preferences = PreferenceHelper(requireContext())
        val token: Token = preferences.activeAccount.token!!

        composeLayout.setContent {
            val pullRefreshState = rememberPullRefreshState(
                refreshing = refreshing,
                onRefresh = {
                    refreshing = true
                    getSearchCourses(previousSearch)
                }
            )

            Box(
                modifier = Modifier
                    .fillMaxSize()
                    .pullRefresh(pullRefreshState)
            ) {
                LazyColumn(Modifier.fillMaxSize()) {
                    itemsIndexed(courses) { index, course ->
                        CourseRow(course, token, courseRowAppearance) {
                            Intent(activity, CourseDetailActivity::class.java).apply {
                                putExtra(Constants.EXTRA_COURSE, course)
                            }.let { startActivity(it) }
                        }
                        if (index < courses.size - 1) Divider()
                    }
                }

                errorState?.let { NoDataView(error = it) }
                PullRefreshIndicator(refreshing, pullRefreshState, Modifier.align(Alignment.TopCenter))
            }
        }

        editText.apply {
            requestFocus()
            (requireContext().getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager)
                .showSoftInput(editText, InputMethodManager.SHOW_IMPLICIT)

            setOnEditorActionListener { _, actionId: Int, _ ->
                if (actionId == EditorInfo.IME_ACTION_SEARCH) {
                    searchButton.callOnClick()
                    true
                } else false
            }

            setOnKeyListener { _, actionId: Int, keyEvent: KeyEvent ->
                if (keyEvent.action == KeyEvent.ACTION_DOWN && actionId == KeyEvent.KEYCODE_ENTER) {
                    searchButton.callOnClick()
                    true
                } else false
            }
        }
        searchButton.setOnClickListener { _ ->
            val searchText = editText.getText().toString().trim { it <= ' ' }
            refreshing = true
            previousSearch = searchText

            getSearchCourses(searchText)

            (requireContext().getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager)
                .hideSoftInputFromWindow(editText.windowToken, 0)
        }
    }

    override fun onViewStateRestored(savedInstanceState: Bundle?) {
        super.onViewStateRestored(savedInstanceState)
        searchButton.callOnClick()
    }

    private fun getSearchCourses(query: String) {
        errorState = null
        val core = MyApplication.moodle().dispatch.getCore()
        requireContext().tryAndThenThread(
            { core.searchCourses(query) },
            { courses: List<Course> ->
                if (courses.isEmpty()) {
                    this.courses.clear()
                    errorState = NoResults(R.string.empty_search)
                } else {
                    this.courses.clear()
                    this.courses.addAll(courses)
                }
                refreshing = false
            },
            { failure: Exception ->
                refreshing = false
                errorState = ExceptionOccurred(failure)
            }
        )
    }

    fun forceReload() {
        getSearchCourses(previousSearch)
    }
}
