package godau.fynn.moodledirect.activity.fragment

import android.content.Intent
import android.os.Bundle
import android.view.Menu
import android.view.MenuInflater
import androidx.appcompat.widget.SearchView
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.material.Divider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.activity.CourseDetailActivity
import godau.fynn.moodledirect.data.persistence.MoodleDatabase.Dispatch
import godau.fynn.moodledirect.data.persistence.PreferenceHelper
import godau.fynn.moodledirect.model.database.Course
import godau.fynn.moodledirect.util.ConfigDownloadHelper
import godau.fynn.moodledirect.util.ExceptionHandler.tryAndThenThread
import godau.fynn.moodledirect.view.CourseRow
import godau.fynn.moodledirect.view.ExceptionOccurred
import godau.fynn.moodledirect.view.NoResults
import godau.fynn.moodledirect.view.NotAvailableOffline
import godau.fynn.moodledirect.view.Token
import java.util.Locale

class EnrolledCoursesFragment : ComposeSwipeRefreshFragment() {
    private val courses: MutableList<Course> = mutableStateListOf()
    private var searchText by mutableStateOf("")

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setHasOptionsMenu(true)
    }

    @Composable
    override fun ContentView() {
        val preferences = PreferenceHelper(requireContext())
        val token: Token = preferences.activeAccount.token!!
        val appearance = preferences.courseRowAppearance
        LazyColumn(Modifier.fillMaxSize()) {
            courses.filter { course ->
                listOfNotNull(
                    course.name, course.shortname, course.topCategoryName,
                    course.categoryName, course.summary
                ).any {
                    it.lowercase(Locale.getDefault()).contains(searchText)
                }
            }.let { courses ->
                itemsIndexed(courses) { index, course ->
                    CourseRow(course, token, appearance) {
                        Intent(activity, CourseDetailActivity::class.java).apply {
                            putExtra("id", course.id)
                            putExtra("course_name", course.shortname)
                        }.let { startActivity(it) }
                    }
                    if (index < courses.size - 1) Divider()
                }
            }
        }
    }

    override fun loadData(dispatch: Dispatch) {
        val coreModule = dispatch.getCore()
        requireContext().tryAndThenThread(
            { coreModule.getCourses() },
            { courses: List<Course> ->
                this.courses.clear()
                this.courses.addAll(courses)

                errorState = if (courses.isEmpty()) {
                    NoResults(R.string.empty_courses)
                } else {
                    null
                }
                refreshing = false
            },
            { e: Exception ->
                refreshing = false
                errorState = ExceptionOccurred(e)
                courses.clear()
            }
        )

        // Also update mobile config
        ConfigDownloadHelper.updateSiteInformation(dispatch, requireContext())
    }

    override fun onCreateOptionsMenu(menu: Menu, inflater: MenuInflater) {
        inflater.inflate(R.menu.my_courses_menu, menu)
        val searchView = menu.findItem(R.id.action_search).actionView as SearchView?
        searchView!!.setOnQueryTextListener(object : SearchView.OnQueryTextListener {
            override fun onQueryTextSubmit(query: String): Boolean {
                return false
            }

            override fun onQueryTextChange(newText: String): Boolean {
                searchText = newText
                return true
            }
        })
        super.onCreateOptionsMenu(menu, inflater)
    }

}
