package godau.fynn.moodledirect.activity.fragment

import android.os.Bundle
import android.view.Menu
import android.view.MenuInflater
import android.view.MenuItem
import android.view.View
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateMapOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.activity.CourseDetailActivity
import godau.fynn.moodledirect.data.persistence.MoodleDatabase.Dispatch
import godau.fynn.moodledirect.model.ResourceType
import godau.fynn.moodledirect.model.api.file.File
import godau.fynn.moodledirect.model.database.Course
import godau.fynn.moodledirect.model.database.Module
import godau.fynn.moodledirect.module.FileManager
import godau.fynn.moodledirect.module.link.ModuleLink
import godau.fynn.moodledirect.util.ActionItem
import godau.fynn.moodledirect.util.ActionItemSupplier
import godau.fynn.moodledirect.util.AutoLoginHelper
import godau.fynn.moodledirect.util.Constants
import godau.fynn.moodledirect.util.ExceptionHandler.tryAndThenThread
import godau.fynn.moodledirect.util.FileManagerWrapper
import godau.fynn.moodledirect.util.MyApplication
import godau.fynn.moodledirect.view.ExceptionOccurred
import godau.fynn.moodledirect.view.ModuleList
import godau.fynn.moodledirect.view.NoResults

class CourseFragment : ComposeSwipeRefreshFragment(), ActionItemSupplier {
    private var courseId = 0
    private var guest = false

    private val modules: MutableList<Module> = mutableStateListOf()
    private var course: Course? by mutableStateOf(null)
    private var fileStates: MutableMap<File, FileManager.DownloadStatus> = mutableStateMapOf()

    private lateinit var fileManager: FileManagerWrapper

    override val actionItems: List<ActionItem>
        get() = listOf(
            ActionItem(
                R.string.open_website_view,
                R.drawable.web
            ) {
                AutoLoginHelper.openWithAutoLogin(
                    requireActivity(),
                    {},
                    Constants.getCourseUrl(courseId)
                )

            }
        ) + (
                if (requireArguments().getBoolean(KEY_GUEST)) emptyList()
                else
                    listOf(ActionItem(
                        R.string.course_information,
                        R.drawable.ic_information
                    ) {
                        getParentFragmentManager().beginTransaction()
                            .setReorderingAllowed(true)
                            .addToBackStack(null)
                            .replace(
                                R.id.course_activity_frame,
                                CourseInformationFragment.newInstance(courseId),
                                null
                            )
                            .commit()

                    })
                )

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val args = arguments
        fileManager = MyApplication.moodle().dispatch.getFile().wrap()

        if (args != null) {
            course = args.getSerializable(KEY_COURSE) as Course?
            guest = args.getBoolean(KEY_GUEST)
            courseId = course!!.id
        } else throw IllegalArgumentException()

        setHasOptionsMenu(true)
    }

    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    override fun ContentView() = ModuleList(
        modules = modules,
        fileStates = fileStates,
        modifier = (requireActivity() as CourseDetailActivity?)?.scrollBehavior?.nestedScrollConnection?.let {
            Modifier.nestedScroll(it, (requireActivity() as CourseDetailActivity).scrollDispatcher)
        } ?: Modifier
    ) {
        if (it.isDownloadable && it.getModuleType() == ResourceType.FILE) {
            val file = it.fileList[0]

            if (fileStates[file] != FileManager.DownloadStatus.DOWNLOADED) {
                fileManager.startDownload(
                    file,
                    course!!.shortname,
                    null,
                    requireContext(),
                    { state -> fileStates[file] = state }
                )
            } else {
                fileManager.openFile(file = file, doesNotExist = {
                    fileStates[file] = FileManager.DownloadStatus.NOT_DOWNLOADED
                }, context = requireContext())
            }
        } else {
            ModuleLink.open(
                it, course!!,
                requireActivity()
            ) {
                // TODO: display status changes for autologin
            }
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        // Enable smooth expanding text transitions
//        (recyclerView.itemAnimator as SimpleItemAnimator?)!!.supportsChangeAnimations = false

    }

    override fun loadData(dispatch: Dispatch) {
        requireContext().tryAndThenThread(
            {
                dispatch.getCourseContent().getCourseModules(courseId)
            },
            { modules: List<Module> ->
                this.modules.clear()
                this.modules.addAll(modules)

                modules.flatMap { it.fileList }.associateWith { it.downloadStatus }
                    .let {
                        fileStates.putAll(it)
                    }

                refreshing = false
                errorState = if (modules.isEmpty()) {
                    NoResults(R.string.empty_course)
                } else {
                    null
                }
            },
            { failure: Exception ->
                this.modules.clear()
                refreshing = false
                errorState = ExceptionOccurred(failure)
            }
        )
    }

    companion object {
        private const val KEY_COURSE = "course"
        private const val KEY_GUEST = "guest"

        @JvmStatic
        @JvmOverloads
        fun newInstance(course: Course?, guest: Boolean = false): CourseFragment =
            CourseFragment().apply {
                arguments = Bundle().apply {
                    putSerializable(KEY_COURSE, course)
                    putBoolean(KEY_GUEST, guest)
                }
            }
    }
}