package godau.fynn.moodledirect.activity.fragment

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.annotation.CallSuper
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material.ExperimentalMaterialApi
import androidx.compose.material.pullrefresh.PullRefreshIndicator
import androidx.compose.material.pullrefresh.pullRefresh
import androidx.compose.material.pullrefresh.rememberPullRefreshState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.ComposeView
import androidx.fragment.app.Fragment
import godau.fynn.moodledirect.data.persistence.MoodleDatabase.Dispatch
import godau.fynn.moodledirect.network.NetworkStateReceiver
import godau.fynn.moodledirect.util.MyApplication
import godau.fynn.moodledirect.view.ErrorState
import godau.fynn.moodledirect.view.NoDataView

/**
 * Fragment that provides two features for descendant classes:
 *
 *  * `pullRefresh` container
 *  * `errorState` is displayed in the middle
 *
 */
abstract class ComposeSwipeRefreshFragment : Fragment() {

    protected var refreshing by mutableStateOf(false)
    protected var errorState: ErrorState? by mutableStateOf(null)

    // Variables are kept when restored from back stack
    private var loaded = false

    @OptIn(ExperimentalMaterialApi::class)
    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? = ComposeView(requireContext()).apply {
        setContent {
            val pullRefreshState = rememberPullRefreshState(
                refreshing = refreshing,
                onRefresh = {
                    refreshing = true
                    loadData(MyApplication.moodle().dispatch)
                }
            )

            Box(
                modifier = Modifier
                    .fillMaxSize()
                    .pullRefresh(pullRefreshState)
            ) {
                ContentView()

                errorState?.let { NoDataView(error = it) }

                PullRefreshIndicator(refreshing, pullRefreshState, Modifier.align(Alignment.TopCenter))
            }
        }
    }

    @Composable
    protected abstract fun ContentView()

    @CallSuper
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        loadData(savedInstanceState != null)
    }

    fun forceReload() {
        loaded = false
        loadData(false)
    }

    /**
     * @param isRecreating: Indicates whether there is an existing `savedInstanceState` from
     * which the fragment can recreate.
     */
    private fun loadData(isRecreating: Boolean) {
        // In offline mode, queries are so quick that the indicator would flicker. Don't show it in the first place.
        if (!loaded && !NetworkStateReceiver.getOfflineStatus()) refreshing = true
        if (!loaded and !isRecreating) {
            loadData(MyApplication.moodle().dispatch)
            loaded = true
        } else if (!loaded) {
            // Force offline dispatch if restoring instance state (avoid lag when navigating back)
            loadData(MyApplication.moodle().forceOffline())
            loaded = true
        }
        /*Legacy code TODO review and remove
        val activity: Activity = requireActivity()
        if (activity is InvokeListener) {
            empty!!.setOnInvokeListener(activity as InvokeListener)
        }*/
    }

    /**
     * Called after the fragment is instantiated or if pull-to-refresh was triggered.
     * At this moment, `refreshing` is already set to `true`.
     *
     * @param dispatch Dispatch that should be used to download data
     */
    protected abstract fun loadData(dispatch: Dispatch)
}


