package godau.fynn.moodledirect.activity;

import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.Toolbar;
import androidx.drawerlayout.widget.DrawerLayout;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentTransaction;
import androidx.transition.AutoTransition;
import androidx.transition.ChangeBounds;
import androidx.transition.TransitionManager;

import com.mikepenz.materialdrawer.AccountHeader;
import com.mikepenz.materialdrawer.AccountHeaderBuilder;
import com.mikepenz.materialdrawer.Drawer;
import com.mikepenz.materialdrawer.DrawerBuilder;
import com.mikepenz.materialdrawer.model.DividerDrawerItem;
import com.mikepenz.materialdrawer.model.PrimaryDrawerItem;
import com.mikepenz.materialdrawer.model.ProfileDrawerItem;
import com.mikepenz.materialdrawer.model.interfaces.IProfile;

import godau.fynn.moodledirect.BuildConfig;
import godau.fynn.moodledirect.R;
import godau.fynn.moodledirect.activity.fragment.*;
import godau.fynn.moodledirect.activity.help.HelpActivity;
import godau.fynn.moodledirect.activity.login.LoginActivity;
import godau.fynn.moodledirect.data.persistence.PreferenceHelper;
import godau.fynn.moodledirect.data.persistence.UserAccount;
import godau.fynn.moodledirect.model.database.Course;
import godau.fynn.moodledirect.network.APIClient;
import godau.fynn.moodledirect.network.NetworkStateReceiver;
import godau.fynn.moodledirect.util.Constants;
import godau.fynn.moodledirect.util.MyApplication;
import godau.fynn.moodledirect.util.UserUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

public class MainActivity extends AppCompatActivity
        implements NetworkStateReceiver.OfflineStatusChangeListener {

    private UserAccount userAccount;

    private TextView offlineIndicator;
    private ViewGroup root;

    private Drawer sidebarDrawer;

    private static final long PROFILE_ID_ADD = 0;

    public static final String INTENT_FLAG_RECREATE_UPON_NEW_INTENT = "friendlyFire";
    private static final String INSTANCE_STATE_FLAG_FORCE_ONLINE_LOAD = "recreateOnline";
    private boolean recreateOnline = false;

    private static final int ID_MY_COURSES = 0;
    private static final int ID_COURSE_SEARCH = 1;
    private static final int ID_CALENDAR = 2;
    private static final int ID_SETTINGS = 3;
    private static final int ID_HELP = 4;
    private static final int ID_LOGOUT = 5;
    private static final int ID_DEBUG = 6;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        if (MyApplication.getInstance().isDarkModeEnabled()) {
            setTheme(R.style.AppTheme_NoActionBar_Dark_TransparentStatus);
        }

        getWindow().getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_STABLE
          | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN);
        super.onCreate(savedInstanceState);

        PreferenceHelper preferences = new PreferenceHelper(this);
        if (!preferences.isLoggedIn()) {
            Intent intent = new Intent(this, LoginActivity.class);
            startActivity(intent);

            finish();
            return;
        }

        setContentView(R.layout.activity_main);

        Toolbar toolbar = findViewById(R.id.toolbar);
        setSupportActionBar(toolbar);

        userAccount = preferences.getActiveAccount();
        Constants.TOKEN = userAccount.getToken();
        Constants.API_URL = userAccount.getUrl();

        Function<UserAccount, IProfile> toProfile = (u) ->
                (IProfile) new ProfileDrawerItem().withName(u.getUserFirstName())
                        .withEmail(u.getDisplayIdentifier())
                        .withIcon(UserUtils.resolveUrl(u.getAvatarUrl(), u))
                        .withTag(u);

        AccountHeader accountHeader = new AccountHeaderBuilder()
                .withActivity(this)
                .withHeaderBackground(R.drawable.navbar_header_light)
                .addProfiles(toProfile.apply(userAccount))
                .addProfiles(
                        preferences.getUserAccounts().stream()
                                .filter((u) -> !u.equals(userAccount))
                                .map(toProfile)
                                .toArray((i) -> new IProfile[i])
                )
                .addProfiles(
                        new ProfileDrawerItem().withSelectable(false)
                                .withName(R.string.login_add)
                                .withIcon(R.drawable.ic_add_with_background)
                                .withIdentifier(PROFILE_ID_ADD)
                )
                .withOnProfileClickDrawerCloseDelay(0)
                .withOnAccountHeaderListener((view, profile, current) -> {

                    APIClient.clearMoodleInstance();

                    if (profile.getIdentifier() == PROFILE_ID_ADD) {
                        Intent intent = new Intent(MainActivity.this, LoginActivity.class);
                        MyApplication.clearMoodle();
                        Constants.API_URL = null;
                        Constants.TOKEN = null;
                        startActivity(intent);
                    } else if (profile instanceof ProfileDrawerItem item) {
                        UserAccount newAccount = ((UserAccount) item.getTag());
                        preferences.setActiveAccount(newAccount);
                        APIClient.clearMoodleInstance();
                        MyApplication.switchMoodleToAccount(newAccount);
                        Constants.API_URL = newAccount.getUrl();
                        Constants.TOKEN = newAccount.getToken();
                        setTitle(newAccount.getMoodleInstanceName());

                        // Reload if necessary
                        forceChildFragmentReload();
                    }
                    return false;
                })
                .build();

        DrawerBuilder drawerBuilder = new DrawerBuilder()
                .withActivity(this)
                .withSavedInstance(savedInstanceState)
                .withToolbar(toolbar)
                .withDelayOnDrawerClose(0)
                .withAccountHeader(accountHeader)
                .addDrawerItems(
                        new PrimaryDrawerItem().withName(R.string.my_courses).withIcon(R.drawable.ic_collections_bookmark)
                                .withIdentifier(ID_MY_COURSES).withIconTintingEnabled(true),
                        new PrimaryDrawerItem().withName(R.string.course_search_navigation_drawer).withIcon(R.drawable.ic_search)
                                .withIdentifier(ID_COURSE_SEARCH).withIconTintingEnabled(true),
                        new PrimaryDrawerItem().withName(R.string.calendar_title).withIcon(R.drawable.ic_calendar)
                                .withIdentifier(ID_CALENDAR).withIconTintingEnabled(true),
                        new PrimaryDrawerItem().withName(R.string.settings_title).withIcon(R.drawable.ic_settings)
                                .withIdentifier(ID_SETTINGS).withIconTintingEnabled(true)
                ).addStickyDrawerItems(
                        new PrimaryDrawerItem().withName(R.string.help).withIcon(R.drawable.ic_help)
                                .withIdentifier(ID_HELP).withIconTintingEnabled(true).withSelectable(false),
                        new PrimaryDrawerItem().withName(R.string.logout).withIcon(R.drawable.ic_logout)
                                .withIdentifier(ID_LOGOUT).withIconTintingEnabled(true).withSelectable(false)
                );

        if (BuildConfig.DEBUG) {
            drawerBuilder.addDrawerItems(
                    new DividerDrawerItem(),
                    new PrimaryDrawerItem().withName(R.string._debug).withIcon(R.drawable.ic_bug)
                            .withIdentifier(ID_DEBUG).withIconTintingEnabled(true)
            );
        }

        drawerBuilder.withOnDrawerItemClickListener((view, position, drawerItem) -> {
            if (view == null) {
                // No actual click happened – this selection was triggered programmatically to update drawer state
                return false;
            }
            switch ((int) drawerItem.getIdentifier()) {
                case ID_MY_COURSES:
                    pushView(new EnrolledCoursesFragment(), getString(R.string.my_courses), true);
                    return false;

                case ID_COURSE_SEARCH:
                    pushView(new SearchCourseFragment(), getString(R.string.browse_courses), false);
                    return false;

                case ID_CALENDAR:
                    pushView(new CalendarFragment(), getString(R.string.calendar_title), false);
                    return false;

                case ID_SETTINGS:
                    pushView(new SettingsFragment(), getString(R.string.settings_title), false);
                    return false;

                case ID_HELP:
                    Intent helpIntent = new Intent(this, HelpActivity.class);
                    startActivity(helpIntent);
                    return false;

                case ID_LOGOUT:
                    askToLogout().show();
                    return false;

                case ID_DEBUG:
                    pushView(new DebugFragment(), "Debug", false);
                    return false;

            }
            return false;
        });
        offlineIndicator = findViewById(R.id.offlineIndicator);
        root = findViewById(R.id.drawer_layout);
        if (root == null) { // for width > 600dp
            root = findViewById(R.id.root);

            root.setOnApplyWindowInsetsListener(((v, insets) -> {
                findViewById(R.id.app_bar_app_layout).setPadding(
                  0, insets.getSystemWindowInsetTop(), 0, 0
                );
                return insets;
            }));

            sidebarDrawer = drawerBuilder.buildView();
            ((ViewGroup) findViewById(R.id.nav_frame)).addView(sidebarDrawer.getSlider());
        }

        setTitle(userAccount.getMoodleInstanceName());

        if (root instanceof DrawerLayout) {
            sidebarDrawer = drawerBuilder.build();
        }

        // Set up fragments
        if (savedInstanceState == null) {
            pushView(new EnrolledCoursesFragment(), "My Courses", true);
        } else if (savedInstanceState.getBoolean(INSTANCE_STATE_FLAG_FORCE_ONLINE_LOAD, false)) {
            forceChildFragmentReload();
            recreateOnline = false;
            updateSidebarState();
        } else {
            updateSidebarState();
        }

        getSupportFragmentManager().addOnBackStackChangedListener(this::updateSidebarState);

        resolveModuleLinkShare();
        getWindow().setSoftInputMode(
                WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN);

        ((MyApplication) getApplication()).networkStateReceiver.addListener(this);
    }

    private void updateSidebarState() {
        Fragment frag = getSupportFragmentManager().findFragmentById(R.id.content_main);
        if (frag instanceof EnrolledCoursesFragment) {
            sidebarDrawer.setSelection(ID_MY_COURSES);
        } else if (frag instanceof SearchCourseFragment) {
            sidebarDrawer.setSelection(ID_COURSE_SEARCH);
        } else if (frag instanceof CalendarFragment) {
            sidebarDrawer.setSelection(ID_CALENDAR);
        } else if (frag instanceof SettingsFragment) {
            sidebarDrawer.setSelection(ID_SETTINGS);
        } else {
            sidebarDrawer.setSelection(ID_DEBUG);
        }
    }

    @Override
    public void onNewOfflineStatus(boolean offline) {
        if (offline) {
            TransitionManager.beginDelayedTransition(root, new ChangeBounds());
            offlineIndicator.setVisibility(View.VISIBLE);
        } else {
            TransitionManager.beginDelayedTransition(root, new AutoTransition());
            offlineIndicator.setVisibility(View.GONE);
        }
    }

    @Deprecated // currently not supported
    private void resolveModuleLinkShare() {
        Intent intent = getIntent();
        String action = intent.getAction();
        Uri uri = intent.getData();
        if(uri != null && action != null && action.equals("android.intent.action.VIEW")) {
            List<Course> courses = new ArrayList<>();//realm.copyFromRealm(realm.where(Course.class).findAll());
            int courseId = Integer.parseInt(uri.getQueryParameter("courseId"));

            boolean isEnrolled = false;
            for(Course course : courses) {
                if(course.id == courseId) {
                    isEnrolled = true;
                    break;
                }
            }

            if(isEnrolled) {
                String fileUrl = uri.getScheme() + "://" + uri.getHost() + uri.getPath().replace("/fileShare", "") + "?forcedownload=1&token=" + userAccount.getToken();
                Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(fileUrl));
                startActivity(browserIntent);
            }
            else {
                Toast.makeText(this, "You need to be enrolled in " + uri.getQueryParameter("courseName") + " in order to view", Toast.LENGTH_LONG).show();
            }

        }
    }

    @Override
    protected void onNewIntent(@NonNull Intent intent) {
        super.onNewIntent(intent);
        if (intent.getBooleanExtra(INTENT_FLAG_RECREATE_UPON_NEW_INTENT, false)) {
            recreateOnline = true;
            recreate();
        }
    }

    @Override
    protected void onSaveInstanceState(@NonNull Bundle outState) {
        outState.putBoolean(INSTANCE_STATE_FLAG_FORCE_ONLINE_LOAD, recreateOnline);
        super.onSaveInstanceState(outState);
    }

    private void pushView(Fragment fragment, String tag, boolean rootFrag){
        if (rootFrag) {
            clearBackStack();
        }
        FragmentTransaction transaction = getSupportFragmentManager().beginTransaction();
        transaction.replace(R.id.content_main, fragment, tag);
        if (!rootFrag) {
            transaction.addToBackStack(null);
        }
        transaction.commit();
    }

    private void forceChildFragmentReload() {
        Fragment fragment = getSupportFragmentManager().findFragmentById(R.id.content_main);
        if (fragment instanceof ComposeSwipeRefreshFragment refreshableFragment) {
            refreshableFragment.forceReload();
        }
        if (fragment instanceof SearchCourseFragment searchFragment) {
            searchFragment.forceReload();
        }
    }

    private AlertDialog askToLogout() {
        AlertDialog.Builder alertDialog;

        alertDialog = new AlertDialog.Builder(MainActivity.this, R.style.LoginTheme_AlertDialog);

        alertDialog.setMessage(R.string.confirm_logout);
        alertDialog.setPositiveButton(R.string._yes, (dialogInterface, i) -> logout());
        alertDialog.setNegativeButton(R.string._cancel, null);
        return alertDialog.create();
    }

    public void logout() {
        UserUtils.logout(this);
        PreferenceHelper preferences = new PreferenceHelper(this);
        preferences.clearActiveAccount();
        APIClient.clearMoodleInstance();
        MyApplication.clearMoodle();
        Constants.API_URL = "";
        Constants.TOKEN = null;

        if (preferences.isLoggedIn()) {
            recreateOnline = true;
            recreate();
        } else {
            finish();
            startActivity(new Intent(this, LoginActivity.class));
        }
    }

    @Override
    public void onBackPressed() {
        if (sidebarDrawer.isDrawerOpen()) {
            sidebarDrawer.closeDrawer();
        } else {
            super.onBackPressed();
        }
    }

    private void clearBackStack() {
        for (int i = 0; i < getSupportFragmentManager().getBackStackEntryCount(); ++i) {
            getSupportFragmentManager().popBackStack();
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        ((MyApplication) getApplication()).networkStateReceiver.removeListener(this);
    }
}
