@file:Suppress("INVISIBLE_MEMBER", "INVISIBLE_REFERENCE")

package godau.fynn.moodledirect.activity

import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.view.LayoutInflater
import android.view.WindowManager
import androidx.activity.compose.setContent
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.material3.fromToken
import androidx.compose.material3.tokens.TopAppBarMediumTokens
import androidx.compose.material3.tokens.TopAppBarSmallTokens
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.input.nestedscroll.NestedScrollDispatcher
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import androidx.fragment.app.Fragment
import coil.compose.AsyncImage
import godau.fynn.moodledirect.R
import godau.fynn.moodledirect.activity.fragment.CourseEnrolFragment
import godau.fynn.moodledirect.activity.fragment.CourseFragment
import godau.fynn.moodledirect.activity.fragment.CourseFragment.Companion.newInstance
import godau.fynn.moodledirect.data.persistence.PreferenceHelper
import godau.fynn.moodledirect.model.database.Course
import godau.fynn.moodledirect.network.NetworkStateReceiver.OfflineStatusChangeListener
import godau.fynn.moodledirect.util.ActionItem
import godau.fynn.moodledirect.util.ActionItemSupplier
import godau.fynn.moodledirect.util.Constants
import godau.fynn.moodledirect.util.ExceptionHandler.tryAndThenThread
import godau.fynn.moodledirect.util.FileManagerWrapper
import godau.fynn.moodledirect.util.FileManagerWrapper.Companion.onRequestPermissionResult
import godau.fynn.moodledirect.util.MyApplication
import godau.fynn.moodledirect.view.NoDataView.InvokeListener
import godau.fynn.moodledirect.view.TwoRowsTopAppBar

class CourseDetailActivity : AppCompatActivity(), OfflineStatusChangeListener, InvokeListener {

    private var course: Course? by mutableStateOf(null)
    private var statusOffline: Boolean by mutableStateOf(false)

    private var actionItems: MutableList<ActionItem> = mutableStateListOf()

    @OptIn(ExperimentalMaterial3Api::class)
    var scrollBehavior: TopAppBarScrollBehavior? = null
    val scrollDispatcher = NestedScrollDispatcher()

    @OptIn(ExperimentalMaterial3Api::class)
    override fun onCreate(savedInstanceState: Bundle?) {
        if (MyApplication.getInstance().isDarkModeEnabled) {
            setTheme(R.style.AppTheme_NoActionBar_Dark_TranslucentStatus)
        }

        super.onCreate(savedInstanceState)
        window.setFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS, WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
        window.statusBarColor = Color.Transparent.toArgb()

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            window.isNavigationBarContrastEnforced = false
        }

        val token = PreferenceHelper(this).activeAccount.token!!

        setContent {
            Column {
                Box(
                    Modifier
                        .height(IntrinsicSize.Max)
                ) {

                    var containerColor = colorResource(id = R.color.colorPrimary)
                    var scrolledContainerColor = colorResource(id = R.color.colorPrimaryDark)

                    course?.headerImage?.let {
                        AsyncImage(
                            model = "${it.url}?token=$token",
                            contentDescription = null,
                            contentScale = ContentScale.Crop,
                            modifier = Modifier
                                .fillMaxWidth()
                        )

                        containerColor = Color.Black.copy(alpha = 0.1875f)
                        scrolledContainerColor = Color.Black.copy(alpha = 0.3125f)
                    }

                    Column {
                        TwoRowsTopAppBar(
                            title = {
                                course?.name?.let {
                                    Text(
                                        text = it,
                                        color = Color.White,
                                        overflow = TextOverflow.Clip
                                    )
                                }
                            },
                            titleTextStyle = MaterialTheme.typography.fromToken(
                                TopAppBarMediumTokens.HeadlineFont),
                            smallTitleTextStyle = MaterialTheme.typography.fromToken(
                                TopAppBarSmallTokens.HeadlineFont),
                            titleBottomPadding = 28.dp,
                            smallTitle = {
                                course?.shortname?.let {
                                    Text(
                                        text = it,
                                        color = Color.White,
                                        overflow = TextOverflow.Ellipsis,
                                        maxLines = 1
                                    )
                                }
                            },
                            navigationIcon = {
                                IconButton(onClick = { onBackPressed() }) {
                                    Icon(
                                        imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                                        contentDescription = stringResource(R.string._navigate_backward),
                                        tint = Color.White
                                    )
                                }
                            },
                            actions = {
                                for (action in actionItems) {
                                    IconButton(onClick = action.action) {
                                        Icon(
                                            painter = painterResource(action.icon),
                                            contentDescription = stringResource(action.title),
                                            tint = Color.White
                                        )
                                    }
                                }
                            },
                            colors = TopAppBarDefaults.largeTopAppBarColors(
                                containerColor = containerColor,
                                scrolledContainerColor = scrolledContainerColor
                            ),
                            windowInsets = TopAppBarDefaults.windowInsets,
                            maxHeight = 228.0.dp,
                            pinnedHeight = TopAppBarSmallTokens.ContainerHeight,
                            scrollBehavior = TopAppBarDefaults.exitUntilCollapsedScrollBehavior()
                                .also { scrollBehavior = it }
                        )

                        if (statusOffline) {
                            Text(
                                text = stringResource(R.string.offline_mode_enabled),
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .background(Color.Black.copy(alpha = 0.3125f))
                                    .padding(8.dp),
                                color = Color.White.copy(),
                                textAlign = TextAlign.Center,
                                style = MaterialTheme.typography.bodySmall
                            )
                        }

                    }

                }

                AndroidView(
                    factory = {
                        LayoutInflater.from(it)
                            .inflate(R.layout.activity_course_detail_view, null, false)
                    }
                )
            }
        }

        val refreshActionItems = { fragment: Fragment ->
            actionItems.clear()
            if (fragment is ActionItemSupplier) {
                actionItems.addAll(fragment.actionItems)
            }

        }

        supportFragmentManager.addFragmentOnAttachListener { _, fragment ->
            refreshActionItems(fragment)
            if (fragment !is CourseFragment)
                collapseAppBar()
        }
        supportFragmentManager.addOnBackStackChangedListener {
            refreshActionItems(supportFragmentManager.fragments[0])
        }

        /* We are getting the course to display either via ID (→ already enrolled)
         * or via serializable extra, in which case the user is not enrolled to the course.
         */
        if (intent.hasExtra(Constants.EXTRA_COURSE)) {
            tryAndThenThread( {
                // TODO this just exists to wait one UI tick
            },
                {
                    course = intent.getSerializableExtra(Constants.EXTRA_COURSE) as Course?
                    if (savedInstanceState == null) {
                        val fragment: Fragment = CourseEnrolFragment.newInstance(course)
                        supportFragmentManager.beginTransaction().replace(
                            R.id.course_activity_frame,
                            fragment, null
                        ).commit()
                    }
                }
            )
        } else {
            val courseId = intent.getIntExtra("id", -1)
            require(courseId != -1)
            tryAndThenThread(
                {
                    MyApplication.moodle().forceOffline().getCore().getCourse(courseId)
                },
                { course: Course? ->
                    if (savedInstanceState == null) {
                        val fragment: Fragment = newInstance(course)
                        supportFragmentManager.beginTransaction().replace(
                            R.id.course_activity_frame,
                            fragment, null
                        ).commit()
                    }
                    this.course = course
                }
            )
        }

        (application as MyApplication).networkStateReceiver.addListener(this)
    }

    @OptIn(ExperimentalMaterial3Api::class)
    fun collapseAppBar() {
        scrollBehavior?.let {
            //AnimationState(initialValue = it.state.heightOffset).animateTo(
            //    it.state.heightOffsetLimit,
            //    animationSpec = it.snapAnimationSpec!!
            //) { it.state.heightOffset = value }
            it.state.heightOffset = it.state.heightOffsetLimit
        }
    }

    override fun onNewOfflineStatus(offline: Boolean) {
        statusOffline = offline
        // TODO layout
        // TODO transition
    }

    override fun onInvokeNoData() {
        collapseAppBar()
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        if (requestCode == FileManagerWrapper.REQUEST_OPEN_DIRECTORY && resultCode == RESULT_OK) {
            onRequestPermissionResult(data, this)
        }
        super.onActivityResult(requestCode, resultCode, data)
    }
}
