/*
  This file is part of Subsonic.
	Subsonic is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.
	Subsonic is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.
	You should have received a copy of the GNU General Public License
	along with Subsonic. If not, see <http://www.gnu.org/licenses/>.
	Copyright 2015 (C) Scott Jackson
*/
package github.paroj.dsub2000.adapter;

import android.content.Context;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.ViewGroup;

import github.paroj.dsub2000.R;
import github.paroj.dsub2000.domain.PodcastChannel;
import github.paroj.dsub2000.domain.PodcastEpisode;
import github.paroj.dsub2000.util.ImageLoader;
import github.paroj.dsub2000.util.Util;
import github.paroj.dsub2000.view.FastScroller;
import github.paroj.dsub2000.view.PodcastChannelView;
import github.paroj.dsub2000.view.SongView;
import github.paroj.dsub2000.view.UpdateView;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;

public class PodcastChannelAdapter extends ExpandableSectionAdapter<Serializable> implements FastScroller.BubbleTextGetter {
	public static final int VIEW_TYPE_PODCAST_LEGACY = 1;
	public static final int VIEW_TYPE_PODCAST_LINE = 2;
	public static final int VIEW_TYPE_PODCAST_CELL = 3;
	public static final int VIEW_TYPE_PODCAST_EPISODE = 4;

	private ImageLoader imageLoader;
	private boolean largeCell;

	public PodcastChannelAdapter(Context context, List<Serializable> podcasts, ImageLoader imageLoader, OnItemClickedListener listener, boolean largeCell) {
		super(context, podcasts);
		this.imageLoader = imageLoader;
		this.onItemClickedListener = listener;
		this.largeCell = largeCell;
	}
	public PodcastChannelAdapter(Context context, List<String> headers, List<List<Serializable>> sections, ImageLoader imageLoader, OnItemClickedListener listener, boolean largeCell) {
		super(context, headers, sections, Arrays.asList(3, null));
		this.imageLoader = imageLoader;
		this.onItemClickedListener = listener;
		this.largeCell = largeCell;
		checkable = true;
	}

	@Override
	public UpdateView.UpdateViewHolder onCreateSectionViewHolder(ViewGroup parent, int viewType) {
		UpdateView updateView;
		if(viewType == VIEW_TYPE_PODCAST_EPISODE) {
			updateView = new SongView(context);
		} else if(viewType == VIEW_TYPE_PODCAST_LEGACY) {
			updateView = new PodcastChannelView(context);
		} else {
			updateView = new PodcastChannelView(context, imageLoader, viewType == VIEW_TYPE_PODCAST_CELL);
		}

		return new UpdateView.UpdateViewHolder(updateView);
	}

	@Override
	public void onBindViewHolder(UpdateView.UpdateViewHolder holder, Serializable item, int viewType) {
		if(viewType == VIEW_TYPE_PODCAST_EPISODE) {
			SongView songView = (SongView) holder.getUpdateView();
			PodcastEpisode episode = (PodcastEpisode) item;
			songView.setShowPodcast(true);
			songView.setObject(episode, !episode.isVideo());
		} else {
			holder.getUpdateView().setObject(item);
		}
	}

	@Override
	public int getItemViewType(Serializable item) {
		if(item instanceof PodcastChannel) {
			PodcastChannel channel = (PodcastChannel) item;

			if (imageLoader != null && channel.getCoverArt() != null) {
				return largeCell ? VIEW_TYPE_PODCAST_CELL : VIEW_TYPE_PODCAST_LINE;
			} else {
				return VIEW_TYPE_PODCAST_LEGACY;
			}
		} else {
			return VIEW_TYPE_PODCAST_EPISODE;
		}
	}

	@Override
	public String getTextToShowInBubble(int position) {
		Serializable item = getItemForPosition(position);
		if(item instanceof PodcastChannel) {
			PodcastChannel channel = (PodcastChannel) item;
			return getNameIndex(channel.getName(), true);
		} else {
			return null;
		}
	}

	@Override
	public void onCreateActionModeMenu(Menu menu, MenuInflater menuInflater) {
		if(Util.isOffline(context)) {
			menuInflater.inflate(R.menu.multiselect_media_offline, menu);
		} else {
			menuInflater.inflate(R.menu.multiselect_media, menu);
		}

		menu.removeItem(R.id.menu_remove_playlist);
		menu.removeItem(R.id.menu_star);
	}
}
