import 'package:collection/collection.dart';
import 'package:flutter/material.dart';
import 'package:trina_grid/trina_grid.dart';

abstract class IRowState {
  List<TrinaRow> get rows;

  /// [refRows] is a List&lt;TrinaRow&gt; type and holds the entire row data.
  ///
  /// [refRows] returns a list of final results
  /// according to pagination and filtering status.
  /// If the total number of rows is 100 and paginated in size 10,
  /// [refRows] returns a list of 10 of the current page.
  ///
  /// [refRows.originalList] to get the entire row data with pagination or filtering.
  ///
  /// A list with pagination and filtering applied and pagination not applied
  /// can be obtained with [refRows.filterOrOriginalList].
  ///
  /// Directly accessing [refRows] to process insert, remove, etc. may cause unexpected behavior.
  /// It is preferable to use the methods
  /// such as insertRows and removeRows of the built-in [TrinaGridStateManager] to handle it.
  FilteredList<TrinaRow> get refRows;

  List<TrinaRow> get checkedRows;

  List<TrinaRow> get checkedRowsViaSelect;

  List<TrinaRow> get unCheckedRows;

  bool get hasCheckedRow;

  bool get hasUnCheckedRow;

  /// Property for [tristate] value in [Checkbox] widget.
  bool? get tristateCheckedRow;

  /// Row index of currently selected cell.
  int? get currentRowIdx;

  /// Row of currently selected cell.
  TrinaRow? get currentRow;

  /// {@template trina_grid_row_wrapper}
  ///
  /// A wrapper that wraps arround each row.
  ///
  /// {@endtemplate}
  RowWrapper? get rowWrapper;

  TrinaRowColorCallback? get rowColorCallback;

  TrinaCellColorCallback? get cellColorCallback;

  int? getRowIdxByOffset(double offset);

  TrinaRow? getRowByIdx(int rowIdx);

  TrinaRow getNewRow();

  List<TrinaRow> getNewRows({int count = 1});

  void setRowChecked(TrinaRow row, bool flag, {bool notify = true});

  void insertRows(int rowIdx, List<TrinaRow> rows, {bool notify = true});

  void prependNewRows({int count = 1});

  void prependRows(List<TrinaRow> rows);

  void appendNewRows({int count = 1});

  void appendRows(List<TrinaRow> rows);

  void removeCurrentRow();

  void removeRows(List<TrinaRow> rows);

  void removeAllRows({bool notify = true});

  void moveRowsByOffset(
    List<TrinaRow> rows,
    double offset, {
    bool notify = true,
  });

  void moveRowsByIndex(List<TrinaRow> rows, int index, {bool notify = true});

  void toggleAllRowChecked(bool flag, {bool notify = true});

  /// Set the height of a specific row by index
  void setRowHeight(int rowIndex, double height);

  /// Reset a specific row to use the default height
  void resetRowHeight(int rowIndex);

  /// Reset all rows to use the default height
  void resetAllRowHeights();

  /// Get the effective height of a specific row
  double getRowHeight(int rowIndex);
}

mixin RowState implements ITrinaGridState {
  @override
  List<TrinaRow> get rows => [...refRows];

  @override
  List<TrinaRow> get checkedRows =>
      refRows.where((row) => row.checked!).toList(growable: false);

  @override
  List<TrinaRow> get checkedRowsViaSelect =>
      checkedRows.where((row) => row.checkedViaSelect).toList(growable: false);

  @override
  List<TrinaRow> get unCheckedRows =>
      refRows.where((row) => !row.checked!).toList(growable: false);

  @override
  bool get hasCheckedRow =>
      refRows.firstWhereOrNull((element) => element.checked!) != null;

  @override
  bool get hasUnCheckedRow =>
      refRows.firstWhereOrNull((element) => !element.checked!) != null;

  @override
  bool? get tristateCheckedRow {
    final length = refRows.length;

    if (length == 0) return false;

    int countTrue = 0;

    int countFalse = 0;

    for (var i = 0; i < length; i += 1) {
      refRows[i].checked == true ? ++countTrue : ++countFalse;

      if (countTrue > 0 && countFalse > 0) return null;
    }

    return countTrue == length;
  }

  @override
  int? get currentRowIdx => currentCellPosition?.rowIdx;

  @override
  TrinaRow? get currentRow {
    if (currentRowIdx == null) {
      return null;
    }

    return refRows[currentRowIdx!];
  }

  @override
  int? getRowIdxByOffset(double offset) {
    offset -= bodyTopOffset - scroll.verticalOffset;

    double currentOffset = 0.0;
    int? indexToMove;

    final int rowsLength = refRows.length;

    for (var i = 0; i < rowsLength; i += 1) {
      // Get row-specific height instead of using global height
      final rowHeight = getRowHeight(i);
      final rowTotalHeight =
          rowHeight + configuration.style.cellHorizontalBorderWidth;

      if (currentOffset <= offset && offset < currentOffset + rowTotalHeight) {
        indexToMove = i;
        break;
      }

      currentOffset += rowTotalHeight;
    }

    return indexToMove;
  }

  @override
  TrinaRow? getRowByIdx(int? rowIdx) {
    if (rowIdx == null || rowIdx < 0 || refRows.length - 1 < rowIdx) {
      return null;
    }

    return refRows[rowIdx];
  }

  @override
  TrinaRow getNewRow() {
    final cells = <String, TrinaCell>{};

    for (var column in refColumns.originalList) {
      cells[column.field] = TrinaCell(value: column.type.defaultValue);
    }

    return TrinaRow(cells: cells);
  }

  @override
  List<TrinaRow> getNewRows({int count = 1}) {
    List<TrinaRow> rows = [];

    for (var i = 0; i < count; i += 1) {
      rows.add(getNewRow());
    }

    if (rows.isEmpty) {
      return [];
    }

    return rows;
  }

  @override
  void setRowChecked(
    TrinaRow row,
    bool flag, {
    bool notify = true,
    bool checkedViaSelect = false,
  }) {
    final findRow = refRows.firstWhereOrNull(
      (element) => element.key == row.key,
    );

    if (findRow == null) {
      return;
    }

    findRow.setChecked(flag, viaSelect: checkedViaSelect);

    notifyListeners(notify, setRowChecked.hashCode);
  }

  @override
  void insertRows(int rowIdx, List<TrinaRow> rows, {bool notify = true}) {
    _insertRows(rowIdx, rows);

    /// Update currentRowIdx
    if (currentCell != null) {
      updateCurrentCellPosition(notify: false);

      // todo : whether to apply scrolling.
    }

    /// Update currentSelectingPosition
    if (currentSelectingPosition != null &&
        rowIdx <= currentSelectingPosition!.rowIdx!) {
      setCurrentSelectingPosition(
        cellPosition: TrinaGridCellPosition(
          columnIdx: currentSelectingPosition!.columnIdx,
          rowIdx: rows.length + currentSelectingPosition!.rowIdx!,
        ),
        notify: false,
      );
    }

    notifyListeners(notify, insertRows.hashCode);
  }

  @override
  void prependNewRows({int count = 1}) {
    prependRows(getNewRows(count: count));
  }

  @override
  void prependRows(List<TrinaRow> rows) {
    _insertRows(0, rows);

    /// Update currentRowIdx
    if (currentCell != null) {
      setCurrentCellPosition(
        TrinaGridCellPosition(
          columnIdx: currentCellPosition!.columnIdx,
          rowIdx: rows.length + currentRowIdx!,
        ),
        notify: false,
      );

      double offsetToMove = rows.length * rowTotalHeight;

      scrollByDirection(TrinaMoveDirection.up, offsetToMove);
    }

    /// Update currentSelectingPosition
    if (currentSelectingPosition != null) {
      setCurrentSelectingPosition(
        cellPosition: TrinaGridCellPosition(
          columnIdx: currentSelectingPosition!.columnIdx,
          rowIdx: rows.length + currentSelectingPosition!.rowIdx!,
        ),
        notify: false,
      );
    }

    notifyListeners(true, prependRows.hashCode);
  }

  @override
  void appendNewRows({int count = 1}) {
    appendRows(getNewRows(count: count));
  }

  @override
  void appendRows(List<TrinaRow> rows) {
    _insertRows(refRows.length, rows);

    notifyListeners(true, appendRows.hashCode);
  }

  @override
  void removeCurrentRow() {
    if (currentRowIdx == null) {
      return;
    }

    if (enabledRowGroups) {
      removeRowAndGroupByKey([currentRow!.key]);
    } else {
      refRows.removeAt(currentRowIdx!);
    }

    resetCurrentState(notify: false);

    notifyListeners(true, removeCurrentRow.hashCode);
  }

  @override
  void removeRows(List<TrinaRow> rows, {bool notify = true}) {
    if (rows.isEmpty) {
      return;
    }

    final Set<Key> removeKeys = Set.from(rows.map((e) => e.key));

    if (currentRowIdx != null &&
        refRows.length > currentRowIdx! &&
        removeKeys.contains(refRows[currentRowIdx!].key)) {
      resetCurrentState(notify: false);
    }

    Key? selectingCellKey;

    if (hasCurrentSelectingPosition) {
      selectingCellKey = refRows
          .originalList[currentSelectingPosition!.rowIdx!]
          .cells
          .entries
          .elementAt(currentSelectingPosition!.columnIdx!)
          .value
          .key;
    }

    if (enabledRowGroups) {
      removeRowAndGroupByKey(removeKeys);
    } else {
      refRows.removeWhereFromOriginal((row) => removeKeys.contains(row.key));
    }

    updateCurrentCellPosition(notify: false);

    setCurrentSelectingPositionByCellKey(selectingCellKey, notify: false);

    currentSelectingRows.removeWhere((row) => removeKeys.contains(row.key));

    notifyListeners(notify, removeRows.hashCode);
  }

  @override
  void removeAllRows({bool notify = true}) {
    if (refRows.originalList.isEmpty) {
      return;
    }

    refRows.clearFromOriginal();

    resetCurrentState(notify: false);

    notifyListeners(notify, removeAllRows.hashCode);
  }

  @override
  void moveRowsByOffset(
    List<TrinaRow> rows,
    double offset, {
    bool notify = true,
  }) {
    int? indexToMove = getRowIdxByOffset(offset);

    moveRowsByIndex(rows, indexToMove, notify: notify);
  }

  @override
  void moveRowsByIndex(
    List<TrinaRow> rows,
    int? indexToMove, {
    bool notify = true,
  }) {
    if (rows.isEmpty || indexToMove == null) {
      return;
    }

    if (indexToMove + rows.length > refRows.length) {
      indexToMove = refRows.length - rows.length;
    }

    if (isPaginated &&
        page > 1 &&
        indexToMove + pageRangeFrom > refRows.originalLength - 1) {
      indexToMove = refRows.originalLength - 1;
    }

    final Set<Key> removeKeys = Set.from(rows.map((e) => e.key));

    refRows.removeWhereFromOriginal((e) => removeKeys.contains(e.key));

    refRows.insertAll(indexToMove, rows);

    int sortIdx = 0;

    for (var element in refRows.originalList) {
      element.sortIdx = sortIdx++;
    }

    updateCurrentCellPosition(notify: false);

    if (onRowsMoved != null) {
      onRowsMoved!(TrinaGridOnRowsMovedEvent(idx: indexToMove, rows: rows));
    }

    notifyListeners(notify, moveRowsByIndex.hashCode);
  }

  @override
  void toggleAllRowChecked(bool flag, {bool notify = true}) {
    for (var row in refRows) {
      row.setChecked(flag);
    }

    notifyListeners(notify, toggleAllRowChecked.hashCode);
  }

  @override
  void setRowHeight(int rowIndex, double height) {
    if (rowIndex >= 0 && rowIndex < refRows.length) {
      // Since height is final, we need to create a new row with the new height
      final oldRow = refRows[rowIndex];
      final newRow = TrinaRow(
        cells: oldRow.cells,
        type: oldRow.type,
        sortIdx: oldRow.sortIdx,
        data: oldRow.data,
        checked: oldRow.checked ?? false,
        key: oldRow.key,
        frozen: oldRow.frozen,
        height: height,
      );

      // Copy over internal state
      newRow.setParent(oldRow.parent);
      newRow.setState(oldRow.state);

      // Replace the row in the list
      refRows[rowIndex] = newRow;

      notifyListeners(true);
    }
  }

  @override
  void resetRowHeight(int rowIndex) {
    if (rowIndex >= 0 && rowIndex < refRows.length) {
      // Since height is final, we need to create a new row without the height
      final oldRow = refRows[rowIndex];
      final newRow = TrinaRow(
        cells: oldRow.cells,
        type: oldRow.type,
        sortIdx: oldRow.sortIdx,
        data: oldRow.data,
        checked: oldRow.checked ?? false,
        key: oldRow.key,
        frozen: oldRow.frozen,
        // height will be null by default
      );

      // Copy over internal state
      newRow.setParent(oldRow.parent);
      newRow.setState(oldRow.state);

      // Replace the row in the list
      refRows[rowIndex] = newRow;

      notifyListeners(true);
    }
  }

  @override
  void resetAllRowHeights() {
    for (int i = 0; i < refRows.length; i++) {
      resetRowHeight(i);
    }
    // No need to call notifyListeners here since resetRowHeight already does it
  }

  @override
  double getRowHeight(int rowIndex) {
    if (rowIndex >= 0 && rowIndex < refRows.length) {
      return refRows[rowIndex].height ?? configuration.style.rowHeight;
    }
    return configuration.style.rowHeight;
  }

  void _insertRows(int index, List<TrinaRow> rows) {
    if (rows.isEmpty) {
      return;
    }

    int safetyIndex = _getSafetyIndexForInsert(index);

    if (enabledRowGroups) {
      insertRowGroup(safetyIndex, rows);
    } else {
      final bool append = refRows.isNotEmpty && index >= refRows.length;
      final targetIdx = append ? refRows.length - 1 : safetyIndex;
      final target = refRows.isEmpty ? null : refRows[targetIdx];
      int sortIdx = target?.sortIdx ?? 0;
      if (append) ++sortIdx;

      _setSortIdx(rows: rows, start: sortIdx);

      if (hasSortedColumn) {
        _increaseSortIdxGreaterThanOrEqual(
          rows: refRows.originalList,
          compare: sortIdx,
          increase: rows.length,
        );
      } else if (!append) {
        _increaseSortIdx(
          rows: refRows.originalList,
          start: target == null ? 0 : refRows.originalList.indexOf(target),
          increase: rows.length,
        );
      }

      for (final row in rows) {
        row.setState(TrinaRowState.added);
      }

      refRows.insertAll(safetyIndex, rows);

      TrinaGridStateManager.initializeRows(
        refColumns.originalList,
        rows,
        forceApplySortIdx: false,
      );
    }

    if (isPaginated) {
      resetPage(notify: false);
    }
  }

  int _getSafetyIndexForInsert(int index) {
    if (index < 0) {
      return 0;
    }

    if (index > refRows.length) {
      return refRows.length;
    }

    return index;
  }

  void _setSortIdx({required List<TrinaRow> rows, int start = 0}) {
    for (final row in rows) {
      row.sortIdx = start++;
    }
  }

  void _increaseSortIdx({
    required List<TrinaRow> rows,
    int start = 0,
    int increase = 1,
  }) {
    final length = rows.length;

    for (int i = start; i < length; i += 1) {
      rows[i].sortIdx += increase;
    }
  }

  void _increaseSortIdxGreaterThanOrEqual({
    required List<TrinaRow> rows,
    int compare = 0,
    int increase = 1,
  }) {
    for (final row in rows) {
      if (row.sortIdx < compare) {
        continue;
      }

      row.sortIdx += increase;
    }
  }
}
