import 'dart:io';
import 'package:flutter/material.dart';
import 'package:io_file_picker_ui/src/extensions.dart';
import 'package:io_file_picker_ui/src/fs_entities.dart';
import 'package:io_file_picker_ui/src/picker_config.dart';
import 'package:io_file_picker_ui/src/picker_result.dart';
import 'package:io_file_picker_ui/src/widgets/directory_list_tile.dart';
import 'package:io_file_picker_ui/src/widgets/file_list_tile.dart';
import 'package:io_file_picker_ui/src/widgets/link_list_tile.dart';
import 'package:path/path.dart' as p;

// TODO: show currently selected files/directories in sidebar
// TODO: when following link directory, remember link path to go back to instead
// of the resolved path

class IoFilePickerUi extends StatefulWidget {
  final PickerConfig pickerConfig;
  final String? dialogTitle;
  const IoFilePickerUi({
    super.key,
    this.pickerConfig = const SingleFilePickerConfig(),
    this.dialogTitle,
  });

  @override
  State<IoFilePickerUi> createState() => _IoFilePickerUiState();
}

class _IoFilePickerUiState extends State<IoFilePickerUi> {
  List<BaseFsEntityWrapper> fsEntities = [];
  Set<String> pickedEntityPaths = {};
  late String currentPath;
  static const rootPath = "/";

  String get initialPath {
    if (widget.pickerConfig.initialPath != null) {
      return widget.pickerConfig.initialPath!;
    }
    if (Platform.isAndroid) {
      return "/sdcard";
    } else if (Platform.isLinux) {
      return String.fromEnvironment("~");
    } else if (Platform.isWindows) {
      return String.fromEnvironment("USERPROFILE");
    } else {
      return "/";
    }
  }

  @override
  void initState() {
    super.initState();
    currentPath = initialPath;
    openDirectory(path: currentPath);
  }

  Future<void> openDirectory({required String path}) async {
    final List<BaseFsEntityWrapper> listing = [];
    // TODO: use stream and streaimport 'dart:io';mbuilder instead of list, for handling folders
    // with large number of files
    await for (final entity in Directory(path).list(followLinks: false)) {
      try {
        listing.add(
            await getCustomFsEntityFromStdlibEntity(fileSystemEntity: entity));
      } on FileSystemException catch (_) {
        // TODO: handle error more gracefully
      }
    }
    // TODO: sort by name, date etc
    listing.sort(
      (a, b) => a.name.compareTo(b.name),
    );
    setState(() {
      fsEntities = listing;
      currentPath = p.canonicalize(Directory(path).absolute.path);
    });
  }

  void showSnackBar(
      {required BuildContext context, required String text}) async {
    ScaffoldMessenger.of(context).clearSnackBars();
    final controller = ScaffoldMessenger.of(
      context,
    ).showSnackBar(SnackBar(content: Text(text)));
    await Future.delayed(Durations.long4);
    try {
      controller.close();
    } on AssertionError {
      // Ignore, it happens when closing an already closed snackbar
    }
  }

  Future<void> goUpDirectory() async {
    await openDirectory(path: Directory(currentPath).parent.path);
  }

  Future<void> onFilePicked({
    required BuildContext context,
    required String path,
  }) async {
    switch (widget.pickerConfig) {
      case SingleFilePickerConfig():
        Navigator.of(context).pop(SingleFilePickerResult(filePath: path));
      case MultipleFilePickerConfig(:final maxFiles):
        if (pickedEntityPaths.contains(path)) {
          setState(() {
            pickedEntityPaths.remove(path);
          });
        } else {
          if (pickedEntityPaths.length == maxFiles) {
            showSnackBar(
                context: context, text: "Maximum $maxFiles files allowed");
          } else {
            setState(() {
              pickedEntityPaths.add(path);
            });
          }
        }
      default:
        break;
    }
  }

  Future<void> onDirectoryPicked({
    required BuildContext context,
    required String path,
  }) async {
    switch (widget.pickerConfig) {
      case SingleDirectoryPickerConfig():
        Navigator.of(
          context,
        ).pop(SingleDirectoryPickerResult(directoryPath: path));
      case MultipleDirectoryPickerConfig(
          :final maxDirectories,
        ):
        if (pickedEntityPaths.contains(path)) {
          setState(() {
            pickedEntityPaths.remove(path);
          });
        } else {
          if (pickedEntityPaths.length == maxDirectories) {
            showSnackBar(
              context: context,
              text: "Maximum $maxDirectories directories allowed",
            );
          } else {
            setState(() {
              pickedEntityPaths.add(path);
            });
          }
        }
      default:
        break;
    }
  }

  @override
  Widget build(BuildContext context) {
    return PopScope(
      canPop: false,
      onPopInvokedWithResult: (didPop, result) async {
        if (!didPop) {
          if (currentPath == rootPath) {
            Navigator.of(context).pop();
          } else {
            await goUpDirectory();
          }
        }
      },
      child: Scaffold(
        appBar: AppBar(
          leading: CloseButton(
            onPressed: () {
              Navigator.of(context).pop();
            },
          ),
          title: Text(
            widget.dialogTitle ??
                switch (widget.pickerConfig) {
                  SingleFilePickerConfig() => "Pick file",
                  MultipleFilePickerConfig() => "Pick files",
                  SingleDirectoryPickerConfig() => "Pick directory",
                  MultipleDirectoryPickerConfig() => "Pick Directories",
                },
          ),
        ),
        body: Column(
          crossAxisAlignment: CrossAxisAlignment.stretch,
          children: [
            ListTile(
              leading: IconButton(
                onPressed: currentPath == rootPath
                    ? null
                    : () async {
                        await goUpDirectory();
                      },
                icon: Icon(Icons.arrow_back),
              ),
              title: Text(currentPath),
            ),
            SizedBox(height: 10),
            Expanded(
              child: ListView.builder(
                itemCount: fsEntities.length,
                itemBuilder: (context, index) {
                  final currentEntity = fsEntities[index];
                  switch (currentEntity) {
                    case FileEntityWrapper():
                      final currentFileCanonicalPath =
                          p.canonicalize(currentEntity.file.absolute.path);
                      return FileListTile(
                        fileEntity: currentEntity,
                        onTap: () async {
                          await onFilePicked(
                            context: context,
                            path: currentFileCanonicalPath,
                          );
                        },
                        enabled: () {
                          if (widget.pickerConfig
                              case FilePickerConfig(:final allowedExtensions)
                              when (allowedExtensions == null ||
                                  allowedExtensions.contains(
                                    currentEntity.file.fileExtension,
                                  ))) {
                            return true;
                          }
                          return false;
                        }(),
                        selected: pickedEntityPaths
                            .contains(currentFileCanonicalPath),
                      );
                    case DirectoryEntityWrapper():
                      final currentDirectoryCanonicalPath =
                          p.canonicalize(currentEntity.directory.absolute.path);
                      return DirectoryListTile(
                        directoryEntity: currentEntity,
                        onTap: () async {
                          await openDirectory(
                            path: currentEntity.directory.path,
                          );
                        },
                        onIconTap: (widget.pickerConfig
                                is MultipleDirectoryPickerConfig)
                            ? () async {
                                await onDirectoryPicked(
                                  context: context,
                                  path: currentDirectoryCanonicalPath,
                                );
                              }
                            : null,
                        onLongPress: (widget.pickerConfig
                                is MultipleDirectoryPickerConfig)
                            ? () async {
                                await onDirectoryPicked(
                                  context: context,
                                  path: currentDirectoryCanonicalPath,
                                );
                              }
                            : null,
                        selected: pickedEntityPaths
                            .contains(currentDirectoryCanonicalPath),
                      );
                    case LinkEntityWrapper():
                      final currentLinkCanonicalPath =
                          p.canonicalize(currentEntity.link.absolute.path);
                      return LinkListTile(
                        linkEntity: currentEntity,
                        onTap: () async {
                          switch (currentEntity.resolvedEntity) {
                            case FileEntityWrapper(:final file):
                              await onFilePicked(
                                context: context,
                                path: p.canonicalize(file.absolute.path),
                              );
                            case DirectoryEntityWrapper(:final directory):
                              await openDirectory(path: directory.path);
                            case LinkEntityWrapper():
                              // This will never happen, because resolvedEntity
                              // is never a link
                              break;
                          }
                        },
                        selected: pickedEntityPaths
                            .contains(currentLinkCanonicalPath),
                      );
                  }
                },
              ),
            ),
            SizedBox(height: 10),
            if (widget.pickerConfig
                case MultipleFilePickerConfig(:final minFiles))
              Padding(
                padding: const EdgeInsets.all(8.0),
                child: FilledButton(
                  style: FilledButton.styleFrom(
                    shape: RoundedRectangleBorder(
                      borderRadius: BorderRadius.zero,
                    ),
                  ),
                  onPressed: () {
                    if (pickedEntityPaths.length < minFiles) {
                      showSnackBar(
                          context: context,
                          text: "Minimum $minFiles files required");
                    } else {
                      Navigator.of(context).pop(
                        MutlipleFilePickerResult(
                          filePaths: pickedEntityPaths.toList(),
                        ),
                      );
                    }
                  },
                  child: Text("Pick (${pickedEntityPaths.length})"),
                ),
              ),
            if (widget.pickerConfig case DirectoryPickerConfig())
              Padding(
                padding: const EdgeInsets.all(8.0),
                child: Row(
                  mainAxisSize: MainAxisSize.max,
                  mainAxisAlignment: MainAxisAlignment.spaceEvenly,
                  spacing: 8,
                  children: [
                    if (widget.pickerConfig
                        case MultipleDirectoryPickerConfig())
                      Expanded(
                        child: ChoiceChip(
                          shape: RoundedRectangleBorder(
                            borderRadius: BorderRadiusGeometry.zero,
                          ),
                          selected: pickedEntityPaths.contains(currentPath),
                          onSelected: (_) async {
                            await onDirectoryPicked(
                              context: context,
                              path: currentPath,
                            );
                          },
                          label: SizedBox(
                            width: double.infinity,
                            child: Center(child: Text("Select")),
                          ),
                        ),
                      ),
                    Expanded(
                      child: FilledButton(
                        style: FilledButton.styleFrom(
                          shape: RoundedRectangleBorder(
                            borderRadius: BorderRadius.zero,
                          ),
                        ),
                        onPressed: () {
                          switch (widget.pickerConfig) {
                            case SingleDirectoryPickerConfig():
                              Navigator.of(context).pop(
                                SingleDirectoryPickerResult(
                                  directoryPath: currentPath,
                                ),
                              );
                            case MultipleDirectoryPickerConfig(
                                :final minDirectories
                              ):
                              if (pickedEntityPaths.length < minDirectories) {
                                showSnackBar(
                                  context: context,
                                  text:
                                      "Minimum $minDirectories directories required",
                                );
                              } else {
                                return Navigator.of(context).pop(
                                  MultipleDirectoryPickerResult(
                                    directoryPaths: pickedEntityPaths.toList(),
                                  ),
                                );
                              }
                            default:
                              break;
                          }
                        },
                        child: (widget.pickerConfig
                                is MultipleDirectoryPickerConfig)
                            ? Text("Pick (${pickedEntityPaths.length})")
                            : Text("Pick"),
                      ),
                    ),
                  ],
                ),
              ),
          ],
        ),
      ),
    );
  }
}
