import 'dart:io';

import 'package:equatable/equatable.dart';
import 'package:io_file_picker_ui/src/extensions.dart';

enum FileType { plainText, audio, video, image, document, archive, unknown }

sealed class BaseFsEntityWrapper extends Equatable {
  String get name;
  const BaseFsEntityWrapper();
}

class FileEntityWrapper extends BaseFsEntityWrapper {
  @override
  String get name => file.name;
  final File file;
  final int fileSize;
  final FileType fileType;
  const FileEntityWrapper({
    required this.file,
    required this.fileSize,
    required this.fileType,
  });
  @override
  List<Object?> get props => [file, fileSize, fileType];
}

class DirectoryEntityWrapper extends BaseFsEntityWrapper {
  @override
  String get name => directory.name;
  final Directory directory;
  final int contentsCount;
  const DirectoryEntityWrapper({
    required this.directory,
    required this.contentsCount,
  });
  @override
  List<Object?> get props => [directory, contentsCount];
}

class LinkEntityWrapper extends BaseFsEntityWrapper {
  @override
  String get name => link.name;
  final Link link;
  final BaseFsEntityWrapper resolvedEntity;
  const LinkEntityWrapper({required this.link, required this.resolvedEntity});
  @override
  List<Object?> get props => [link, resolvedEntity];
}

Future<BaseFsEntityWrapper> getCustomFsEntityFromStdlibEntity({
  required FileSystemEntity fileSystemEntity,
  int? depth,
}) async {
  switch (fileSystemEntity) {
    case File():
      return FileEntityWrapper(
        file: fileSystemEntity,
        fileSize: await fileSystemEntity.length(),
        fileType: await fileSystemEntity.getFileTypeFromFileExtension(),
      );
    case Directory():
      return DirectoryEntityWrapper(
        directory: fileSystemEntity,
        contentsCount: (await fileSystemEntity.list().toList()).length,
      );
    case Link():
      final resolvedPath = await fileSystemEntity.resolveSymbolicLinks();
      final FileSystemEntity resolvedEntity;
      if (await File(resolvedPath).exists()) {
        resolvedEntity = File(resolvedPath);
      } else if (await Directory(resolvedPath).exists()) {
        resolvedEntity = Directory(resolvedPath);
      } else {
        resolvedEntity = Link(resolvedPath);
      }
      if (depth != null && depth > 10) {
        throw FileSystemException(
          "Symbolic link recursion limit exceeded",
          fileSystemEntity.path,
        );
      }
      return LinkEntityWrapper(
        link: fileSystemEntity,
        resolvedEntity: await getCustomFsEntityFromStdlibEntity(
            fileSystemEntity: resolvedEntity, depth: (depth ?? 0) + 1),
      );
    default:
      throw UnsupportedError("This shouldn't happen");
  }
}
