A custom file picker which uses `dart:io` to browse the file system.

## Features

- Pick one or more files
- Pick one or more directories
- Filter files based on extensions.
- Put limits on number of files/directories selected

## Getting started

Since this library requires `dart:io` for browsing files, you need to ensure that you have the necessary permissions.  

### Android

For Android API level >=30, add
```xml
<uses-permission android:name="android.permission.MANAGE_EXTERNAL_STORAGE" />
```
under the `manifests` tag in `AndroidManifest.xml`, and then request the permission during runtime using [permission_handler](https://pub.dev/packages/permission_handler) package, with the permission `Permission.manageExternalStorage`.  

For Android API level >=16 and <=29, add
```xml
<uses-permission android:name="android.permission.READ_EXTERNAL_STORAGE" />
<uses-permission android:name="android.permission.WRITE_EXTERNAL_STORAGE" />
```
and request `Permission.storage`

> Be warned, it's highly likely that Google Play Store will reject your app with these storage permission, unless you have a good enough reason. 

## Usage

This library exposes a function which shows a full screen file picker dialog.

### To pick a file
```dart
import 'package:io_file_picker_ui/io_file_picker_ui.dart';

...


final filePickerResult = await showIoFilePicker(
    context, 
    dialogTitle: "Pick your avatar image",
    pickerConfig: SingleFilePickerConfig(
        allowedExtensions: [".jpg", ".png", ".gif"],
    ),
);

if(filePickerResult case SingleFilePickerResult(:final filePath)) {
    print("Image picked: $filePath");
} else {
    print("No image picked");
}

```

### To pick a directory
```dart
import 'package:io_file_picker_ui/io_file_picker_ui.dart';

...


final dirPickerResult = await showIoFilePicker(
    context, 
    dialogTitle: "Pick the backup folder",
    pickerConfig: SingleDirectoryPickerConfig(),
);

if(dirPickerResult case SingleDirectoryPickerResult(:final directoryPath)) {
    print("Backup folder: $directoryPath");
} else {
    print("No folder picked");
}

```
### PickerConfig
`pickerConfig` can be one of these types:

1. `SingleFilePickerConfig()`  
2. `MultipleFilePickerConfig()`
3. `SingleDirectoryPickerConfig()`
4. `MultipleDirectoryPickerConfig()`

### Parameters

- `initialPath`: All configs accept an optional `initialPath` parameter, which is the path opened by the dialog initially.  

- `allowedExtensions`: Both file picker configs accept an optional `allowedExtensions` parameter which is a list of extensions (including the dot) that are allowed to be picked.

- `minFiles` and `maxFiles`: `MultipleFilePickerConfig` accepts both these parameters which can be used to limit the number of files to be selected.

- `minDirectories` and `maxDirectories`: `MultipleDirectoryPickerConfig` accepts both these parameters which can be used to limit the number of directories to be selected.

### Result
`showIoFilePicker` returns a subclass of `PickerResult` or null, depending on the `pickerConfig` passed, ie  
Config | Result | Property
-| - | -
SingleFilePickerConfig | SingleFilePickerResult | filePath (String)
MultipleFilePickerConfig | MultipleFilePickerResult | filePaths (List\<String\>)
SingleDirectoryPickerConfig | SingleDirectoryPickerResult | directoryPath (String)
MultipleDirectoryPickerConfig | MultipleDirectoryPickerResult | directoryPaths (List\<String\>)

The result is `null` if the user closes the dialog. For multiple file/directory picking, the `filePaths`/`directoryPaths` may be an empty list, indicating that the user didn't select anything but pressed finish.


## Additional information
TODO (maybe):
- Add side menu to show selected items
- Fix for lag when exploring directories with large number of items
- Different ordering of items, like size, creation date, modified date, etc
- Better error handling
- Grid view