package fr.nocle.passegares.outils

import android.content.Context
import android.text.TextUtils
import fr.nocle.passegares.R
import org.junit.After
import org.junit.Assert
import org.junit.Before
import org.junit.Test
import org.junit.runner.RunWith
import org.mockito.Mock
import org.mockito.MockedStatic
import org.mockito.Mockito
import org.mockito.junit.MockitoJUnitRunner
import org.mockito.stubbing.Answer
import java.util.Calendar
import java.util.Date
import java.util.Locale

/**
 * Created by jonathanmm on 08/01/17.
 */
@RunWith(MockitoJUnitRunner::class)
class StringOutilsTest {
    private lateinit var textUtilsMockedStatic: MockedStatic<TextUtils>

    @Mock
    private lateinit var mockApplicationContext: Context

    @Before
    fun initMock() {
        textUtilsMockedStatic = Mockito.mockStatic(TextUtils::class.java)
        textUtilsMockedStatic.`when`<Any?> {
            TextUtils.join(
                Mockito.anyString(),
                Mockito.any<MutableIterable<String>>()
            )
        }.thenAnswer(Answer { invocation ->
            val delimiter = invocation.arguments[0] as String
            val bouts = invocation.arguments[1] as MutableIterable<*>

            var premier = true
            var retour = ""
            for (mot in bouts) {
                if (!premier) retour += delimiter
                else premier = false

                retour += mot
            }

            retour
        })
    }

    @After
    fun tearDown() {
        // Closing the mockStatic after each test
        textUtilsMockedStatic.close()
    }

    //DisplayBeautifulNameStation
    @Test
    fun displayBeautifullNameStation_unMot_formattageCorrect() {
        Assert.assertEquals("Bonhomme", StringOutils.displayBeautifullNameStation("BONHOMME"))
    }

    @Test
    fun displayBeautifullNameStation_troisMot_formattageCorrect() {
        Assert.assertEquals(
            "Bonhomme de Neige",
            StringOutils.displayBeautifullNameStation("BONHOMME DE NEIGE")
        )
    }

    @Test
    fun displayBeautifullNameStation_troisMotTiret_formattageCorrect() {
        Assert.assertEquals(
            "Bonhomme-de-Neige",
            StringOutils.displayBeautifullNameStation("BONHOMME-DE-NEIGE")
        )
    }

    @Test
    fun displayBeautifullNameStation_troisMotEspaceTiret_formattageCorrect() {
        Assert.assertEquals(
            "Bonhomme de-Neige",
            StringOutils.displayBeautifullNameStation("BONHOMME DE-NEIGE")
        )
    }

    @Test
    fun displayBeautifullNameStation_troisMotParentheses_formattageCorrect() {
        Assert.assertEquals(
            "Bonhomme (Neige)",
            StringOutils.displayBeautifullNameStation("BONHOMME (NEIGE)")
        )
    }

    @Test
    fun displayBeautifullNameStation_troisMotParenthesesLiaisons_formattageCorrect() {
        Assert.assertEquals(
            "Bonhomme (de Neige)",
            StringOutils.displayBeautifullNameStation("BONHOMME (DE NEIGE)")
        )
    }

    @Test
    fun displayBeautifullNameStation_troisMotApostrophe_formattageCorrect() {
        Assert.assertEquals(
            "Banane de l'Autruche",
            StringOutils.displayBeautifullNameStation("BANANE DE L'AUTRUCHE")
        )
    }

    @Test
    fun displayBeautifullNameStation_premierMotApostropheLiaison_formattageCorrect() {
        Assert.assertEquals("L'Autruche", StringOutils.displayBeautifullNameStation("L'AUTRUCHE"))
    }

    @Test
    fun displayBeautifullNameStation_tiretEntreEspaces_formattageCorrect() {
        Assert.assertEquals(
            "La Défense − Grand Arche",
            StringOutils.displayBeautifullNameStation("LA DÉFENSE - GRAND ARCHE")
        )
        Assert.assertEquals(
            "Dugny − La Courneuve",
            StringOutils.displayBeautifullNameStation("DUGNY - LA COURNEUVE")
        )
    }

    @Test
    fun displayBeautifullNameStation_casReel_formattageCorrect() {
        Assert.assertEquals(
            "Esplanade de la Défense",
            StringOutils.displayBeautifullNameStation("ESPLANADE DE LA DÉFENSE")
        )
        Assert.assertEquals(
            "Porte de Neuilly (Avenue de Madrid)",
            StringOutils.displayBeautifullNameStation("PORTE DE NEUILLY (AVENUE DE MADRID)")
        )
        Assert.assertEquals(
            "Arts et Métiers",
            StringOutils.displayBeautifullNameStation("ARTS ET MÉTIERS")
        )
        Assert.assertEquals(
            "Château d'Eau",
            StringOutils.displayBeautifullNameStation("CHÂTEAU D'EAU")
        )
        Assert.assertEquals(
            "Gare de l'Est",
            StringOutils.displayBeautifullNameStation("GARE DE L'EST")
        )
        Assert.assertEquals("La Chapelle", StringOutils.displayBeautifullNameStation("LA CHAPELLE"))
        Assert.assertEquals(
            "Asnières-sur-Seine",
            StringOutils.displayBeautifullNameStation("ASNIÈRES-SUR-SEINE")
        )
        Assert.assertEquals(
            "Soisy-sous-Mortmorency",
            StringOutils.displayBeautifullNameStation("SOISY-SOUS-MORTMORENCY")
        )
        Assert.assertEquals(
            "La Remise à Jorelle",
            StringOutils.displayBeautifullNameStation("LA REMISE À JORELLE")
        )
        Assert.assertEquals(
            "Rue des Boulets",
            StringOutils.displayBeautifullNameStation("RUE DES BOULETS")
        )
        Assert.assertEquals(
            "Saint-Germain-en-Laye",
            StringOutils.displayBeautifullNameStation("SAINT-GERMAIN-EN-LAYE")
        )
        Assert.assertEquals(
            "L'Onde (Maison des Arts)",
            StringOutils.displayBeautifullNameStation("L'ONDE (MAISON DES ARTS)")
        )
        Assert.assertEquals(
            "Viarme Talensac",
            StringOutils.displayBeautifullNameStation("VIARME -TALENSAC")
        )
    }

    // getRelativeDate
    @Test
    fun getRelativeDate_null_formattageCorrect() {
        Assert.assertEquals("", StringOutils.getRelativeDate(mockApplicationContext, null))
    }

    @Test
    fun getRelativeDate_maintenant_formattageCorrect() {
        Mockito.`when`(mockApplicationContext.getString(R.string.tempsRelatifSeconde, 0))
            .thenReturn("Il y a 0 seconde")
        Assert.assertEquals(
            "Il y a 0 seconde",
            StringOutils.getRelativeDate(mockApplicationContext, Date())
        )
    }

    @Test
    fun getRelativeDate_cinqSecondes_formattageCorrect() {
        Mockito.`when`(
            mockApplicationContext.getString(
                R.string.tempsRelatifSecondes,
                5
            )
        ).thenReturn("Il y a 5 secondes")
        Assert.assertEquals(
            "Il y a 5 secondes",
            StringOutils.getRelativeDate(mockApplicationContext, Date((Date()).time - 5000))
        )
    }

    @Test
    fun getRelativeDate_uneMinute_formattageCorrect() {
        Mockito.`when`(mockApplicationContext.getString(R.string.tempsRelatifMinute))
            .thenReturn("Il y a 1 minute")
        Assert.assertEquals(
            "Il y a 1 minute",
            StringOutils.getRelativeDate(mockApplicationContext, Date((Date()).time - 60000))
        )
    }

    @Test
    fun getRelativeDate_uneMinuteCinq_formattageCorrect() {
        Mockito.`when`(
            mockApplicationContext.getString(
                R.string.tempsRelatifMinuteSec,
                "05"
            )
        ).thenReturn("Il y a 1 minute 05 sec")
        Assert.assertEquals(
            "Il y a 1 minute 05 sec",
            StringOutils.getRelativeDate(mockApplicationContext, Date((Date()).time - 65000))
        )
    }

    @Test
    fun getRelativeDate_cinqMinutes_formattageCorrect() {
        Mockito.`when`(mockApplicationContext.getString(R.string.tempsRelatifMinutes, 5))
            .thenReturn("Il y a 5 minutes")
        Assert.assertEquals(
            "Il y a 5 minutes",
            StringOutils.getRelativeDate(mockApplicationContext, Date((Date()).time - 300000))
        ) //1000 * 60 * 5
    }

    @Test
    fun getRelativeDate_cinqMinutesCinq_formattageCorrect() {
        Mockito.`when`(
            mockApplicationContext.getString(
                R.string.tempsRelatifMinutesSec,
                5,
                "05"
            )
        ).thenReturn("Il y a 5 minutes 05 sec")
        Assert.assertEquals(
            "Il y a 5 minutes 05 sec",
            StringOutils.getRelativeDate(mockApplicationContext, Date((Date()).time - 305000))
        ) //1000 * 60 * 5
    }

    @Test
    fun getRelativeDate_uneHeure_formattageCorrect() {
        Mockito.`when`(mockApplicationContext.getString(R.string.tempsRelatifHeure))
            .thenReturn("Il y a 1 heure")
        Assert.assertEquals(
            "Il y a 1 heure",
            StringOutils.getRelativeDate(mockApplicationContext, Date((Date()).time - 3600000))
        ) //1000 * 60 * 60
    }

    @Test
    fun getRelativeDate_uneHeureCinq_formattageCorrect() {
        Mockito.`when`(
            mockApplicationContext.getString(
                R.string.tempsRelatifHeureMin,
                "05"
            )
        ).thenReturn("Il y a 1 heure 05 min")
        Assert.assertEquals(
            "Il y a 1 heure 05 min",
            StringOutils.getRelativeDate(mockApplicationContext, Date((Date()).time - 3900000))
        ) //1000 * 60 * 60
    }

    @Test
    fun getRelativeDate_cinqHeures_formattageCorrect() {
        Mockito.`when`(mockApplicationContext.getString(R.string.tempsRelatifHeures, 5))
            .thenReturn("Il y a 5 heures")
        Assert.assertEquals(
            "Il y a 5 heures",
            StringOutils.getRelativeDate(
                mockApplicationContext,
                Date((Date()).time - 18000000)
            )
        ) //1000 * 60 * 60 * 5
    }

    @Test
    fun getRelativeDate_cinqHeurescinq_formattageCorrect() {
        Mockito.`when`(
            mockApplicationContext.getString(
                R.string.tempsRelatifHeuresMin,
                5,
                "05"
            )
        ).thenReturn("Il y a 5 heures 05 min")
        Assert.assertEquals(
            "Il y a 5 heures 05 min",
            StringOutils.getRelativeDate(
                mockApplicationContext,
                Date((Date()).time - 18300000)
            )
        ) //1000 * 60 * 60 * 5
    }

    @Test
    fun getRelativeDate_unjour_formattageCorrect() {
        Mockito.`when`(mockApplicationContext.getString(R.string.tempsRelatifJour))
            .thenReturn("Il y a 1 jour")
        Assert.assertEquals(
            "Il y a 1 jour",
            StringOutils.getRelativeDate(
                mockApplicationContext,
                Date((Date()).time - 86400000)
            )
        ) //1000 * 60 * 60 * 24
    }

    @Test
    fun getRelativeDate_unjourUneHeureDix_formattageCorrect() {
        Mockito.`when`(
            mockApplicationContext.getString(
                R.string.tempsRelatifJourH,
                1,
                "10"
            )
        ).thenReturn("Il y a 1 jour, 1h10")
        Assert.assertEquals(
            "Il y a 1 jour, 1h10",
            StringOutils.getRelativeDate(
                mockApplicationContext,
                Date((Date()).time - 90600000)
            )
        ) //1000 * 60 * 60 * 24
    }

    @Test
    fun getRelativeDate_cinqjour_formattageCorrect() {
        Mockito.`when`(mockApplicationContext.getString(R.string.tempsRelatifJours, 5))
            .thenReturn("Il y a 5 jours")
        Assert.assertEquals(
            "Il y a 5 jours",
            StringOutils.getRelativeDate(
                mockApplicationContext,
                Date((Date()).time - 432000000)
            )
        ) //1000 * 60 * 60 * 24 * 5
    }

    @Test
    fun getRelativeDate_cinqjourUneHeureUne_formattageCorrect() {
        Mockito.`when`(
            mockApplicationContext.getString(
                R.string.tempsRelatifJoursH,
                5,
                1,
                "01"
            )
        ).thenReturn("Il y a 5 jours, 1h01")
        Assert.assertEquals(
            "Il y a 5 jours, 1h01",
            StringOutils.getRelativeDate(
                mockApplicationContext,
                Date((Date()).time - 435660000)
            )
        ) //1000 * 60 * 60 * 24 * 5
    }

    private fun putZero(number: Int): String {
        if (number <= 9) return '0'.toString() + number.toString()

        return number.toString()
    }

    @Test
    fun getRelativeDate_ancien_formattageCorrect() {
        Locale.setDefault(Locale.FRANCE)
        val cal = Calendar.getInstance(Locale.getDefault())
        cal.set(Calendar.YEAR, 2007)
        cal.set(Calendar.MONTH, Calendar.FEBRUARY) //Commence à 0 et non à 1…
        cal.set(Calendar.DATE, 14)
        cal.set(Calendar.HOUR_OF_DAY, 23)
        cal.set(Calendar.MINUTE, 37)
        cal.set(Calendar.SECOND, 25)
        val expectedDate =
            putZero(cal.get(Calendar.DATE)) + "/" + putZero(cal.get(Calendar.MONTH) + 1) + "/" + putZero(
                cal.get(Calendar.YEAR)
            ) + " " + putZero(cal.get(Calendar.HOUR_OF_DAY)) + "h" + putZero(cal.get(Calendar.MINUTE))
        Assert.assertEquals(
            expectedDate,
            StringOutils.getRelativeDate(mockApplicationContext, cal.getTime())
        )
    }

    //manageDeParticule
    @Test
    fun manageDeParticule_casGeneraux() {
        Assert.assertEquals("de la Girafe", StringOutils.manageDeParticule("la Girafe"))
        Assert.assertEquals("du Crabe", StringOutils.manageDeParticule("le Crabe"))
        Assert.assertEquals("des Moutons", StringOutils.manageDeParticule("les Moutons"))
        Assert.assertEquals("de Chat", StringOutils.manageDeParticule("Chat"))
    }


    //displayBeautifullDistance
    @Test
    fun displayBeautifullDistance_km_arrondiSup() {
        Locale.setDefault(Locale.FRANCE)
        Assert.assertEquals("2,35 km", StringOutils.displayBeautifullDistance(2349f))
    }

    @Test
    fun displayBeautifullDistance_km_deuxDecimales() {
        Locale.setDefault(Locale.FRANCE)
        Assert.assertEquals("2,34 km", StringOutils.displayBeautifullDistance(2344f))
    }

    @Test
    fun displayBeautifullDistance_km_uneDecimale() {
        Locale.setDefault(Locale.FRANCE)
        Assert.assertEquals("2,30 km", StringOutils.displayBeautifullDistance(2301f))
    }

    @Test
    fun displayBeautifullDistance_km_zeroDecimale() {
        Locale.setDefault(Locale.FRANCE)
        Assert.assertEquals("2,00 km", StringOutils.displayBeautifullDistance(2002f))
    }

    @Test
    fun displayBeautifullDistance_km_un() {
        Locale.setDefault(Locale.FRANCE)
        Assert.assertEquals("1,00 km", StringOutils.displayBeautifullDistance(999.7f))
    }

    @Test
    fun displayBeautifullDistance_m() {
        Locale.setDefault(Locale.FRANCE)
        Assert.assertEquals("50 m", StringOutils.displayBeautifullDistance(50f))
    }
}