package fr.nocle.passegares.visa;

import static android.content.res.Configuration.ORIENTATION_LANDSCAPE;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.core.view.MenuHost;
import androidx.core.view.MenuProvider;
import androidx.fragment.app.Fragment;

import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

import java.util.ArrayList;

import fr.nocle.passegares.GareActivity;
import fr.nocle.passegares.R;
import fr.nocle.passegares.controlleur.GareCtrl;
import fr.nocle.passegares.controlleur.LigneCtrl;
import fr.nocle.passegares.interfaces.OnNavigateIntentManager;
import fr.nocle.passegares.interfaces.ToolbarManager;
import fr.nocle.passegares.modele.GareTamponnee;
import fr.nocle.passegares.modele.Ligne;

public class VisaFragment extends Fragment {
    private ToolbarManager _toolbarManager;
    private OnNavigateIntentManager _onNavigateIntentManager;
    private TamponsAdapter adapter;

    private long _idLigne;
    private boolean _voirTamponDuJour;
    private String _titleToolbar;

    private boolean _voirFourni;
    private boolean _voirNecessite;
    private boolean _voirNiveau;

    private static String PREFERENCE_VOIR_FOURNI = "voirFourni";
    private static String PREFERENCE_VOIR_NECESSITE = "voirNecessite";
    private static String PREFERENCE_VOIR_NIVEAU = "voirNiveau";

    private static String PREFERENCE_NOM_FICHIER = "listeStationsPreferences";

    private void readBundle(Bundle b)
    {
        if(b != null)
        {
            _idLigne = b.getLong(ResumeVisaFragment.ARG_IDLIGNE, 0);
            _voirTamponDuJour = b.getBoolean(ResumeVisaFragment.ARG_DUJOUR, false);
        }
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        SharedPreferences preferences = requireActivity().getSharedPreferences(PREFERENCE_NOM_FICHIER, Context.MODE_PRIVATE);
        this._voirFourni = preferences.getBoolean(PREFERENCE_VOIR_FOURNI, false);
        this._voirNecessite = preferences.getBoolean(PREFERENCE_VOIR_NECESSITE, false);
        this._voirNiveau = preferences.getBoolean(PREFERENCE_VOIR_NIVEAU, false);
    }

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        super.onCreateView(inflater, container, savedInstanceState);
        View view = inflater.inflate(R.layout.activity_visa, null);

        readBundle(getArguments());

        generateListeGares(view);

        return view;
    }

    public void generateListeGares(@NonNull View view) {
        GareCtrl gareControlleur = new GareCtrl(getActivity());

        Ligne l = null;
        final ArrayList<GareTamponnee> listeTampons = gareControlleur.getGareTamponneeByLine(_idLigne, _voirTamponDuJour);

        //On s'occupe du titre
        int nbStations = 0;
        if(_idLigne == 0)
        {
            nbStations = listeTampons.size();
            String titre;
            if(_voirTamponDuJour)
            {
                if(nbStations >= 2)
                    titre = getString(R.string.tousTamponsDuJour);
                else
                    titre = getString(R.string.tousTamponDuJour);
            } else
            {
                if(nbStations >= 2)
                    titre = getString(R.string.tousTampons);
                else
                    titre = getString(R.string.tousTampon);
            }
            _titleToolbar = nbStations + " " + titre;
        } else {
            LigneCtrl ligneControlleur = new LigneCtrl(getActivity());
            l = ligneControlleur.get(_idLigne);
            ligneControlleur.close();

            String precision;
            if(!_voirTamponDuJour)
            {
                //On regarde ce qui est vraiment tamponné
                for(GareTamponnee g : listeTampons)
                {
                    if(g.getNbValidations() > 0)
                        nbStations++;
                }
                precision = " (" + nbStations + "/" + listeTampons.size() + ")";
            }
            else
                precision = " (" + listeTampons.size() + ")";
            _titleToolbar = l.getNom() + precision;

            TextView champSousNom = (TextView) view.findViewById(R.id.sousNom);
            if(l.getSousNom() != null && !l.getSousNom().isEmpty())
            {
                champSousNom.setText(l.getSousNom());
                champSousNom.setVisibility(View.VISIBLE);
            } else {
                champSousNom.setVisibility(View.GONE);
            }
        }
        // On met à jour le titre
        setToolbarTitle();

        ListView listView = (ListView) view.findViewById(R.id.listeTampons);
        LinearLayout emptyStateLayout = view.findViewById(R.id.emptyStateLayout);
        if(nbStations == 0 && _idLigne == 0)
        {
            TextView emptyStateLabel = view.findViewById(R.id.emptyStateLabel);

            if(this._voirTamponDuJour)
                emptyStateLabel.setText(R.string.emptyStateLigneTamponneeDuJour);
            else
                emptyStateLabel.setText(R.string.emptyStateLigneTamponnee);

            if(getResources().getConfiguration().orientation == ORIENTATION_LANDSCAPE)
            {
                emptyStateLayout.setOrientation(LinearLayout.HORIZONTAL);
            } else {
                emptyStateLayout.setOrientation(LinearLayout.VERTICAL);
            }

            // On affiche le empty state
            emptyStateLayout.setVisibility(View.VISIBLE);
            listView.setVisibility(View.GONE);
        } else {
            // Create the adapter to convert the array to views
            adapter = new TamponsAdapter(getActivity(), listeTampons, l);

            adapter.setAfficherFournit(_voirFourni);
            adapter.setAfficherNecessite(_voirNecessite);
            adapter.setAfficherNiveau(_voirNiveau);

            // Attach the adapter to a ListView
            listView.setAdapter(adapter);
            listView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
                @Override
                public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                    GareTamponnee gareTamponnee = listeTampons.get(position);
                    Intent i = new Intent(getActivity(), GareActivity.class);
                    i.putExtra(GareActivity.ARG_IDGARE, gareTamponnee.getIdGare());
                    startActivity(i);
                }
            });
            emptyStateLayout.setVisibility(View.GONE);
            listView.setVisibility(View.VISIBLE);
        }

        gareControlleur.close();
    }

    @RequiresApi(Build.VERSION_CODES.M)
    @Override
    public void onAttach(@NonNull Context context)
    {
        super.onAttach(context);

        this.createCallbackToolbarManager();
    }

    @Override
    public void onAttach(@NonNull Activity activity)
    {
        super.onAttach(activity);

        if(Build.VERSION.SDK_INT < Build.VERSION_CODES.M)
        {
            this.createCallbackToolbarManager();
        }
    }

    private void createCallbackToolbarManager() {
        try
        {
            _toolbarManager = (ToolbarManager) getActivity();
            _onNavigateIntentManager = (OnNavigateIntentManager) getActivity();
        } catch(ClassCastException ex)
        {
            Log.e("Visa", "Problème lors du cast de l'activité : " + ex.getMessage());
        }
    }

    private void setToolbarTitle() {
        _toolbarManager.setTitleToolbar(this._titleToolbar);
    }

    private MenuProvider menuProvider = new MenuProvider() {
        @Override
        public void onCreateMenu(@NonNull Menu menu, @NonNull MenuInflater menuInflater) {
            menuInflater.inflate(R.menu.menu_visa, menu);

            if(!_voirTamponDuJour)
            {
                // On cache la dernière option du menu pour tout voir
                MenuItem optionTout = menu.findItem(R.id.voirTout); // Option voir tout
                optionTout.setVisible(false);
            }

            menu.findItem(R.id.voirFourni).setChecked(_voirFourni);
            menu.findItem(R.id.voirNecessite).setChecked(_voirNecessite);
            menu.findItem(R.id.voirNiveau).setChecked(_voirNiveau);
        }

        @Override
        public boolean onMenuItemSelected(@NonNull MenuItem menuItem) {
            switch (menuItem.getItemId()) {
                case R.id.voirFourni:
                    triggerFourni(menuItem);
                    return true;
                case R.id.voirNecessite:
                    triggerNecessite(menuItem);
                    return true;
                case R.id.voirNiveau:
                    triggerNiveau(menuItem);
                    return true;
                case R.id.voirTout:
                    _onNavigateIntentManager.onMoveActivity(ResumeVisaActivity.class, ResumeVisaFragment.ARG_DUJOUR, _voirTamponDuJour);
                    return true;
                case android.R.id.home:
                    getActivity().finish();
                    return true;
            }
            return false;
        }
    };

    @Override
    public void onResume() {
        super.onResume();
        this.setToolbarTitle();

        ((MenuHost) requireActivity()).addMenuProvider(menuProvider);
    }

    @Override
    public void onPause() {
        super.onPause();

        ((MenuHost) requireActivity()).removeMenuProvider(menuProvider);
    }

    private void triggerNiveau(MenuItem item) {
        //On regarde le statut précédent de la case
        boolean nouveauStatut = !item.isChecked();

        item.setChecked(nouveauStatut);
        SharedPreferences preferences = requireActivity().getSharedPreferences(PREFERENCE_NOM_FICHIER, Context.MODE_PRIVATE);
        preferences.edit().putBoolean(PREFERENCE_VOIR_NIVEAU, nouveauStatut).apply();
        if(adapter != null)
        {
            adapter.setAfficherNiveau(nouveauStatut);
            adapter.notifyDataSetChanged();
        }
    }

    private void triggerFourni(MenuItem item) {
        //On regarde le statut précédent de la case
        boolean nouveauStatut = !item.isChecked();

        item.setChecked(nouveauStatut);
        SharedPreferences preferences = requireActivity().getSharedPreferences(PREFERENCE_NOM_FICHIER, Context.MODE_PRIVATE);
        preferences.edit().putBoolean(PREFERENCE_VOIR_FOURNI, nouveauStatut).apply();
        if(adapter != null)
        {
            adapter.setAfficherFournit(nouveauStatut);
            adapter.notifyDataSetChanged();
        }
    }

    private void triggerNecessite(MenuItem item) {
        //On regarde le statut précédent de la case
        boolean nouveauStatut = !item.isChecked();

        item.setChecked(nouveauStatut);
        SharedPreferences preferences = requireActivity().getSharedPreferences(PREFERENCE_NOM_FICHIER, Context.MODE_PRIVATE);
        preferences.edit().putBoolean(PREFERENCE_VOIR_NECESSITE, nouveauStatut).apply();
        if(adapter != null)
        {
            adapter.setAfficherNecessite(nouveauStatut);
            adapter.notifyDataSetChanged();
        }
    }
}
