package fr.nocle.passegares.visa;

import static android.content.res.Configuration.ORIENTATION_LANDSCAPE;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.core.view.MenuHost;
import androidx.core.view.MenuProvider;
import androidx.fragment.app.Fragment;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

import fr.nocle.passegares.R;
import fr.nocle.passegares.controlleur.LigneCtrl;
import fr.nocle.passegares.interfaces.OnNavigateIntentManager;
import fr.nocle.passegares.modele.LigneTamponnee;
import fr.nocle.passegares.interfaces.ToolbarManager;
import fr.nocle.passegares.region.RegionOutils;

public class ResumeVisaFragment extends Fragment {

    private ToolbarManager _toolbarManager;
    private OnNavigateIntentManager _onNavigateIntentManager;
    private boolean _uniquementDuJour = false;
    private boolean _voirLignesNonCommencees = false;
    private String _titleToolbar;
    private ListView _listeLigneTamponneeView;
    private int _premierElementVisible = 0;
    private int _scrollElementVisible = 0;
    private LigneTamponneeAdapter _adapter;

    public static String ARG_DUJOUR = "DUJOUR";
    public static String ARG_IDLIGNE = "IDLIGNE";

    private static String PREFERENCE_VOIR_LIGNES_NON_COMMENCEES = "voirLignesNonCommencees";

    private void readBundle(Bundle b)
    {
        if(b != null)
        {
            _uniquementDuJour = b.getBoolean(ARG_DUJOUR);
        }
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        SharedPreferences preferences = requireActivity().getPreferences(Context.MODE_PRIVATE);
        this._voirLignesNonCommencees = preferences.getBoolean(PREFERENCE_VOIR_LIGNES_NON_COMMENCEES, false);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        super.onCreateView(inflater, container, savedInstanceState);
        View view = inflater.inflate(R.layout.activity_resume_visa, null);

        readBundle(getArguments());

        if(savedInstanceState != null)
        {
            _premierElementVisible = savedInstanceState.getInt("POSITION_PREMIERELEMENT");
            _scrollElementVisible = savedInstanceState.getInt("POSITION_SCROLL");
        }

        generateListeLigne(view);

        return view;
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        int topPosition = 0;
        if(_listeLigneTamponneeView != null)
        {
            outState.putInt("POSITION_PREMIERELEMENT", _listeLigneTamponneeView.getFirstVisiblePosition());
            View firstChild = _listeLigneTamponneeView.getChildAt(0);
            if(firstChild != null)
            {
                topPosition = firstChild.getTop();
            }
        }
        outState.putInt("POSITION_SCROLL", topPosition);
    }

    private ArrayList<LigneTamponnee> getListeLigne()
    {
        LigneCtrl ligneControlleur = new LigneCtrl(getActivity());
        ArrayList<LigneTamponnee> listeLignes = ligneControlleur.getListeLignesTamponnees(this._uniquementDuJour, this._voirLignesNonCommencees);
        ligneControlleur.close();
        return listeLignes;
    }

    public void generateListeLigne(View view) {
        final ArrayList<LigneTamponnee> listeLignes = getListeLigne();

        Collections.sort(listeLignes, (o1, o2) -> {
            int comparaisonRegion = RegionOutils.Companion.comparerRegions(o1.getRegion(), o2.getRegion());
            if(comparaisonRegion != 0)
            {
                return comparaisonRegion;
            }

            // Même région, on va trier selon l'ordre de la ligne
            return Integer.compare(o1.getOrdre(), o2.getOrdre());
        });

        //On s'occupe du titre
        _titleToolbar = generateTitle(listeLignes);
        setToolbarTitle();

        _listeLigneTamponneeView = view.findViewById(R.id.listeLignes);

        if(listeLignes.isEmpty())
        {
            LinearLayout emptyStateLayout = view.findViewById(R.id.emptyStateLayout);

            TextView emptyStateLabel = view.findViewById(R.id.emptyStateLabel);

            if(this._uniquementDuJour)
                emptyStateLabel.setText(R.string.emptyStateLigneTamponneeDuJour);
            else
                emptyStateLabel.setText(R.string.emptyStateLigneTamponnee);

            if(getResources().getConfiguration().orientation == ORIENTATION_LANDSCAPE)
            {
                emptyStateLayout.setOrientation(LinearLayout.HORIZONTAL);
            } else {
                emptyStateLayout.setOrientation(LinearLayout.VERTICAL);
            }

            // On affiche le empty state
            emptyStateLayout.setVisibility(View.VISIBLE);
            _listeLigneTamponneeView.setVisibility(View.INVISIBLE);
        } else {
            // Create the adapter to convert the array to views
            _adapter = new LigneTamponneeAdapter(getActivity(), listeLignes, this._uniquementDuJour);

            // Attach the adapter to a ListView
            _listeLigneTamponneeView.invalidate();
            _listeLigneTamponneeView.setAdapter(_adapter);
            final boolean voirTamponDuJour = this._uniquementDuJour;
            _listeLigneTamponneeView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
                @Override
                public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                    int realPosition = _adapter.getRealPositionOfLigne(position);
                    LigneTamponnee ligneTamponnee = listeLignes.get(realPosition);
                    Intent i = new Intent(getActivity(), VisaActivity.class);
                    i.putExtra(ARG_IDLIGNE, ligneTamponnee.getIdLigne());
                    if (voirTamponDuJour)
                        i.putExtra(ARG_DUJOUR, true);
                    startActivity(i);
                }
            });
        }

        this.restoreScrollPosition();
    }

    private void setToolbarTitle() {
        _toolbarManager.setTitleToolbar(this._titleToolbar);
    }

    private String generateTitle(ArrayList<LigneTamponnee> listeLignes)
    {
        int nbTampons = listeLignes.size();
        int titre;
        if(this._uniquementDuJour)
        {
            if(nbTampons >= 2)
                titre = R.string.titre_tampons_du_jour;
            else
                titre = R.string.titre_tampon_du_jour;
        } else if(this._voirLignesNonCommencees)
        {
            if(nbTampons >= 2)
                titre = R.string.titreLignesEnTout;
            else
                titre = R.string.titreLigneEnTout;
        } else
        {
            if(nbTampons >= 2)
                titre = R.string.titre_tampons;
            else
                titre = R.string.titre_tampon;
        }
        return getString(titre, nbTampons);
    }

    @androidx.annotation.RequiresApi(Build.VERSION_CODES.M)
    @Override
    public void onAttach(@NonNull Context context)
    {
        super.onAttach(context);

        this.createCallbackToolbarManager();
    }

    @Override
    public void onAttach(@NonNull Activity activity)
    {
        super.onAttach(activity);

        if(Build.VERSION.SDK_INT < Build.VERSION_CODES.M)
        {
            this.createCallbackToolbarManager();
        }
    }

    private void createCallbackToolbarManager() {
        try
        {
            _toolbarManager = (ToolbarManager) getActivity();
            _onNavigateIntentManager = (OnNavigateIntentManager) getActivity();
        } catch(ClassCastException ex)
        {
            Log.e("ResumeVisa", "Problème lors du cast de l'activité : " + ex.getMessage());
        }
    }

    private MenuProvider menuProvider = new MenuProvider() {
        @Override
        public void onCreateMenu(@NonNull Menu menu, @NonNull MenuInflater menuInflater) {
            // Si on a toutes les lignes, on ajoute le bouton pour la liste des stations
            menuInflater.inflate(R.menu.menu_tampon, menu);

            if(_uniquementDuJour)
            {
                menu.findItem(R.id.voirTout).setEnabled(false).setVisible(false);
                menu.findItem(R.id.voirLignesNonCommencees).setEnabled(false).setVisible(false);
            }

            if(_voirLignesNonCommencees)
            {
                menu.findItem(R.id.voirLignesNonCommencees).setChecked(true);
            }
        }

        @Override
        public boolean onMenuItemSelected(@NonNull MenuItem menuItem) {
            switch (menuItem.getItemId()) {
                case R.id.voirTout:
                    _onNavigateIntentManager.onMoveActivity(VisaActivity.class, ARG_IDLIGNE, 0, ARG_DUJOUR, _uniquementDuJour);
                    return true;
                case R.id.voirLignesNonCommencees:
                    triggerVoirLignesNonCommencees(menuItem);
                    return true;
                case android.R.id.home:
                    getActivity().finish();
                    return true;
            }
            return false;
        }
    };

    private void triggerVoirLignesNonCommencees(MenuItem item) {
        //On regarde le statut précédent de la case
        boolean nouveauStatut = !item.isChecked();

        item.setChecked(nouveauStatut);
        _voirLignesNonCommencees = nouveauStatut;
        SharedPreferences preferences = requireActivity().getPreferences(Context.MODE_PRIVATE);
        preferences.edit().putBoolean(PREFERENCE_VOIR_LIGNES_NON_COMMENCEES, nouveauStatut).apply();
        if(_adapter != null)
        {
            View view = getView();
            generateListeLigne(view);
            _adapter.notifyDataSetChanged();
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        this.setToolbarTitle();
        this.restoreScrollPosition();

        ((MenuHost) requireActivity()).addMenuProvider(menuProvider);
    }

    @Override
    public void onPause() {
        super.onPause();

        _premierElementVisible = _listeLigneTamponneeView.getFirstVisiblePosition();
        View firstChild = _listeLigneTamponneeView.getChildAt(0);
        if(firstChild == null)
        {
            _scrollElementVisible = 0;
        } else {
            _scrollElementVisible = firstChild.getTop();
        }
        ((MenuHost) requireActivity()).removeMenuProvider(menuProvider);
    }

    private void restoreScrollPosition()
    {
        _listeLigneTamponneeView.setSelectionFromTop(_premierElementVisible, _scrollElementVisible);
    }
}
