package fr.nocle.passegares.visa;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;
import android.widget.Toast;

import java.util.ArrayList;
import java.util.Objects;

import fr.nocle.passegares.R;
import fr.nocle.passegares.modele.LigneTamponnee;
import fr.nocle.passegares.region.Region;
import fr.nocle.passegares.region.RegionCtrl;

/**
 * Created by jonathanmm on 01/10/16.
 */

public class LigneTamponneeAdapter extends BaseAdapter {
    private static final int TYPE_REGION = 0;
    private static final int TYPE_VILLE = 1;
    private static final int TYPE_LIGNE = 2;
    private final Context context;
    private final boolean voirTamponDuJour;
    private ArrayList<Region> listeRegions;
    private ArrayList<String> listeVilles;
    private ArrayList<Integer> positionRegions;
    private ArrayList<Integer> positionVilles;
    private final ArrayList<LigneTamponnee> listeLigne;

    public LigneTamponneeAdapter(Context context, ArrayList<LigneTamponnee> liste, boolean voirTamponDuJour)
    {
        this.context = context;
        this.voirTamponDuJour = voirTamponDuJour;
        this.listeLigne = liste;
        initRegion(liste);
    }

    private void initRegion(ArrayList<LigneTamponnee> listeLignes) {
        listeRegions = new ArrayList<>();
        positionRegions = new ArrayList<>();
        listeVilles = new ArrayList<>();
        positionVilles = new ArrayList<>();
        long lastRegion = -1;
        String lastVille = null;
        int currentPosition = 0;
        for(LigneTamponnee l : listeLignes) {
            if (l.getRegion().getId() != lastRegion) {
                listeRegions.add(l.getRegion());
                positionRegions.add(currentPosition);
                currentPosition++;
                lastVille = null;
            }

            String ville = l.getVille();
            if(ville != null && ville.isEmpty())
            {
                // On normalise la donnée
                ville = null;
            }
            if (!Objects.equals(ville, lastVille))
            {
                if(ville == null) {
                    listeVilles.add(this.context.getString(R.string.region));
                }
                else{
                    listeVilles.add(ville);
                }
                positionVilles.add(currentPosition);
                currentPosition++;
            }
            lastRegion = l.getRegion().getId();
            lastVille = ville;
            currentPosition++;
        }
    }

    @Override
    public int getCount() {
        return listeLigne.size() + listeRegions.size() + listeVilles.size();
    }

    @Override
    public LigneTamponnee getItem(int position) {
        return listeLigne.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent)
    {
        int itemType = getItemViewType(position);
        if(itemType == TYPE_LIGNE) {
            // Check if an existing view is being reused, otherwise inflate the view
            if (convertView == null) {
                convertView = LayoutInflater.from(this.context).inflate(R.layout.ligne_tamponnee_viewer, parent, false);
            }

            //Il faut trouver le nombre de région qui existent avant
            int realPosition = getRealPositionOfLigne(position);
            LigneTamponnee l = getItem(realPosition);

            TextView champ;
            champ = convertView.findViewById(R.id.nomLigne);
            setNomLigne(champ, l.getNomLigne(), l.getRegion());

            TextView champSousNom = convertView.findViewById(R.id.sousNomLigne);
            if(l.getSousNomLigne() != null && !l.getSousNomLigne().isEmpty())
            {
                champSousNom.setText(l.getSousNomLigne());
                champSousNom.setVisibility(View.VISIBLE);
            } else {
                champSousNom.setVisibility(View.GONE);
            }

            champ = convertView.findViewById(R.id.nbTampons);

            ProgressBar progressBar = convertView.findViewById(R.id.progressBar);
            int nbTampons = l.getNbTampons();
            int maxTampons = l.getNbGares();
            if (voirTamponDuJour) {
                String tampon;
                if (l.getNbTampons() > 1)
                    tampon = convertView.getContext().getString(R.string.tampons);
                else
                    tampon = convertView.getContext().getString(R.string.tampon);

                champ.setText(nbTampons + " " + tampon);
                progressBar.setVisibility(View.GONE);
            } else {
                champ.setText(nbTampons + "/" + maxTampons);

                Drawable progressDrawable = progressBar.getProgressDrawable().mutate();
                if(nbTampons == maxTampons)
                {
                    progressDrawable.setColorFilter(
                            context.getResources().getColor(
                                    R.color.succes
                            ),
                            android.graphics.PorterDuff.Mode.SRC_IN);
                } else {
                    progressDrawable.clearColorFilter();
                }
                progressBar.setProgressDrawable(progressDrawable);

                progressBar.setMax(maxTampons);
                progressBar.setProgress(nbTampons);

                progressBar.setVisibility(View.VISIBLE);
            }

            ImageView iconView = convertView.findViewById(R.id.iconeLigne);

            Integer icon = IndiceLigneDisplayer.Companion.getIcon(l.getRegion().getId(), l.getVille(), l.getNomLigne());

            if(icon == null)
            {
                iconView.setImageDrawable(null);
                iconView.setVisibility(View.INVISIBLE);
            } else {
                iconView.setImageResource(icon);
                iconView.setVisibility(View.VISIBLE);
            }
        } else if(itemType == TYPE_REGION)
        {
            // Check if an existing view is being reused, otherwise inflate the view
            if (convertView == null) {
                convertView = LayoutInflater.from(this.context).inflate(R.layout.titre_nom_region_viewer, parent, false);
            }

            TextView champ = convertView.findViewById(R.id.titreRegion);
            Region region = listeRegions.get(getSectionForPositionRegion(position));
            champ.setText(region.getNom());

            configurerBoutonFavori(convertView, region);
        } else if(itemType == TYPE_VILLE)
        {
            // Check if an existing view is being reused, otherwise inflate the view
            if (convertView == null) {
                convertView = LayoutInflater.from(this.context).inflate(R.layout.titre_nom_ville_viewer, parent, false);
            }

            TextView champ = convertView.findViewById(R.id.titreVille);
            champ.setText(listeVilles.get(getSectionForPositionVille(position)));
        }
        return convertView;
    }

    private void configurerBoutonFavori(View v, Region r)
    {
        ImageButton estFavori = v.findViewById(R.id.estFavori);
        if(r.getFavori() > 0) {
            estFavori.setImageResource(R.drawable.fa_star);
        } else {
            estFavori.setImageResource(R.drawable.fa_starempty);
        }
        estFavori.setOnClickListener(view -> {
            RegionCtrl regionCtrl = new RegionCtrl(context);
            if(r.getFavori() > 0)
            {
                regionCtrl.removeFavori(r);
                Toast.makeText(context, R.string.notificationRegionRetraitFavori, Toast.LENGTH_SHORT).show();
            } else {
                regionCtrl.setFavori(r);
                Toast.makeText(context, R.string.notificationRegionAjoutFavori, Toast.LENGTH_SHORT).show();
            }
            regionCtrl.close();
            notifyDataSetInvalidated();
        });
        estFavori.setVisibility(View.VISIBLE);
    }

    public int getRealPositionOfLigne(int position) {
        int nbItems = 0;
        for(int positionItem : positionRegions)
        {
            if(positionItem < position)
                nbItems++;
        }
        for(int positionItem : positionVilles)
        {
            if(positionItem < position)
                nbItems++;
        }
        return position - nbItems;
    }

    static public void setNomLigne(TextView champ, String nomLigne, Region region)
    {
        if(nomLigne.equals("Ligne Unique") && region != null)
            champ.setText(nomLigne + " (" + region.getNom() + ")");
        else
            champ.setText(nomLigne);
    }

    private int getSectionForPositionRegion(int position) {
        return positionRegions.indexOf(position);
    }

    private int getSectionForPositionVille(int position) {
        return positionVilles.indexOf(position);
    }

    @Override
    public int getItemViewType(int position) {
        if(positionRegions.contains(position)) return TYPE_REGION;
        if(positionVilles.contains(position)) return TYPE_VILLE;
        return TYPE_LIGNE;
    }

    @Override
    public int getViewTypeCount() {
        return 3;
    }

    @Override
    public boolean isEnabled(int position) {
        return getItemViewType(position) != TYPE_REGION && getItemViewType(position) != TYPE_VILLE;
    }
}
