package fr.nocle.passegares.radar;

import android.app.Activity;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.location.Location;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import androidx.core.app.NotificationCompat;
import androidx.appcompat.app.AlertDialog;

import android.preference.PreferenceManager;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.Toast;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import fr.nocle.passegares.boutique.BoutiqueActivity;
import fr.nocle.passegares.GareActivity;
import fr.nocle.passegares.R;
import fr.nocle.passegares.adapter.GaresAdapter;
import fr.nocle.passegares.boutique.BoutiqueConstantes;
import fr.nocle.passegares.boutique.OuvrirBoutique;
import fr.nocle.passegares.controlleur.GareCtrl;
import fr.nocle.passegares.controlleur.GareDansLigneCtrl;
import fr.nocle.passegares.controlleur.InventaireCtrl;
import fr.nocle.passegares.controlleur.LigneCtrl;
import fr.nocle.passegares.controlleur.TamponCtrl;
import fr.nocle.passegares.gares.Coordonnee;
import fr.nocle.passegares.gares.CoordonneeCtrl;
import fr.nocle.passegares.modele.Gare;
import fr.nocle.passegares.modele.Ligne;
import fr.nocle.passegares.modele.Tampon;
import fr.nocle.passegares.outils.CouleurOutils;
import fr.nocle.passegares.outils.StringOutils;
import fr.nocle.passegares.succes.SuccesManager;
import fr.nocle.passegares.succes.SuccesCtrl;

/**
 * Created by jonathanmm on 15/08/17.
 */
public class MessageHandler extends Handler {
    private final Activity mainActivity;
    private final GareCtrl gareControlleur;
    private final GareDansLigneCtrl gareDansLigneControlleur;
    private final TamponCtrl tamponControlleur;
    private final NotificationCompat.Builder mBuilder;
    private ArrayList<Gare> listeGares;
    private final InventaireCtrl inventaireControlleur;
    private final SuccesManager succesManager;
    private final CoordonneeCtrl coordonneeCtrl;
    private final LigneCtrl ligneControlleur;

    private Gare gareCouranteRadar;
    private boolean isGarePlusProcheSelectionnee;

    static public final int ACTUALISER_GARE_PLUS_PROCHE = 3;
    private static final String PREFERENCE_NOTIFICATION_AVANCEMENT_LIGNE = "Notification_Avancement_Ligne";
    private static final String PREFERENCE_NOUVEL_AFFICHAGE_RADAR = "Nouvel_Affichage_Radar";
    private static final int maxDistanceValidationEnMetres = 150;

    public MessageHandler(Activity activity)
    {
        mainActivity = activity;
        gareControlleur = new GareCtrl(activity);
        gareDansLigneControlleur = new GareDansLigneCtrl(activity);
        tamponControlleur = new TamponCtrl(activity);
        inventaireControlleur = new InventaireCtrl(activity);
        coordonneeCtrl = new CoordonneeCtrl(activity);
        ligneControlleur = new LigneCtrl(activity);
        SuccesCtrl succesControlleur = new SuccesCtrl(activity);
        succesManager = new SuccesManager(activity, succesControlleur, ligneControlleur, gareControlleur, tamponControlleur, gareDansLigneControlleur);
        mBuilder = null;
        gareCouranteRadar = null;
        isGarePlusProcheSelectionnee = true;
    }

    @Override
    public void handleMessage(Message message) {
        Bundle recu = message.getData();
        double lat = recu.getDouble("LAT", -1.0);
        double lon = recu.getDouble("LON", -1.0);

        if (lat == -1.0 && lon == -1.0) {
            if (mBuilder == null) {
                mettreAJourAffichageGareNoLocation(R.string.localisationEnCours);
            }

            return;
        }

        Location location = new Location("GPS");
        location.setLatitude(lat);
        location.setLongitude(lon);

        ArrayList<Coordonnee> listeCoordonnees = coordonneeCtrl.getNearlest(location);
        float minDistance = -1;

        if(isGarePlusProcheSelectionnee)
        {
            gareCouranteRadar = null;
        }

        // On va trouver la distance la plus petite pour chaque gare
        HashMap<Long, ArrayList<Float>> distancesGares = new HashMap<>();
        for (Coordonnee coor : listeCoordonnees) {
            float distance = location.distanceTo(coor.getLocation());
            ArrayList<Float> listCoorGare;
            if (distancesGares.containsKey(coor.getIdGare())) {
                listCoorGare = distancesGares.get(coor.getIdGare());
            } else {
                listCoorGare = new ArrayList<Float>();
            }
            listCoorGare.add(distance);
            distancesGares.put(coor.getIdGare(), listCoorGare);
        }

        // Maintenant, on va créer notre listeGares
        listeGares = new ArrayList<>();
        boolean hasGareCourante = isGarePlusProcheSelectionnee;
        Gare garePlusProche = null;
        for (Map.Entry<Long, ArrayList<Float>> kvp : distancesGares.entrySet()) {
            float minDistanceGare = 9999999;
            for (float distance : kvp.getValue()) {
                minDistanceGare = Math.min(minDistanceGare, distance);
            }

            int distanceEntiere = Math.round(minDistanceGare);
            if (distanceEntiere < 1000) {
                Gare g = gareControlleur.get(kvp.getKey());
                if (g != null) {
                    g.setDistance(minDistanceGare);
                    listeGares.add(g);

                    if (minDistance == -1 || minDistanceGare < minDistance) {
                        minDistance = minDistanceGare;

                        if (isGarePlusProcheSelectionnee) {
                            gareCouranteRadar = g;
                        } else {
                            garePlusProche = g;
                        }
                    }

                    if(!isGarePlusProcheSelectionnee && g.getId() == gareCouranteRadar.getId())
                    {
                        gareCouranteRadar.setDistance(minDistanceGare);
                        hasGareCourante = true;
                    }
                }
            }
        }

        if(!hasGareCourante)
        {
            // La gare qu'on a choisi n'est plus dans la liste, ça fallback sur la plus proche
            gareCouranteRadar = garePlusProche;
            isGarePlusProcheSelectionnee = true;
        }

        if(gareCouranteRadar != null)
        {
            gareDansLigneControlleur.remplirListeLignesDeGare(gareCouranteRadar);
        }

        Collections.sort(listeGares, (o1, o2) -> {
            return Double.compare(o1.getDistance(), o2.getDistance());
        });

        if(mBuilder == null) {
            mettreAJourAffichage(gareControlleur, gareCouranteRadar);
        }
    }

    private EvoluateInfo canIEvoluate(Gare garePlusProche)
    {
        int niveauCourant = garePlusProche.getNiveau();
        EvoluateInfo retour = new EvoluateInfo();
        if(niveauCourant == 0)
        {
            int nbTampon = garePlusProche.getNbTampons();
            retour.canIEvoluate = nbTampon >= 3;
            retour.nbElementsToEvoluate = nbTampon;
        }
        else if(niveauCourant == 1)
        {
            int nbTickets = inventaireControlleur.getNbTicket(garePlusProche.getCouleurEvo());
            retour.canIEvoluate = nbTickets >= 10;
            retour.nbElementsToEvoluate = nbTickets;
        }
        else if(niveauCourant == 2)
        {
            int nbTickets = inventaireControlleur.getNbTicket(garePlusProche.getCouleurEvo());
            retour.canIEvoluate = nbTickets >= 25;
            retour.nbElementsToEvoluate = nbTickets;
        }

        return retour;
    }

    private void prepareDialogEvolution(View viewDialog, Gare garePlusProche, String labelEvolution, int nbTicketsNecessaire)
    {
        TextView labelMessage = viewDialog.findViewById(R.id.evolutionLabel);
        TextView nbTicket = viewDialog.findViewById(R.id.nombreTicket);
        ImageView iconeTicket = viewDialog.findViewById(R.id.iconTicket);
        CouleurOutils.setTicketIcon(iconeTicket, garePlusProche.getCouleurEvo());

        labelMessage.setText(labelEvolution);
        int nbTicketsNiveauSuivant = 0;
        if(garePlusProche.getNiveau() == 1)
            nbTicketsNiveauSuivant = 10;
        else if(garePlusProche.getNiveau() == 2)
            nbTicketsNiveauSuivant = 25;

        nbTicket.setText(mainActivity.getString(R.string.progressionSucces, nbTicketsNecessaire, nbTicketsNiveauSuivant));
    }

    private void gestionBoutonEvolution(final GareCtrl gareControlleur, final Gare garePlusProche, final ImageView boutonEvolution) {
        boutonEvolution.setVisibility(View.VISIBLE);
        boutonEvolution.setEnabled(true);

        //On va créer la boîte de dialogue pour l'évolution
        AlertDialog.Builder dialogBuilder = new AlertDialog.Builder(mainActivity);
        dialogBuilder.setTitle(R.string.evoluerGare);

        //On regarde si la condition de changement de niveau est respectée
        EvoluateInfo evoluateInfo = canIEvoluate(garePlusProche);
        if (evoluateInfo.canIEvoluate)
        {
            String labelEvolution = "Voulez-vous vraiment faire passer la gare au niveau "+ (garePlusProche.getNiveau() + 1) + " ?";
            if(garePlusProche.getNiveau() == 0)
                dialogBuilder.setMessage(labelEvolution);
            else {
                LayoutInflater inflater = mainActivity.getLayoutInflater();

                View viewDialog = inflater.inflate(R.layout.dialog_evolution, null);
                prepareDialogEvolution(viewDialog, garePlusProche, labelEvolution, evoluateInfo.nbElementsToEvoluate);
                dialogBuilder.setView(viewDialog);
            }
            dialogBuilder.setPositiveButton(R.string.boutonValider, (dialog, which) -> {
            Gare g = gareControlleur.get(garePlusProche.getId());
            g.setNiveau(g.getNiveau() + 1);
            gareControlleur.update(g);

            //On met aussi à jour la vue
            if(!listeGares.isEmpty())
                listeGares.get(0).setNiveau(g.getNiveau());

            //On prélève aussi les tickets
            if(g.getNiveau() == 2)
                inventaireControlleur.jeterTicket(garePlusProche.getCouleurEvo(), 10);
            else if(g.getNiveau() == 3)
                inventaireControlleur.jeterTicket(garePlusProche.getCouleurEvo(), 25);

            //Il faut invalider le menu pour le mettre à jour
            mainActivity.invalidateOptionsMenu();

            dialog.cancel();
            mettreAJourAffichage(gareControlleur, g);
           });
            dialogBuilder.setNegativeButton(R.string.boutonAnnuler, (dialog, id) -> {
                // Refus, on ferme
                dialog.cancel();
            });

            //On va aussi changer la couleur du bouton
            boutonEvolution.setColorFilter(Color.GREEN, PorterDuff.Mode.MULTIPLY);
        } else
        {
            if(garePlusProche.getNiveau() == 0) {
                int labelId = evoluateInfo.nbElementsToEvoluate > 1 ? R.string.evolutionGareNiveauUnPasAssezS : R.string.evolutionGareNiveauUnPasAssez;
                dialogBuilder.setMessage(String.format(mainActivity.getString(labelId), evoluateInfo.nbElementsToEvoluate));
            } else if(garePlusProche.getNiveau() >= 1 && garePlusProche.getNiveau() <= 2) {
                LayoutInflater inflater = mainActivity.getLayoutInflater();

                View viewDialog = inflater.inflate(R.layout.dialog_evolution, null);

                prepareDialogEvolution(viewDialog, garePlusProche, mainActivity.getString(R.string.evolutionGareNiveauPlusUnConfirmation)+ " " + (garePlusProche.getNiveau() + 1), evoluateInfo.nbElementsToEvoluate);

                dialogBuilder.setView(viewDialog);
            }
            else {
                dialogBuilder.setMessage(R.string.evolutionGareNiveauMax);
            }

            //TODO : Mettre un bouton d'accord
            dialogBuilder.setPositiveButton(R.string.boutonValider, (dialog, id) -> {
                //On ferme la boîte
                dialog.cancel();
            });

            //Et on remet la couleur de base du bouton
            boutonEvolution.clearColorFilter();
        }
        final AlertDialog dialog = dialogBuilder.create();

        boutonEvolution.setOnClickListener(v -> dialog.show());
    }

    private void gestionBoutonFourniture(Gare garePlusProche, final ImageView boutonFournisseur)
    {
        //On ne peut prendre un ticket que si on est au moins niveau 1
        if(garePlusProche.getNiveau() > 0)
        {
            boutonFournisseur.setVisibility(View.VISIBLE);
            CouleurOutils.setTicketIcon(boutonFournisseur, garePlusProche.getCouleur());
        }
        else
            boutonFournisseur.setVisibility(View.INVISIBLE);
    }

    private void gestionBoutonCreerBoutique(final Gare garePlusProche, final Button boutonCreerBoutique, final Button boutonVoirBoutique) {
        //Uniquement s'il n'y a pas déjà une boutique, et qu'on est au moins niveau deux
        if(garePlusProche.getNiveau() >= BoutiqueConstantes.niveauOuverture && (garePlusProche.getIdBoutique() == null || garePlusProche.getIdBoutique() == 0)) {
            boutonCreerBoutique.setVisibility(View.VISIBLE);
            boutonCreerBoutique.setOnClickListener(v -> {
                OuvrirBoutique ouvrirBoutique = new OuvrirBoutique(mainActivity, gareControlleur, garePlusProche);
                ouvrirBoutique.setCallbackSuccessDialog(() -> {
                    Toast.makeText(mainActivity, R.string.boutiqueOuvertureConfirmation, Toast.LENGTH_LONG).show();
                    boutonCreerBoutique.setVisibility(View.GONE);
                    gestionBoutonVoirBoutique(garePlusProche, boutonVoirBoutique);
                });
                ouvrirBoutique.ouvrirDialog();
            });
        } else {
            boutonCreerBoutique.setVisibility(View.GONE);
        }
    }

    private void gestionBoutonVoirBoutique(final Gare garePlusProche, final Button boutonVoirBoutique) {
        //Uniquement s'il n'y a pas déjà une boutique, et qu'on est au moins niveau deux
        if(garePlusProche.getIdBoutique() > 0) {
            boutonVoirBoutique.setVisibility(View.VISIBLE);
            boutonVoirBoutique.setOnClickListener(v -> {
                Intent i = new Intent(mainActivity.getApplicationContext(), BoutiqueActivity.class);
                i.putExtra("BOUTIQUEID", garePlusProche.getIdBoutique());
                mainActivity.startActivity(i);
            });
        } else {
            boutonVoirBoutique.setVisibility(View.GONE);
        }
    }

    private boolean canStamp(Gare garePlusProche)
    {
        int nombreHeureEntreDeuxTampons = 8;
        Date limiteTampon = new Date(System.currentTimeMillis() - (3600 * nombreHeureEntreDeuxTampons * 1000));
        return garePlusProche.getDerniereValidationDate() == null ||
                garePlusProche.getDerniereValidationDate().before(limiteTampon);
    }

    private void ouvrirInformationsGare(Gare gare)
    {
        Intent i = new Intent(mainActivity, GareActivity.class);
        i.putExtra(GareActivity.ARG_IDGARE, gare.getId());
        mainActivity.startActivity(i);
    }

    private void mettreAJourAffichage(final GareCtrl gareControlleur, final Gare garePlusProche) {
        TextView champ = mainActivity.findViewById(R.id.garePlusProcheNom); //Le premier champ récupéré sera toujours celui du nom de la gare
        if (champ == null) //On a quitté le fragment
            return;

        //On récupère les boutons
        final Button boutonTamponner = mainActivity.findViewById(R.id.boutonTampon);
        final Button boutonCreerBoutique = mainActivity.findViewById(R.id.boutonCreerBoutique);
        final Button boutonVoirBoutique = mainActivity.findViewById(R.id.boutonVoirBoutique);
        final ImageView boutonFournisseur = mainActivity.findViewById(R.id.boutonFournisseur);
        final ImageView boutonEvolution = mainActivity.findViewById(R.id.boutonEvolution);
        final Button boutonInfoGare = mainActivity.findViewById(R.id.boutonInfoGare);

        SharedPreferences preferences = PreferenceManager.getDefaultSharedPreferences(mainActivity);
        boolean isNewInterface = preferences.getBoolean(PREFERENCE_NOUVEL_AFFICHAGE_RADAR, true);

        if (garePlusProche == null) {
            champ.setText(R.string.aucuneGareProche);
            TextView surTitre = mainActivity.findViewById(R.id.garePlusProcheSurTitre);
            surTitre.setVisibility(View.GONE);
            TextView sousTitre = mainActivity.findViewById(R.id.garePlusProcheSousTitre);
            sousTitre.setVisibility(View.GONE);
            champ = mainActivity.findViewById(R.id.garePlusProcheNiveau);
            champ.setVisibility(View.GONE);
            champ = mainActivity.findViewById(R.id.garePlusProcheDistance);
            champ.setText(R.string.naDistance);
            boutonTamponner.setEnabled(false);
            boutonFournisseur.setVisibility(View.GONE);
            boutonEvolution.setVisibility(View.GONE);
            boutonCreerBoutique.setVisibility(View.GONE);
            boutonVoirBoutique.setVisibility(View.GONE);
            boutonInfoGare.setVisibility(View.GONE);
            ListView listView = mainActivity.findViewById(R.id.listeGaresProches);
            listView.setAdapter(null);
            return;
        }

        // Create the adapter to convert the array to views
        GaresAdapter adapter = new GaresAdapter(mainActivity, listeGares);

        // Attach the adapter to a ListView
        ListView listView = mainActivity.findViewById(R.id.listeGaresProches);
        listView.setAdapter(adapter);
        listView.setOnItemClickListener((parent, view, position, id) -> {
            if(!isNewInterface)
            {
                Gare gareCourante = listeGares.get(position);
                ouvrirInformationsGare(gareCourante);
                return;
            }

            gareCouranteRadar = listeGares.get(position);
            isGarePlusProcheSelectionnee = position == 0;
            mettreAJourAffichage(gareControlleur, gareCouranteRadar);
        });

        int distanceEntiere = Math.toIntExact(Math.round(gareCouranteRadar.getDistance()));

        champ.setText(StringOutils.displayBeautifullNameStation(garePlusProche.getNom()));

        TextView surTitreChamp = mainActivity.findViewById(R.id.garePlusProcheSurTitre);
        String surTitre = garePlusProche.getSurTitre();
        if (surTitre != null && !surTitre.trim().isEmpty()) {
            surTitreChamp.setText(surTitre);
            surTitreChamp.setVisibility(View.VISIBLE);
        } else {
            surTitreChamp.setVisibility(View.INVISIBLE);
        }

        TextView sousTitreChamp = mainActivity.findViewById(R.id.garePlusProcheSurTitre);
        String sousTitre = garePlusProche.getSousTitre();
        if (sousTitre != null && !sousTitre.trim().isEmpty()) {
            sousTitreChamp.setText(sousTitre);
            sousTitreChamp.setVisibility(View.VISIBLE);
        } else {
            sousTitreChamp.setVisibility(View.INVISIBLE);
        }

        champ = mainActivity.findViewById(R.id.garePlusProcheNiveau);
        champ.setText(mainActivity.getString(R.string.niveauNumero, garePlusProche.getNiveau()));
        champ.setVisibility(View.VISIBLE);
        champ = mainActivity.findViewById(R.id.garePlusProcheDistance);
        champ.setText(mainActivity.getString(R.string.distanceEnMetres, distanceEntiere));

        if(isNewInterface)
        {
            boutonInfoGare.setOnClickListener(l -> ouvrirInformationsGare(garePlusProche));
            boutonInfoGare.setVisibility(View.VISIBLE);
        } else {
            ViewGroup.MarginLayoutParams lp = (ViewGroup.MarginLayoutParams) boutonVoirBoutique.getLayoutParams();
            lp.setMarginEnd(0);
            boutonVoirBoutique.setLayoutParams(lp);
            lp = (ViewGroup.MarginLayoutParams) boutonCreerBoutique.getLayoutParams();
            lp.setMarginEnd(0);
            boutonCreerBoutique.setLayoutParams(lp);
            boutonInfoGare.setVisibility(View.GONE);
        }
        gestionBoutonFourniture(garePlusProche, boutonFournisseur);

        if (distanceEntiere <= maxDistanceValidationEnMetres) {
            if (canStamp(garePlusProche)) {
                boutonTamponner.setEnabled(true);
                boutonTamponner.setOnClickListener(v -> {
                    //On tamponne
                    Tampon t = new Tampon(-1, garePlusProche.getId(), new Date());
                    tamponControlleur.create(t);

                    //On met à jour la gare
                    garePlusProche.setDerniereValidationDate(t.getDate());
                    garePlusProche.incrementeNbTampons();
                    gareControlleur.update(garePlusProche);

                    //On donne les éventuels succès
                    succesManager.verifierSucces(garePlusProche);

                    boutonTamponner.setEnabled(false);

                    Toast.makeText(mainActivity, R.string.gareTamponneeConfirmation, Toast.LENGTH_SHORT).show();

                    afficherNotificationAvancementLigne(garePlusProche);

                    //On va au passage lui donner un ticket s'il peut en avoir un
                    if (garePlusProche.getNiveau() >= 1)
                        donnerTicket(garePlusProche.getCouleur());

                    //Et on regarde si on peut monter de niveau ou pas
                    gestionBoutonEvolution(gareControlleur, garePlusProche, boutonEvolution);
                });
            }

            gestionBoutonEvolution(gareControlleur, garePlusProche, boutonEvolution);
            gestionBoutonCreerBoutique(garePlusProche, boutonCreerBoutique, boutonVoirBoutique);
            gestionBoutonVoirBoutique(garePlusProche, boutonVoirBoutique);
            return;
        }

        boutonTamponner.setEnabled(false);
        boutonEvolution.setVisibility(View.GONE);
        boutonCreerBoutique.setVisibility(View.GONE);
        boutonVoirBoutique.setVisibility(View.GONE);
    }

    private void cacherBouton(int idBouton)
    {
        View bouton = mainActivity.findViewById(idBouton);
        bouton.setVisibility(View.GONE);
    }

    public void mettreAJourAffichageGareNoLocation(int texteResourceId)
    {
        TextView champ = mainActivity.findViewById(R.id.garePlusProcheNom); //Le premier champ récupéré sera toujours celui du nom de la gare
        if (champ == null) //On a quitté le fragment
            return;

        champ.setText(texteResourceId);
        TextView surTitre = mainActivity.findViewById(R.id.garePlusProcheSurTitre);
        surTitre.setVisibility(View.GONE);
        TextView sousTitre = mainActivity.findViewById(R.id.garePlusProcheSousTitre);
        sousTitre.setVisibility(View.GONE);
        champ = mainActivity.findViewById(R.id.garePlusProcheNiveau);
        champ.setVisibility(View.GONE);
        champ = mainActivity.findViewById(R.id.garePlusProcheDistance);
        champ.setText(R.string.naDistance);
        Button boutonTamponner = mainActivity.findViewById(R.id.boutonTampon);
        boutonTamponner.setEnabled(false);
        cacherBouton(R.id.boutonFournisseur);
        cacherBouton(R.id.boutonEvolution);
        cacherBouton(R.id.boutonCreerBoutique);
        cacherBouton(R.id.boutonVoirBoutique);
        cacherBouton(R.id.boutonInfoGare);
        ListView listView = mainActivity.findViewById(R.id.listeGaresProches);
        listView.setAdapter(null);
    }

    private void afficherNotificationAvancementLigne(Gare garePlusProche) {
        // Si on a activé l'option, on affiche aussi un petit message sur l'avancement
        if(garePlusProche.getNbTampons() == 1)
        {
            SharedPreferences preferences = PreferenceManager.getDefaultSharedPreferences(mainActivity);
            boolean notifEvolution = preferences.getBoolean(PREFERENCE_NOTIFICATION_AVANCEMENT_LIGNE, false);
            if(notifEvolution)
            {
                ArrayList<Ligne> lignes = gareControlleur.getCorrespondances(garePlusProche);
                int numLigneAffiche = 0;
                int maxLigneAffiche = 3;
                for(Ligne l : lignes)
                {
                    String notificationTexte = String.format(mainActivity.getString(R.string.notificationAvancementLigne), l.getNom(), ligneControlleur.getNombreGareTamponnees(l), l.getNbGares());
                    Toast.makeText(mainActivity, notificationTexte, Toast.LENGTH_SHORT).show();
                    numLigneAffiche++;
                    if(numLigneAffiche >= maxLigneAffiche)
                    {
                        break;
                    }
                }

                if(lignes.size() > maxLigneAffiche)
                {
                    String notificationTexte = String.format(mainActivity.getString(R.string.notificationAvancementLigneOverflow), lignes.size() - maxLigneAffiche);
                    Toast.makeText(mainActivity, notificationTexte, Toast.LENGTH_SHORT).show();
                }
            }
        }
    }

    private void donnerTicket(int couleur) {
        inventaireControlleur.donnerTicket(couleur);

        mainActivity.invalidateOptionsMenu();
    }
}
