package fr.nocle.passegares.controlleur;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.location.Location;
import android.text.TextUtils;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;

import fr.nocle.passegares.bdd.BoutiqueBDD;
import fr.nocle.passegares.bdd.GareDansLigneBDD;
import fr.nocle.passegares.bdd.InventaireBDD;
import fr.nocle.passegares.bdd.LigneBDD;
import fr.nocle.passegares.region.RegionBDD;
import fr.nocle.passegares.bdd.TamponBDD;
import fr.nocle.passegares.gares.CoordonneeBDD;
import fr.nocle.passegares.modele.Gare;
import fr.nocle.passegares.bdd.GareBDD;
import fr.nocle.passegares.modele.GareTamponnee;
import fr.nocle.passegares.modele.Ligne;
import fr.nocle.passegares.region.Region;
import fr.nocle.passegares.succes.SuccesManager;
import fr.nocle.passegares.succes.SuccesBDD;

/**
 * Created by jonathanmm on 03/09/16.
 */
public class GareCtrl extends Controlleur {
    public GareCtrl(Context contexte) {
        super(contexte);
        this.open();
    }

    public GareCtrl(SQLiteDatabase bdd)
    {
        super(bdd);
    }

    public void create(Gare g)
    {
        if(get(g.getIdStif()) == null)
            g.setId(bdd.insert(GareBDD.TABLE_NOM, null, creerDepuisObj(g)));
    }

    public void delete(long id)
    {
        bdd.delete(GareBDD.TABLE_NOM, GareBDD.TABLE_CLE + " = ?", new String[] {String.valueOf(id)});
        bdd.delete(GareDansLigneBDD.TABLE_NOM, GareDansLigneBDD.TABLE_ID_GARE + " = ?", new String[] {String.valueOf(id)});
        bdd.delete(TamponBDD.TABLE_NOM, TamponBDD.TABLE_NOM_GARE + " = ?", new String[] {String.valueOf(id)});
    }

    public void update(Gare g)
    {
        bdd.update(GareBDD.TABLE_NOM, creerDepuisObj(g), GareBDD.TABLE_CLE + " = ?", new String[] {String.valueOf(g.getId())});
    }

    public Gare get(long id)
    {
        Cursor c = bdd.query(GareBDD.TABLE_NOM,
                new String[] {GareBDD.TABLE_ID_STIF, GareBDD.TABLE_NOM_GARE,
                        GareBDD.TABLE_SUR_TITRE, GareBDD.TABLE_SOUS_TITRE,
                        GareBDD.TABLE_LONGITUDE, GareBDD.TABLE_LATITUDE,
                        GareBDD.TABLE_EXPLOITANT, GareBDD.TABLE_NIVEAU,
                        GareBDD.TABLE_COULEUR, GareBDD.TABLE_COULEUR_EVO,
                        GareBDD.TABLE_NB_VALIDATIONS, GareBDD.TABLE_DERNIERE_VALIDATION, 
			            GareBDD.TABLE_BOUTIQUE},
                GareBDD.TABLE_CLE + " = ?", new String[] {String.valueOf(id)}, null, null, null);
        if(!c.moveToFirst())
        {
            c.close();
            return null;
        }
        Gare g = new Gare(id, c.getString(0), c.getString(1),
                c.getString(2), c.getString(3),
                c.getDouble(4), c.getDouble(5),
                c.getString(6), c.getInt(7),
                c.getInt(8), c.getInt(9),
                c.getInt(10), c.getString(11),
                c.getLong(12));
        c.close();
        return g;
    }

    public Gare get(String idStif)
    {
        Cursor c = bdd.query(GareBDD.TABLE_NOM,
                new String[] {GareBDD.TABLE_CLE, GareBDD.TABLE_NOM_GARE,
                        GareBDD.TABLE_SUR_TITRE, GareBDD.TABLE_SOUS_TITRE,
                        GareBDD.TABLE_LONGITUDE, GareBDD.TABLE_LATITUDE,
                        GareBDD.TABLE_EXPLOITANT, GareBDD.TABLE_NIVEAU,
                        GareBDD.TABLE_COULEUR, GareBDD.TABLE_COULEUR_EVO,
                        GareBDD.TABLE_NB_VALIDATIONS, GareBDD.TABLE_DERNIERE_VALIDATION,
                        GareBDD.TABLE_BOUTIQUE},
                GareBDD.TABLE_ID_STIF + " = ?", new String[] {String.valueOf(idStif)}, null, null, null);
        if(!c.moveToFirst())
        {
            c.close();
            return null;
        }
        Gare g = new Gare(c.getLong(0), idStif, c.getString(1),
            c.getString(2), c.getString(3),
            c.getDouble(4), c.getDouble(5),
            c.getString(6), c.getInt(7),
            c.getInt(8), c.getInt(9),
            c.getInt(10), c.getString(11),
            c.getLong(12));
        c.close();
        return g;
    }

    public ArrayList<Ligne> getCorrespondances(Gare g)
    {
        Cursor c = bdd.rawQuery("SELECT l.id, l.idStif, l.nom, l." + LigneBDD.TABLE_SOUS_NOM + ", l.type, l.ordre, " +
                "l." + LigneBDD.TABLE_COULEUR + ", l." + LigneBDD.TABLE_VILLE + ", l." + LigneBDD.TABLE_NB_GARES +
                ", r." + RegionBDD.TABLE_CLE + " idRegion, r." + RegionBDD.TABLE_NOM_REGION + " nomRegion " +
                "FROM Ligne l " +
                "INNER JOIN GareDansLigne gdl ON gdl.idLigne = l.id " +
                "INNER JOIN " + RegionBDD.TABLE_NOM + " r ON r." + RegionBDD.TABLE_CLE + " = l." + LigneBDD.TABLE_REGION + " " +
                "WHERE gdl.idGare = ? " +
                "GROUP BY l.nom " +
                "ORDER BY CASE WHEN r." + RegionBDD.TABLE_FAVORI + " = 0 THEN 1 ELSE 0 END ASC, r." + RegionBDD.TABLE_FAVORI + " ASC, r." + RegionBDD.TABLE_ORDRE + " ASC, l." + LigneBDD.TABLE_ORDRE + " ASC, l." + LigneBDD.TABLE_NOM_LIGNE + " ASC;", new String[] {String.valueOf(g.getId())});
        ArrayList<Ligne> listeLigne = new ArrayList<Ligne>();
        while (c.moveToNext()) {
            Region r = new Region(c.getLong(9), c.getString(10));
            Ligne l = new Ligne(
                    c.getLong(0),
                    c.getString(1),
                    c.getString(2),
                    c.getString(3),
                    c.getString(4),
                    c.getInt(5),
                    c.getString(6),
                    c.getString(7),
                    r, c.getInt(8));
            listeLigne.add(l);
        }
        c.close();
        return listeLigne;
    }

    public static ContentValues creerDepuisObj(Gare g)
    {
        ContentValues valeur = new ContentValues();
        valeur.put(GareBDD.TABLE_ID_STIF, g.getIdStif());
        valeur.put(GareBDD.TABLE_NOM_GARE, g.getNom());
        valeur.put(GareBDD.TABLE_SUR_TITRE, g.getSurTitre());
        valeur.put(GareBDD.TABLE_SOUS_TITRE, g.getSousTitre());
        valeur.put(GareBDD.TABLE_LONGITUDE, g.getLongitude());
        valeur.put(GareBDD.TABLE_LATITUDE, g.getLatitude());
        valeur.put(GareBDD.TABLE_EXPLOITANT, g.getExploitant());
        valeur.put(GareBDD.TABLE_NIVEAU, g.getNiveau());
        valeur.put(GareBDD.TABLE_COULEUR, g.getCouleur());
        valeur.put(GareBDD.TABLE_COULEUR_EVO, g.getCouleurEvo());
        valeur.put(GareBDD.TABLE_NB_VALIDATIONS, g.getNbTampons());
        Date derniereValidation = g.getDerniereValidationDate();
        if(derniereValidation == null) {
            valeur.put(GareBDD.TABLE_DERNIERE_VALIDATION, (String) null);
        } else {
            SimpleDateFormat formatDate = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            valeur.put(GareBDD.TABLE_DERNIERE_VALIDATION, formatDate.format(derniereValidation));
        }
        valeur.put(GareBDD.TABLE_BOUTIQUE, g.getIdBoutique());
        return valeur;
    }

    public static ContentValues creerRelationGareLigne(long idGare, long idLigne)
    {
        ContentValues valeur = new ContentValues();
        valeur.put(GareDansLigneBDD.TABLE_ID_GARE, idGare);
        valeur.put(GareDansLigneBDD.TABLE_ID_LIGNE, idLigne);
        return valeur;
    }

    public static void updateAllLevels(Context contexte, SQLiteDatabase bdd)
    {
        Cursor c = bdd.rawQuery("SELECT g." + GareBDD.TABLE_CLE + ", count(t." + TamponBDD.TABLE_CLE + ") AS nbTampon, g." + GareBDD.TABLE_COULEUR + " couleur " +
                "FROM " + GareBDD.TABLE_NOM + " g " +
                "INNER JOIN " + TamponBDD.TABLE_NOM + " t ON g." + GareBDD.TABLE_CLE + " = t." + TamponBDD.TABLE_NOM_GARE + " " +
                "GROUP BY g." + GareBDD.TABLE_CLE + ";", null);
        int[] tickets = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}; //12 couleurs possible
        while (c.moveToNext()) {
            long idGare = c.getLong(0);
            int nbTampons = c.getInt(1);
            int couleur = c.getInt(2);
            if(nbTampons >= 3) //On met à jour le niveau
            {
                ContentValues valeur = new ContentValues();
                valeur.put(GareBDD.TABLE_NIVEAU, 1);
                bdd.update(GareBDD.TABLE_NOM, valeur, GareBDD.TABLE_CLE + " = ?", new String[] {String.valueOf(idGare)});
                tickets[couleur] += nbTampons - 3;
            }
        }
        c.close();

        //Et on crédite les tickets, sans regarder la limite
        for(int couleur = 0; couleur < 12; couleur++)
        {
            ContentValues valeur = new ContentValues();
            valeur.put(InventaireBDD.TABLE_NOMBRE, tickets[couleur]);
            bdd.update(InventaireBDD.TABLE_NOM, valeur, InventaireBDD.TABLE_TYPE + " = " + InventaireBDD.TYPE_MONNAIE + " AND " + InventaireBDD.TABLE_ID_OBJ + " = ?", new String[] {String.valueOf(couleur)});
        }
    }

    public int getNbGaresTamponnees() {
        Cursor c = bdd.rawQuery("SELECT " + GareBDD.TABLE_CLE + " " +
                "FROM " + GareBDD.TABLE_NOM + " g " +
                "WHERE " + GareBDD.TABLE_NB_VALIDATIONS + " > 0;", null);
        if(!c.moveToFirst())
        {
            c.close();
            return 0;
        }
        int nbGares = c.getCount();
        c.close();
        return nbGares;
    }

    public static void updateAllSuccesConcerningGares(Context contexte, SQLiteDatabase bdd)
    {
        //On commence par le nombre de gare tamponnées
        Cursor c = bdd.rawQuery("SELECT " + GareBDD.TABLE_CLE + " " +
                "FROM " + GareBDD.TABLE_NOM + " " +
                "WHERE " + GareBDD.TABLE_NB_VALIDATIONS + " > 0;", null);

        int nbGaresTamponnees;
        if(!c.moveToFirst())
            nbGaresTamponnees = 0;
        else
            nbGaresTamponnees = c.getCount();
        c.close();

        bdd.execSQL("UPDATE " + SuccesBDD.TABLE_NOM + " SET " + SuccesBDD.TABLE_EST_VALIDE + " = ? WHERE " + SuccesBDD.TABLE_TYPE + " = ? AND " + SuccesBDD.TABLE_QTE_NECESSAIRE + " <= ?",
                new String[] {String.valueOf(SuccesManager.EstValide), String.valueOf(SuccesManager.TypeGare), String.valueOf(nbGaresTamponnees)});

        //Et on fait pareil pour le nombre de validation dans une gare
        int nbMaxTampon;
        c = bdd.rawQuery("SELECT MAX(" + GareBDD.TABLE_NB_VALIDATIONS + ") AS nbTampon " +
                "FROM " + GareBDD.TABLE_NOM + " g " +
                "WHERE " + GareBDD.TABLE_NB_VALIDATIONS + " > 0;", null);
        if(!c.moveToFirst())
            nbMaxTampon = 0;
        else
            nbMaxTampon = c.getInt(0);
        c.close();
        bdd.execSQL("UPDATE " + SuccesBDD.TABLE_NOM + " SET " + SuccesBDD.TABLE_EST_VALIDE + " = ? WHERE " + SuccesBDD.TABLE_TYPE + " = ? AND " + SuccesBDD.TABLE_QTE_NECESSAIRE + " <= ?",
                new String[] {String.valueOf(SuccesManager.EstValide), String.valueOf(SuccesManager.TypeValidation), String.valueOf(nbMaxTampon)});

        // Warning : Ne pas fermer la connexion à la BDD !
    }

    public static void fixIssueGaresHorsIdF(Context contexte, SQLiteDatabase bdd) {
        //Centre Val de Loire
        boolean centreVdLEstInstalle = isRegionEstInstalle(bdd, "CentreVdL");

        if(!centreVdLEstInstalle)
        {
            //On supprime les gares qui n'ont rien à faire en IdF seulement si on n'a pas Centre Val de Loire installé
            bdd.execSQL("DELETE FROM " + GareBDD.TABLE_NOM +
                    " WHERE " + GareBDD.TABLE_ID_STIF + " IN ('411498', '411504', '411507', '59302', '411501', '59049', '411449', '411446', '411443')");
        }

        // Picardie
        boolean picardieEstInstalle = isRegionEstInstalle(bdd, "HautsFrance");

        if(!picardieEstInstalle)
        {
            //On supprime les gares qui n'ont rien à faire en IdF seulement si on n'a pas Hauts de France installé
            bdd.execSQL("DELETE FROM " + GareBDD.TABLE_NOM +
                    " WHERE " + GareBDD.TABLE_ID_STIF + " IN ('411317')");
        }

        //Normandie
        boolean normandieEstInstalle = isRegionEstInstalle(bdd, "Normandie");

        if(!normandieEstInstalle)
        {
            //On supprime les gares qui n'ont rien à faire en IdF seulement si région pas installé
            bdd.execSQL("DELETE FROM " + GareBDD.TABLE_NOM +
                    " WHERE " + GareBDD.TABLE_ID_STIF + " IN ('411365', '411362')");
        }

        //Bourgogne Franche Comté
        boolean bourgogneFCEstInstalle = isRegionEstInstalle(bdd, "BourgogneFC");

        if(!bourgogneFCEstInstalle)
        {
            //On supprime les gares qui n'ont rien à faire en IdF seulement si région pas installé
            bdd.execSQL("DELETE FROM " + GareBDD.TABLE_NOM +
                    " WHERE " + GareBDD.TABLE_ID_STIF + " IN ('411461', '411476', '411479', '411473', '411464', '411452', '411458', '411467', '411455')");
        }

        //On doit gérer aussi le doublon de la gare de Vernon.
        if(normandieEstInstalle) //Le problème n'apparaît que si on a Normandie…
        {
            boolean idfEstInstalle = isRegionEstInstalle(bdd, "Paris");

            if(idfEstInstalle) //… et l'Île-de-France !
            {
                //On supprime juste celle de Normandie
                bdd.execSQL("DELETE FROM " + GareBDD.TABLE_NOM + " WHERE " + GareBDD.TABLE_ID_STIF + " = 'SNCF01838'");
            } else { //On n'a juste à changer l'id de la gare de normandie
                bdd.execSQL("UPDATE " + GareBDD.TABLE_NOM + " SET " + GareBDD.TABLE_ID_STIF + " = '59038' WHERE " + GareBDD.TABLE_ID_STIF + " = 'SNCF01838'");
            }
        }

        // Warning : Ne pas fermer la connexion à la BDD !
    }

    private static Date maxDate(Date d1, Date d2) {
        if (d1 == null && d2 == null) return null;
        if (d1 == null) return d2;
        if (d2 == null) return d1;
        return (d1.after(d2)) ? d1 : d2;
    }

    /**
     * Utilitaire pour fusionner des gares
     * @param bdd Connexion à la base de données
     * @param listeGaresACorriger Liste des gares à corriger, sous forme de tableau où le premier élément
     *                            est la gare qui restera à la fin
     */
    private static void fusionnerGares(SQLiteDatabase bdd, ArrayList<String[]> listeGaresACorriger)
    {
        GareCtrl gareCtrl = new GareCtrl(bdd);
        for (String[] gareACorrige : listeGaresACorriger) {
            String bonId = gareACorrige[0];
            String[] listeMauvaisesId = Arrays.copyOfRange(gareACorrige, 1, gareACorrige.length);
            for(String mauvaisId : listeMauvaisesId)
            {
                Gare gare = gareCtrl.get(bonId);
                if (gare == null) {
                    // La gare n'existe pas, on va juste changer l'id de l'autre
                    bdd.execSQL("UPDATE " + GareBDD.TABLE_NOM + " SET " + GareBDD.TABLE_ID_STIF + " = ? WHERE " + GareBDD.TABLE_ID_STIF + " = ?", new Object[]{bonId, mauvaisId});
                } else {
                    Gare garePourrie = gareCtrl.get(mauvaisId);
                    if(garePourrie == null)
                    {
                        // La gare pourrie n'existe pas, on a donc besoin de rien faire
                        continue;
                    }
                    // La gare existe, on doit fusionner et déplacer les tampons
                    bdd.execSQL("UPDATE " + TamponBDD.TABLE_NOM + " SET " + TamponBDD.TABLE_NOM_GARE + " = ? WHERE " + TamponBDD.TABLE_NOM_GARE + " = ?", new Object[]{gare.getId(), garePourrie.getId()});

                    ArrayList<String> listeInfoMettreAJour = new ArrayList<String>();
                    if (gare.getDerniereValidationDate() != garePourrie.getDerniereValidationDate()) {
                        SimpleDateFormat formatDate = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        listeInfoMettreAJour.add(GareBDD.TABLE_DERNIERE_VALIDATION + " = \"" + formatDate.format(maxDate(gare.getDerniereValidationDate(), garePourrie.getDerniereValidationDate())) + "\"");
                    }

                    if (garePourrie.getNbTampons() > 0) {
                        listeInfoMettreAJour.add(GareBDD.TABLE_NB_VALIDATIONS + " = " + (Math.max(gare.getNbTampons(), garePourrie.getNbTampons())));
                    }

                    if (garePourrie.getNiveau() != gare.getNiveau()) {
                        listeInfoMettreAJour.add(GareBDD.TABLE_NIVEAU + " = " + Math.max(gare.getNiveau(), garePourrie.getNiveau()));
                    }

                    if (garePourrie.getIdBoutique() != 0) {
                        if (gare.getIdBoutique() != 0) {
                            // Deux boutiques, on supprime celle de la gare pourrie
                            bdd.execSQL("DELETE FROM " + BoutiqueBDD.TABLE_NOM + " WHERE " + BoutiqueBDD.TABLE_CLE + " = ?", new Object[]{garePourrie.getIdBoutique()});
                        } else {
                            listeInfoMettreAJour.add(GareBDD.TABLE_BOUTIQUE + " = " + garePourrie.getIdBoutique());
                        }
                    }

                    if (!listeInfoMettreAJour.isEmpty()) {
                        bdd.execSQL("UPDATE " + GareBDD.TABLE_NOM + " SET " + String.join(", ", listeInfoMettreAJour) + " WHERE " + GareBDD.TABLE_CLE + " = ?", new Object[]{gare.getId()});
                    }
                    // Plus qu'à supprimer
                    bdd.execSQL("DELETE FROM " + GareBDD.TABLE_NOM + " WHERE " + GareBDD.TABLE_CLE + " = ?", new Object[]{garePourrie.getId()});
                    bdd.execSQL("DELETE FROM " + CoordonneeBDD.TABLE_NOM + " WHERE " + CoordonneeBDD.TABLE_ID_GARE + " = ?", new Object[]{garePourrie.getId()});
                }
            }
        }
    }

    public static void fixIssueGaresDoublons191(Context contexte, SQLiteDatabase bdd)
    {
        //Normandie
        boolean normandieEstInstalle = isRegionEstInstalle(bdd, "Normandie");

        if (normandieEstInstalle) //Le problème n'apparaît que si on a Normandie…
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"43157", "SNCF43157"});
            listeGaresACorriger.add(new String[]{"43100", "SNCF43100"});
            listeGaresACorriger.add(new String[]{"43219", "SNCF43219"});
            listeGaresACorriger.add(new String[]{"411492", "SNCF411492"});
            listeGaresACorriger.add(new String[]{"58566", "SNCF58566"});
            listeGaresACorriger.add(new String[]{"58578", "SNCF58578"});
            listeGaresACorriger.add(new String[]{"46984", "SNCF46984"});
            listeGaresACorriger.add(new String[]{"46983", "SNCF479059", "479059"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }

        // Nouvelle Aquitaine
        boolean nvelleAquitaineEstInstalle = isRegionEstInstalle(bdd, "NAquitaine");

        if(nvelleAquitaineEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"SNCF01303", "TBMTSJEAN"});
            listeGaresACorriger.add(new String[]{"SNCF02446", "TBMTARLAC"});
            listeGaresACorriger.add(new String[]{"SNCF01288", "TBMTPESSA"});
            listeGaresACorriger.add(new String[]{"SNCF01287", "TBMTPESGAR"});
            listeGaresACorriger.add(new String[]{"SNCF01286", "TBMTDORGAT"});
            listeGaresACorriger.add(new String[]{"SNCF01292", "TBMTCENOG"});
            listeGaresACorriger.add(new String[]{"SNCF87738096", "TBMTGERMA"});
            listeGaresACorriger.add(new String[]{"SNCF01289", "TBMTBRUGAR"});
            listeGaresACorriger.add(new String[]{"SNCF01290", "TBMTGABLAN"});
            listeGaresACorriger.add(new String[]{"TBMTHDVA", "TBMTHDVB"});
            listeGaresACorriger.add(new String[]{"TBMTAUBI1C", "TBMTAUBI2C"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }

        // Centre Val-de-Loire
        boolean centreVdLEstInstalle = isRegionEstInstalle(bdd, "CentreVdL");

        if (normandieEstInstalle || centreVdLEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"43238", "58578"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }

        if (centreVdLEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"43072", "462352"});
            listeGaresACorriger.add(new String[]{"43185", "427870"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }

        // Pays-de-la-Loire
        boolean paysLoireEstInstalle = isRegionEstInstalle(bdd, "PaysLoire");
        if (paysLoireEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"IRIGO2ACAS", "IRIGOzone_ACACIAS"});
            listeGaresACorriger.add(new String[]{"IRIGO2JMOU", "IRIGOzone_JEAN_MOULIN"});
            listeGaresACorriger.add(new String[]{"IRIGO1MOLIE", "IRIGOzone_MOLIERE"});
            listeGaresACorriger.add(new String[]{"IRIGO2RALL", "IRIGOzone_RALLIEMENT"});
            listeGaresACorriger.add(new String[]{"IRIGO1ANRO", "IRIGOzone_ANGERS_-_ROSERAIE"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }

        // PACA
        boolean pacaEstInstalle = isRegionEstInstalle(bdd, "PACA");
        if (pacaEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"LignesAzurplace_WSTIS1", "LignesAzur0699100"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }

        // España
        boolean espanaEstInstalle = isRegionEstInstalle(bdd, "Espana");
        if (espanaEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"Renfe97007", "Renfe97008", "Renfe97014"});
            listeGaresACorriger.add(new String[]{"Renfe65409", "Renfe65422"});
            listeGaresACorriger.add(new String[]{"Renfe60901", "Renfe55555"});
            listeGaresACorriger.add(new String[]{"Renfe10203", "Renfe99207"});
            listeGaresACorriger.add(new String[]{"Renfe15402", "Renfe15403", "Renfe15410"});
            listeGaresACorriger.add(new String[]{"Renfe79300", "Renfe79206"});
            listeGaresACorriger.add(new String[]{"Renfe65420", "Renfe65407"});
            listeGaresACorriger.add(new String[]{"Renfe16408", "Renfe16406"});
            listeGaresACorriger.add(new String[]{"Renfe18000", "Renfe98000"});
            listeGaresACorriger.add(new String[]{"Renfe17000", "Renfe99140"});
            listeGaresACorriger.add(new String[]{"Renfe67113", "Renfe67102"});
            listeGaresACorriger.add(new String[]{"Renfe60103", "Renfe99203"});
            listeGaresACorriger.add(new String[]{"Renfe97000", "Renfe97010"});
            listeGaresACorriger.add(new String[]{"Renfe70001", "Renfe70005"});
            listeGaresACorriger.add(new String[]{"Renfe70100", "Renfe99204"});
            listeGaresACorriger.add(new String[]{"Renfe14215", "Renfe14229"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }

        // Bourgogne
        boolean bourgogneEstInstalle = isRegionEstInstalle(bdd, "BourgogneFC");
        if (bourgogneEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"58728", "479035"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }
    }

    private static boolean isRegionEstInstalle(SQLiteDatabase bdd, String nomRegion) {
        Cursor c = bdd.rawQuery("SELECT " + RegionBDD.TABLE_EST_INSTALLE +
                " FROM " + RegionBDD.TABLE_NOM + " WHERE " + RegionBDD.TABLE_DOSSIER + " = '" + nomRegion + "'", null);

        boolean estInstalle = false;
        if (c.getCount() > 0 && c.moveToFirst())
            estInstalle = c.getInt(0) == 1;
        c.close();
        return estInstalle;
    }

    public static void fixIssueGaresLondres192(Context contexte, SQLiteDatabase bdd) {
        //Londres
        boolean londresEstInstalle = isRegionEstInstalle(bdd, "London");

        if (londresEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"Tfl1000071", "Tfl1000072"});
            listeGaresACorriger.add(new String[]{"Tfl1000097", "Tfl1000096"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }
    }

    public static void fixIssueGaresEspagne195(Context contexte, SQLiteDatabase bdd) {
        // Espagne
        boolean espagneEstInstalle = isRegionEstInstalle(bdd, "Espana");

        if (espagneEstInstalle)
        {
            ArrayList<String[]> listeGaresACorriger = new ArrayList<>();
            listeGaresACorriger.add(new String[]{"Renfe07007", "Renfe07005"});
            listeGaresACorriger.add(new String[]{"Renfe78805", "Renfe97009"});
            listeGaresACorriger.add(new String[]{"Renfe71801", "Renfe97007"});
            listeGaresACorriger.add(new String[]{"Renfe78806", "Renfe79002"});
            listeGaresACorriger.add(new String[]{"Renfe18003", "Renfe35701"});
            listeGaresACorriger.add(new String[]{"Renfe18000", "Renfe99000"});

            GareCtrl.fusionnerGares(bdd, listeGaresACorriger);
        }
    }

    public ArrayList<GareTamponnee> getGareTamponneeByLine(long idLigne, boolean voirTamponDuJour) {
        boolean cacherPlanLigne = voirTamponDuJour || idLigne == 0;

        String[] conditionsWhere = new String[] {};
        String finWhere = "";
        ArrayList<String> where = new ArrayList<>();
        ArrayList<String> criteresOrdre = new ArrayList<>();
        if(idLigne != 0)
        {
            conditionsWhere = new String[] {String.valueOf(idLigne)};
            where.add("gdl." + GareDansLigneBDD.TABLE_ID_LIGNE + " = ? ");
            criteresOrdre.add("gdl." + GareDansLigneBDD.TABLE_ORDRE + " ASC");
        }

        if(voirTamponDuJour)
            where.add("g." + GareBDD.TABLE_DERNIERE_VALIDATION + " >= date('now')");

        if(where.size() > 0)
            finWhere = "WHERE " + TextUtils.join(" AND ", where);

        // Quoi qu’il arrive, on finit par trier par nom de gare
        criteresOrdre.add("g." + GareBDD.TABLE_NOM_GARE + " COLLATE LOCALIZED ASC");
        String ordre = "ORDER BY " + TextUtils.join(", ", criteresOrdre);

        Cursor c = bdd.rawQuery("SELECT g." + GareBDD.TABLE_CLE + ", g." + GareBDD.TABLE_NOM_GARE + ", " +
                "g." + GareBDD.TABLE_NB_VALIDATIONS + ", " + GareBDD.TABLE_DERNIERE_VALIDATION + ", " +
                GareBDD.TABLE_COULEUR  + ", " + GareBDD.TABLE_COULEUR_EVO + ", " + GareBDD.TABLE_NIVEAU + ", " +
                "gdl."+ GareDansLigneBDD.TABLE_PLANDELIGNE_FOND + ", gdl." + GareDansLigneBDD.TABLE_PLANDELIGNE_POINT + ", " +
                "gdl." + GareDansLigneBDD.TABLE_NOMGARE + " " +
                "FROM " + GareDansLigneBDD.TABLE_NOM + " gdl " +
                "INNER JOIN " + GareBDD.TABLE_NOM + " g ON g." + GareBDD.TABLE_CLE + " = gdl." + GareDansLigneBDD.TABLE_ID_GARE + " " +
                finWhere +
                "GROUP BY gdl." + GareDansLigneBDD.TABLE_ID_GARE + " " +
                ordre + ";", conditionsWhere);
        ArrayList<GareTamponnee> listeTampons = new ArrayList<>();
        while (c.moveToNext()) {
            String nomGare = c.getString(1);

            // Si on a un nom dans GareDansLigne, on l’utilise
            if(c.getString(9) != null && !c.getString(9).isEmpty())
            {
                nomGare = c.getString(9);
            }

            GareTamponnee t = new GareTamponnee(
                    c.getLong(0),
                    nomGare,
                    c.getInt(2),
                    c.getString(3),
                    cacherPlanLigne ? 0 : c.getInt(7),
                    cacherPlanLigne ? 0 : c.getInt(8),
                    c.getInt(6),
                    c.getInt(4),
                    c.getInt(5));
            listeTampons.add(t);
        }
        c.close();
        return listeTampons;
    }

    public int getMaxTamponInGare()
    {
        Cursor c = bdd.rawQuery("SELECT MAX(" + GareBDD.TABLE_NB_VALIDATIONS + ") AS nbTampon " +
                "FROM " + GareBDD.TABLE_NOM + " g " +
                "WHERE " + GareBDD.TABLE_NB_VALIDATIONS + " > 0;", null);
        int nbMaxTampon;
        if(!c.moveToFirst())
            nbMaxTampon = 0;
        else
            nbMaxTampon = c.getInt(0);
        c.close();
        return nbMaxTampon;
    }

    /**
     * Supprimer les gares orphelines
     */
    public void deleteOrphan()
    {
        bdd.delete(GareBDD.TABLE_NOM, "NOT EXISTS (SELECT * FROM " + GareDansLigneBDD.TABLE_NOM + " gdl WHERE gdl."+GareDansLigneBDD.TABLE_ID_GARE+" = "+ GareBDD.TABLE_NOM+"."+GareBDD.TABLE_CLE+" )", null);
    }
}
