package fr.jnda.ipcalc

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.net.ConnectivityManager
import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.View
import android.view.inputmethod.InputMethodManager
import android.widget.TextView
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate
import androidx.preference.PreferenceManager
import com.google.android.material.bottomnavigation.BottomNavigationView
import com.google.android.material.textfield.TextInputEditText
import com.google.android.material.textfield.TextInputLayout
import fr.jnda.ipcalc.databinding.ActivityV2Binding
import fr.jnda.ipcalc.helper.IPV4_PATTERN
import fr.jnda.ipcalc.helper.IPV6_STD_PATTERN
import fr.jnda.ipcalc.ui.*
import fr.jnda.ipcalc.utils.IPv4Helper
import fr.jnda.ipcalc.utils.IPv6Helper
import java.net.Inet4Address


class MainActivity : AppCompatActivity() {

    private lateinit var binding: ActivityV2Binding
    private lateinit var subnetMask: TextInputEditText
    private lateinit var ipAddress: TextView
    private lateinit var subnetLayout: TextInputLayout

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val isNightMode =
            PreferenceManager.getDefaultSharedPreferences(this).getBoolean("pref_key_dark_style", false)
        binding = ActivityV2Binding.inflate(layoutInflater)
        subnetMask = binding.idSubnetMask
        ipAddress = binding.idIpAddress
        subnetLayout = binding.subnetLayout

        delegate.localNightMode =
            if (isNightMode) AppCompatDelegate.MODE_NIGHT_YES else AppCompatDelegate.MODE_NIGHT_NO

        val view = binding.root
        setContentView(view)

        val navView: BottomNavigationView = binding.navView

        navView.setOnItemSelectedListener {
            when (it.itemId) {
                R.id.navigation_info -> {
                    supportFragmentManager.beginTransaction().replace(R.id.id_main_content, MainInfo())
                        .commitNow()
                }
                R.id.navigation_binaries -> {
                    supportFragmentManager.beginTransaction().replace(R.id.id_main_content, BinariesInfo())
                        .commitNow()
                }
                R.id.navigation_location -> {
                    supportFragmentManager.beginTransaction().replace(R.id.id_main_content, LocationInfo())
                        .commitNow()
                }
                R.id.navigation_settings -> {
                    supportFragmentManager.beginTransaction().replace(R.id.id_main_content, AppInfo())
                        .commitNow()
                }
            }
            hideKeyboard(this@MainActivity)
            displayInfo()

            true
        }

        ipAddress.addTextChangedListener(object : TextWatcher {
            override fun afterTextChanged(s: Editable?) {
                displayInfo()
            }

            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {}

            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {}

        })
        subnetMask.addTextChangedListener(object : TextWatcher {
            override fun afterTextChanged(s: Editable?) {
                displayInfo()
            }

            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {}

            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {}

        })
        supportFragmentManager.beginTransaction().replace(R.id.id_main_content, MainInfo()).commitNow()
        setInitialIpAddress()
        displayInfo()

    }

    private fun setInitialIpAddress() {
        val (ip, prefix) = getDeviceIpAndPrefix()
        if (ip != null && prefix != null) {
            ipAddress.text = ip
            subnetMask.setText(prefix.toString())
        }
    }

    @SuppressLint("MissingPermission")
    private fun getDeviceIpAndPrefix(): Pair<String?, Int?> {
        val connectivityManager = getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
        val network = connectivityManager.activeNetwork ?: return Pair(null, null)
        val linkProperties = connectivityManager.getLinkProperties(network) ?: return Pair(null, null)

        for (linkAddress in linkProperties.linkAddresses) {
            val address = linkAddress.address
            if (address is Inet4Address && !address.isLoopbackAddress) {
                return Pair(address.hostAddress, linkAddress.prefixLength)
            }
        }
        // Fallback to IPv6 if no suitable IPv4 address is found
        for (linkAddress in linkProperties.linkAddresses) {
            val address = linkAddress.address
            if (!address.isLoopbackAddress && !address.isLinkLocalAddress) {
                return Pair(address.hostAddress, linkAddress.prefixLength)
            }
        }
        return Pair(null, null)
    }


    private fun displayInfo() {
        val address = ipAddress.text.toString()
        val prefix = subnetMask.text.toString().toIntOrNull() ?: 0
        val fragment = supportFragmentManager.findFragmentById(R.id.id_main_content)

        try {
            when {
                IPV4_PATTERN.matcher(address).matches() -> {
                    val ipv4Helper = IPv4Helper(address, prefix)
                    subnetLayout.helperText = String.format(
                        getString(R.string.lbl_subnet),
                        ipv4Helper.getNetmask(),
                        prefix.toString()
                    )
                    subnetMask.isEnabled = true

                    fragment?.let {
                        if (fragment is BaseFragment)
                            fragment.displayInfo(ipv4Helper, address)
                    }
                }
                IPV6_STD_PATTERN.matcher(address).matches() -> {
                    val ipv6Helper = IPv6Helper(address, prefix)
                    subnetLayout.helperText = String.format(getString(R.string.lbl_subnet), "/${prefix}", "")
                    subnetMask.isEnabled = true
                    fragment?.let {
                        if (fragment is BaseFragment) {
                            fragment.displayInfo(ipv6Helper, address)
                        }
                    }
                }
                else -> {
                    subnetLayout.helperText = String.format(getString(R.string.lbl_subnet), "", "")
                    subnetMask.isEnabled = false
                    fragment?.let {
                        if (fragment is BaseFragment)
                            fragment.clearInfo()
                    }
                }
            }
        } catch (e: Exception) {
            subnetLayout.helperText = String.format(getString(R.string.lbl_subnet), "", "")
            subnetMask.isEnabled = false
            fragment?.let {
                if (fragment is BaseFragment)
                    fragment.clearInfo()
            }
        }
    }

    private fun hideKeyboard(activity: Activity) {
        val imm = activity.getSystemService(Activity.INPUT_METHOD_SERVICE) as InputMethodManager
        val view = activity.currentFocus ?: View(activity)
        imm.hideSoftInputFromWindow(view.windowToken, 0)
    }
}