"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.packageJsonLicense = void 0;
const tslib_1 = require("tslib");
const path_1 = require("path");
const console_utils_1 = tslib_1.__importDefault(require("../../utils/console.utils"));
const file_utils_1 = require("../../utils/file.utils");
// This file specifically handles cases where the package.json links
// to a license file that is on disk and is a part of the package.
//
// If it instead finds a URL to a license file, it will return that URL as-is.
const packageJsonLicense = (inputs) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const { packageJson, directory } = inputs;
    const { license, licenses } = packageJson;
    if (typeof license === "string") {
        return parseStringLicense(license, directory);
    }
    if (Array.isArray(license)) {
        return parseArrayLicense(license, packageJson);
    }
    if (typeof license === "object") {
        return parseObjectLicense(license);
    }
    if (Array.isArray(licenses)) {
        return parseArrayLicense(licenses, packageJson);
    }
    return null;
});
exports.packageJsonLicense = packageJsonLicense;
const parseStringLicense = (spdxExpression, directory) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    if (!spdxExpression) {
        return null;
    }
    const lowerCaseExpression = spdxExpression.toLowerCase();
    if (lowerCaseExpression.startsWith("http") || lowerCaseExpression.startsWith("www")) {
        return spdxExpression;
    }
    if (!lowerCaseExpression.startsWith("see license in ")) {
        return null;
    }
    const licenseFilePath = sanitise(spdxExpression.substring(15));
    if (licenseFilePath.startsWith("http") || licenseFilePath.startsWith("www")) {
        return spdxExpression;
    }
    return yield readLicenseFromDisk(directory, licenseFilePath);
});
// Removes characters that we've witnessed people use in license file paths
const sanitise = (str) => str.replace(/['"<>]/g, "");
const readLicenseFromDisk = (dir, path) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const absolutePath = (0, path_1.join)(dir, path);
    const fileExists = yield (0, file_utils_1.doesFileExist)(absolutePath);
    if (!fileExists) {
        console_utils_1.default.warn(`Could not find license file '${absolutePath}'`);
        return null;
    }
    try {
        return yield (0, file_utils_1.readFile)(absolutePath, { encoding: "utf-8" });
    }
    catch (e) {
        console_utils_1.default.warn(`Could not read license file '${absolutePath}'`);
        return null;
    }
});
const parseArrayLicense = (license, packageJson) => {
    if (license.length === 0) {
        return null;
    }
    if (license.length === 1) {
        return parseObjectLicense(license[0]);
    }
    const warningLines = [
        `The license key for ${packageJson.name}@${packageJson.version} contains multiple licenses`,
        "We suggest you determine which license applies to your project and replace the license content",
        `for ${packageJson.name}@${packageJson.version} using a generate-license-file config file.`,
        "See: https://generate-license-file.js.org/docs/cli/config-file for more information.",
        "", // Empty line for spacing
    ];
    console_utils_1.default.warn(warningLines.join("\n"));
    return parseObjectLicense(license[0]);
};
const parseObjectLicense = (license) => {
    if (!license.url) {
        return null;
    }
    return license.url;
};
//# sourceMappingURL=packageJsonLicense.js.map