"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mainCommand = void 0;
const tslib_1 = require("tslib");
const extra_typings_1 = require("@commander-js/extra-typings");
const path_1 = require("path");
const generateLicenseFile_1 = require("../../generateLicenseFile");
const console_utils_1 = tslib_1.__importDefault(require("../../utils/console.utils"));
const packageJson_utils_1 = require("../../utils/packageJson.utils");
const eol_1 = require("../args/eol");
const inputs_1 = require("../args/inputs");
const output_1 = require("../args/output");
const config_1 = require("../config");
const spinner_1 = require("../spinner");
exports.mainCommand = new extra_typings_1.Command()
    .name("generate-license-file")
    .description("Generates a text file containing all of the licenses for your production dependencies")
    .helpOption("-h,--help", "Display help text")
    .option("-c,--config <path>", "Specify a path to a generate-license-file config file. Files will be automatically detected if this flag is not given")
    .option("-i,--input <paths...>", "Specify one or more paths to package.json files to include")
    .option("-o,--output <path>", "Specify the path of the file to be created")
    .option("--overwrite", "Indicates that the output file should be overwritten if it already exists")
    .option("--eol <eol>", "Specify a particular line ending to use in the output file. Otherwise, the line ending of the current OS will be used")
    .option("--ci", "Fail with a non-zero exit code if user input is required")
    .option("--no-spinner", "Don't show the progress spinner while generating the license file. This is implicitly true if --ci is given")
    .option("--omit-versions", "Omit the package version numbers from the output.")
    .option("-v,--version", "Prints the installed version of generate-license-file")
    .action((givenArgs) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    if (givenArgs.version) {
        yield printPackageVersion();
        return;
    }
    const cliArgs = {
        // Commander sets spinner to true by default, meaning it will always override
        // a spinner value in the config file - set to undefined if it is true (ie,
        // omitted from CLI args) so the filtering below removes it.
        spinner: givenArgs.spinner ? undefined : false,
        ci: givenArgs.ci,
        eol: givenArgs.eol,
        inputs: givenArgs.input,
        output: givenArgs.output,
        overwrite: givenArgs.overwrite,
        omitVersions: givenArgs.omitVersions,
    };
    const configFile = yield (0, config_1.loadConfigFile)(givenArgs.config);
    // Filter out undefined values in the CLI args, so they do not
    // potentially override values provided in the config file.
    const filteredCliArgs = Object.fromEntries(Object.entries(cliArgs).filter(([, v]) => v !== undefined));
    const combinedConfig = Object.assign(Object.assign({}, configFile), filteredCliArgs);
    const { inputs, showSpinner, output, eol, omitVersions } = yield parseArgumentsIntoOptions(combinedConfig);
    if (showSpinner) {
        spinner_1.spinner.start();
    }
    yield (0, generateLicenseFile_1.generateLicenseFile)(inputs, output, {
        lineEnding: eol,
        replace: configFile === null || configFile === void 0 ? void 0 : configFile.replace,
        exclude: configFile === null || configFile === void 0 ? void 0 : configFile.exclude,
        append: configFile === null || configFile === void 0 ? void 0 : configFile.append,
        omitVersions,
    });
    spinner_1.spinner.stop();
}));
function parseArgumentsIntoOptions(config) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (config.ci) {
            config.spinner = false;
            try {
                return yield getOptionsOrThrow(config);
            }
            catch (e) {
                const errorMessage = getMessageFromCaughtUnknown(e);
                throw new Error(`Error parsing arguments in --ci mode: ${errorMessage}`);
            }
        }
        return yield promptForMissingOptions(config);
    });
}
function getOptionsOrThrow(config) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        var _a, _b;
        const inputs = yield new inputs_1.Inputs().parse(config);
        const output = yield new output_1.Output().parse(config);
        const eol = yield new eol_1.Eol().parse(config);
        const showSpinner = (_a = config.spinner) !== null && _a !== void 0 ? _a : true;
        const omitVersions = (_b = config.omitVersions) !== null && _b !== void 0 ? _b : false;
        return { inputs, output, eol, showSpinner, omitVersions };
    });
}
function promptForMissingOptions(config) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        var _a, _b;
        const inputs = yield new inputs_1.Inputs().resolve(config);
        const output = yield new output_1.Output().resolve(config);
        const eol = yield new eol_1.Eol().resolve(config);
        const showSpinner = (_a = config.spinner) !== null && _a !== void 0 ? _a : true;
        const omitVersions = (_b = config.omitVersions) !== null && _b !== void 0 ? _b : false;
        return { inputs, output, eol, showSpinner, omitVersions };
    });
}
function getMessageFromCaughtUnknown(e) {
    if (e instanceof Error) {
        return e.message;
    }
    if (typeof e === "string") {
        return e;
    }
    return "Unknown error";
}
const printPackageVersion = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const packageJsonLocation = (0, path_1.join)(__dirname, "../../../../package.json");
    const { version } = yield (0, packageJson_utils_1.readPackageJson)(packageJsonLocation);
    console_utils_1.default.log(`v${version}`);
});
//# sourceMappingURL=main.js.map