"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inputs = void 0;
const tslib_1 = require("tslib");
const file_utils_1 = require("../../utils/file.utils");
const spinner_1 = require("../spinner");
const argument_1 = require("./argument");
class Inputs extends argument_1.Argument {
    constructor() {
        super(...arguments);
        this.question = "Package.json location: ";
    }
    resolve(config) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { inputs } = config;
            if (!inputs) {
                return yield this.resolveOne(undefined);
            }
            if (inputs.length === 1) {
                return yield this.resolveOne(inputs[0]);
            }
            return yield this.resolveMany(inputs);
        });
    }
    parse(config) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { inputs } = config;
            if (!inputs || inputs.length === 0) {
                throw new Error("No --input argument given.");
            }
            let allValid = true;
            for (const input of inputs) {
                const inputExists = yield (0, file_utils_1.doesFileExist)(input);
                if (!inputExists) {
                    spinner_1.spinner.warn(`${input} could not be found.`);
                    allValid = false;
                }
            }
            if (!allValid && inputs.length === 1) {
                throw new Error("Given --input file not found");
            }
            if (!allValid) {
                throw new Error("One or more given --input files not found");
            }
            return inputs;
        });
    }
    resolveOne(input) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const initialValue = yield this.getInputPromptInitialValue();
            let inputExists = input ? yield (0, file_utils_1.doesFileExist)(input) : false;
            while (!input || !inputExists) {
                if (!!input && !inputExists) {
                    spinner_1.spinner.fail("Given --input file not found");
                }
                input = yield this.promptForString(this.question, initialValue);
                inputExists = yield (0, file_utils_1.doesFileExist)(input);
            }
            return [input];
        });
    }
    resolveMany(inputs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const validInputs = [];
            let allValid = true;
            for (const input of inputs) {
                const inputExists = yield (0, file_utils_1.doesFileExist)(input);
                if (!inputExists) {
                    spinner_1.spinner.warn(`${input} could not be found.`);
                    allValid = false;
                    continue;
                }
                validInputs.push(input);
            }
            if (!allValid) {
                yield this.promptForTermination();
            }
            return validInputs;
        });
    }
    getInputPromptInitialValue() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const packageJsonExists = yield (0, file_utils_1.doesFileExist)("./package.json");
            return packageJsonExists ? "./package.json" : "";
        });
    }
    promptForTermination() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const shouldContinue = yield this.promptForBoolean("One or more given --input files not found. Do you want to continue?");
            if (!shouldContinue) {
                throw new Error("Process terminated by user");
            }
        });
    }
}
exports.Inputs = Inputs;
//# sourceMappingURL=inputs.js.map