"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.spdxExpression = void 0;
const tslib_1 = require("tslib");
const console_utils_1 = tslib_1.__importDefault(require("../../utils/console.utils"));
// This file specifically handles cases where the package.json contains an SPDX license expression.
const spdxExpression = (input) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const { packageJson } = input;
    const { license, licenses } = packageJson;
    if (typeof license === "string") {
        return handleStringLicense(license, packageJson);
    }
    if (Array.isArray(license)) {
        return handleArrayLicense(license, packageJson);
    }
    if (typeof license === "object") {
        return handleObjectLicense(license, packageJson);
    }
    if (Array.isArray(licenses)) {
        return handleArrayLicense(licenses, packageJson);
    }
    return null;
});
exports.spdxExpression = spdxExpression;
const handleArrayLicense = (licenses, packageJson) => {
    if (licenses.length === 0) {
        return null;
    }
    if (licenses.length === 1) {
        return handleObjectLicense(licenses[0], packageJson);
    }
    const warningLines = [
        `The license field for ${packageJson.name}@${packageJson.version} contains multiple licenses:`,
        JSON.stringify(licenses),
        "We suggest you determine which license applies to your project and replace the license content",
        `for ${packageJson.name}@${packageJson.version} using a generate-license-file config file.`,
        "See: https://generate-license-file.js.org/docs/cli/config-file for more information.",
        "", // Empty line for spacing
    ];
    console_utils_1.default.warn(warningLines.join("\n"));
    return handleObjectLicense(licenses[0], packageJson);
};
const handleObjectLicense = (packageJsonLicence, packageJson) => {
    if (!packageJsonLicence.type) {
        return null;
    }
    return handleStringLicense(packageJsonLicence.type, packageJson);
};
const handleStringLicense = (expression, packageJson) => {
    if (expression.length === 0) {
        return null;
    }
    const containsOr = expression.includes(" OR ");
    if (containsOr) {
        const warningLines = [
            `The license expression for ${packageJson.name}@${packageJson.version} contains multiple licenses: "${expression}"`,
            "We suggest you determine which license applies to your project and replace the license content",
            `for ${packageJson.name}@${packageJson.version} using a generate-license-file config file.`,
            "See: https://generate-license-file.js.org/docs/cli/config-file for more information.",
            "", // Empty line for spacing
        ];
        console_utils_1.default.warn(warningLines.join("\n"));
    }
    return expression;
};
//# sourceMappingURL=spdxExpression.js.map