"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.licenseFile = exports.extensionDenyList = void 0;
const tslib_1 = require("tslib");
const glob_1 = require("glob");
const path_1 = require("path");
const console_utils_1 = tslib_1.__importDefault(require("../../utils/console.utils"));
const file_utils_1 = require("../../utils/file.utils");
// This file specifically handles cases where we're able to find
// a license file on disk that is a part of the package but it's
// not referenced in the package.json file.
// A 'best guess' for file extensions that are not license files
// but that may have the same name as a license file
exports.extensionDenyList = [".js", ".ts", ".sh", ".ps1"];
const licenseFile = (inputs) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const { directory, packageJson } = inputs;
    const licenseFiles = yield (0, glob_1.glob)("{license,licence,copying}{,-*,.*}", {
        nocase: true,
        nodir: true,
        absolute: true,
        cwd: directory,
        maxDepth: 1,
    });
    const filteredLicenseFiles = licenseFiles.filter(file => !exports.extensionDenyList.includes((0, path_1.extname)(file)));
    if (filteredLicenseFiles.length === 0) {
        return null;
    }
    if (filteredLicenseFiles.length > 1) {
        const relativeLicenseFiles = filteredLicenseFiles.map(file => ` - ./${(0, path_1.relative)(directory, file)}`);
        const warningLines = [
            `Found multiple license files for ${packageJson.name}@${packageJson.version}:`,
            ...relativeLicenseFiles,
            "We suggest you determine which file you wish to use and replace the license content",
            `for ${packageJson.name}@${packageJson.version} using a generate-license-file config file.`,
            "See: https://generate-license-file.js.org/docs/cli/config-file for more information.",
            "", // Empty line for spacing
        ];
        console_utils_1.default.warn(warningLines.join("\n"));
    }
    return yield (0, file_utils_1.readFile)(filteredLicenseFiles[0], { encoding: "utf-8" });
});
exports.licenseFile = licenseFile;
//# sourceMappingURL=licenseFile.js.map