"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveDependenciesForPnpmProject = void 0;
const tslib_1 = require("tslib");
const path_1 = require("path");
const console_utils_1 = tslib_1.__importDefault(require("../../utils/console.utils"));
const packageJson_utils_1 = require("../../utils/packageJson.utils");
const pnpmCli_utils_1 = require("../../utils/pnpmCli.utils");
const resolveLicenseContent_1 = require("../resolveLicenseContent");
const resolveNoticeContent_1 = require("../resolveNoticeContent");
const resolveDependenciesForPnpmProject = (packageJson, licensesMap, options) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    var _a, _b, _c;
    const replacements = (_a = options === null || options === void 0 ? void 0 : options.replace) !== null && _a !== void 0 ? _a : {};
    const exclude = (_b = options === null || options === void 0 ? void 0 : options.exclude) !== null && _b !== void 0 ? _b : [];
    yield verifyPnpmVersion();
    const projectDirectory = (0, path_1.dirname)(packageJson);
    const dependencies = yield (0, pnpmCli_utils_1.getPnpmProjectDependencies)(projectDirectory);
    for (const dependency of dependencies) {
        for (const dependencyPath of dependency.paths) {
            const packageJson = yield (0, packageJson_utils_1.readPackageJson)((0, path_1.join)(dependencyPath, "package.json"));
            if (exclude.includes(`${packageJson.name}@${packageJson.version}`) ||
                exclude.includes(`${packageJson.name}`)) {
                continue;
            }
            try {
                const licenseContent = yield (0, resolveLicenseContent_1.resolveLicenseContent)(dependencyPath, packageJson, replacements);
                const notices = yield (0, resolveNoticeContent_1.resolveNotices)(dependencyPath);
                const noticeKey = notices.length === 0 ? "" : notices.join("\n");
                const licenseNoticePair = `${licenseContent}:${noticeKey !== null && noticeKey !== void 0 ? noticeKey : ""}`;
                const resolvedLicense = (_c = licensesMap.get(licenseNoticePair)) !== null && _c !== void 0 ? _c : {
                    dependencies: [],
                    licenseContent,
                    notices,
                };
                const alreadyExists = resolvedLicense.dependencies.find(dep => dep.name === dependency.name && dep.version === packageJson.version);
                if (!alreadyExists) {
                    resolvedLicense.dependencies.push({
                        name: dependency.name,
                        version: packageJson.version,
                    });
                }
                licensesMap.set(licenseNoticePair, resolvedLicense);
            }
            catch (error) {
                const warningLines = [
                    `Unable to determine license content for ${packageJson.name}@${packageJson.version} with error:`,
                    error instanceof Error ? error.message : error === null || error === void 0 ? void 0 : error.toString(),
                    "", // Empty line for spacing
                ];
                console_utils_1.default.warn(warningLines.join("\n"));
            }
        }
    }
});
exports.resolveDependenciesForPnpmProject = resolveDependenciesForPnpmProject;
const allowedPnpmMinorVersions = {
    10: 0,
    9: 0,
    8: 0,
    7: 33,
};
const verifyPnpmVersion = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const pnpmVersion = yield (0, pnpmCli_utils_1.getPnpmVersion)();
    const allowedMinorVersion = allowedPnpmMinorVersions[pnpmVersion.major];
    if (allowedMinorVersion !== undefined && pnpmVersion.minor >= allowedMinorVersion) {
        return;
    }
    const errorLines = [
        `Unsupported pnpm version: ${pnpmVersion.major}.${pnpmVersion.minor}.${pnpmVersion.patch}.`,
        "Generate license file currently only supports pnpm versions >=7.33.0 & >=8.0.0",
        "Please either switch to a supported version of pnpm or raise an issue on the generate-license-file repository for us to support your version of pnpm:",
        "https://github.com/TobyAndToby/generate-license-file",
    ];
    throw new Error(errorLines.join("\n"));
});
//# sourceMappingURL=resolvePnpmDependencies.js.map