import { LineString, MultiLineString, Feature, Point } from 'geojson';
import { Coord, Units } from '@turf/helpers';

/**
 * Returns the nearest point on a line to a given point.
 *
 * If any of the segments in the input line string are antipodal and therefore
 * have an undefined arc, this function will instead return that the point lies
 * on the line.
 *
 * @function
 * @param {Geometry|Feature<LineString|MultiLineString>} lines lines to snap to
 * @param {Geometry|Feature<Point>|number[]} pt point to snap from
 * @param {Object} [options={}] Optional parameters
 * @param {Units} [options.units='kilometers'] Supports all valid Turf {@link https://turfjs.org/docs/api/types/Units Units}
 * @returns {Feature<Point>} closest point on the `line` to `point`. The properties object will contain four values: `index`: closest point was found on nth line part, `multiFeatureIndex`: closest point was found on the nth line of the `MultiLineString`, `dist`: distance between pt and the closest point, `location`: distance along the line between start and the closest point.
 * @example
 * var line = turf.lineString([
 *     [-77.031669, 38.878605],
 *     [-77.029609, 38.881946],
 *     [-77.020339, 38.884084],
 *     [-77.025661, 38.885821],
 *     [-77.021884, 38.889563],
 *     [-77.019824, 38.892368]
 * ]);
 * var pt = turf.point([-77.037076, 38.884017]);
 *
 * var snapped = turf.nearestPointOnLine(line, pt, {units: 'miles'});
 *
 * //addToMap
 * var addToMap = [line, pt, snapped];
 * snapped.properties['marker-color'] = '#00f';
 */
declare function nearestPointOnLine<G extends LineString | MultiLineString>(lines: Feature<G> | G, pt: Coord, options?: {
    units?: Units;
}): Feature<Point, {
    dist: number;
    index: number;
    multiFeatureIndex: number;
    location: number;
    [key: string]: any;
}>;

export { nearestPointOnLine as default, nearestPointOnLine };
