/*
    This file is part of Helio music sequencer.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Common.h"
#include "HelioTheme.h"
#include "SequencerSidebarLeft.h"
#include "ColourIDs.h"
#include "Config.h"
#include "SerializationKeys.h"

#include "PageBackgroundA.h"
#include "PageBackgroundB.h"
#include "PanelBackground.h"

HelioTheme::HelioTheme() :
    backgroundNoise(ImageCache::getFromMemory(BinaryData::noise_png, BinaryData::noise_pngSize)),
    backgroundStripes(ImageCache::getFromMemory(BinaryData::stripes_png, BinaryData::stripes_pngSize)) {}

HelioTheme &HelioTheme::getCurrentTheme() noexcept
{
    // This assumes the app have set an instance of HelioTheme as default look-and-feel
    return static_cast<HelioTheme &>(LookAndFeel::getDefaultLookAndFeel());
}

static constexpr auto noiseAlpha = 0.01f;

void HelioTheme::drawNoise(Graphics &g, float alphaMultiply /*= 1.f*/) const
{
    g.setTiledImageFill(this->backgroundNoise, 0, 0, noiseAlpha * alphaMultiply);
    g.fillAll();
}

void HelioTheme::drawNoise(const HelioTheme &theme, Graphics &g, float alphaMultiply /*= 1.f*/)
{
    theme.drawNoise(g, alphaMultiply);
}

void HelioTheme::drawStripes(Rectangle<float> bounds, Graphics &g, float alphaMultiply /*= 1.f*/)
{
    const auto &theme = getCurrentTheme();
    g.setTiledImageFill(theme.backgroundStripes, 0, 0,
        alphaMultiply * (theme.isDarkTheme ? 0.25f : 0.1f));
    g.fillRect(bounds);
}

//===----------------------------------------------------------------------===//
// Frames/lines helpers
//===----------------------------------------------------------------------===//

void HelioTheme::drawFrame(Graphics &g, int width, int height,
    float lightAlphaMultiplier, float darkAlphaMultiplier)
{
    g.setColour(findDefaultColour(ColourIDs::Common::borderLineDark).
        withMultipliedAlpha(darkAlphaMultiplier));

    g.fillRect(1, 0, width - 2, 1);
    g.fillRect(1, height - 1, width - 2, 1);
    g.fillRect(0, 1, 1, height - 2);
    g.fillRect(width - 1, 1, 1, height - 2);

    g.setColour(findDefaultColour(ColourIDs::Common::borderLineLight).
        withMultipliedAlpha(lightAlphaMultiplier));

    g.fillRect(2, 1, width - 4, 1);
    g.fillRect(2, height - 2, width - 4, 1);
    g.fillRect(1, 2, 1, height - 4);
    g.fillRect(width - 2, 2, 1, height - 4);
}

void HelioTheme::drawDashedFrame(Graphics &g, const Rectangle<int> &area, int dashLength /*= 4*/)
{
    const auto a = area.toFloat();
    HelioTheme::drawDashedHorizontalLine(g, a.getX() + 1.f, a.getY(), a.getWidth() - 2.f, float(dashLength));
    HelioTheme::drawDashedHorizontalLine(g, a.getX() + 1.f, a.getHeight() - 1.f, a.getWidth() - 2.f, float(dashLength));
    HelioTheme::drawDashedVerticalLine(g, a.getX(), a.getY() + 1.f, a.getHeight() - 2.f, float(dashLength));
    HelioTheme::drawDashedVerticalLine(g, a.getWidth() - 1.f, a.getY() + 1.f, a.getHeight() - 2.f, float(dashLength));
}

void HelioTheme::drawBrackets(Graphics &g,
    const Rectangle<int> &bounds, int lh, int lv, int t)
{
    const auto x = bounds.getX();
    const auto y = bounds.getY();
    const auto r = bounds.getRight();
    const auto b = bounds.getBottom();

    g.fillRect(x, y, lh, t);
    g.fillRect(x, y + t, t, lv - t);
    g.fillRect(x, b - t, lh, t);
    g.fillRect(x, b - lv, t, lv - t);
    g.fillRect(r - lh, y, lh, t);
    g.fillRect(r - t, y + t, t, lv - t);
    g.fillRect(r - lh, b - t, lh, t);
    g.fillRect(r - t, b - lv, t, lv - t);
}

void HelioTheme::drawDashedHorizontalLine(Graphics &g, float x, float y, float width, float dashLength)
{
    const auto r = x + width;
    // dash spaces are 1 pixel shorter than strokes because it looks nicer:
    for (; x < width - dashLength; x += ((dashLength * 2.f) - 1.f))
    {
        g.fillRect(x, y, dashLength, 1.f);
    }

    if (x < r)
    {
        g.fillRect(x, y, r - x, 1.f);
    }
}

void HelioTheme::drawDashedHorizontalLine2(Graphics &g, float x, float y, float width, float dashLength)
{
    const auto r = x + width;
    for (; x < width - dashLength; x += ((dashLength * 2.f) - 1.f))
    {
        g.fillRect(x + 1.f, y, dashLength, 1.f);
        g.fillRect(x, y + 1.f, dashLength, 1.f);
    }

    if (x < r)
    {
        g.fillRect(x + 1.f, y, jmax(0.f, r - x - 1.f), 1.f);
        g.fillRect(x, y + 1.f, r - x, 1.f);
    }
}

void HelioTheme::drawDashedVerticalLine(Graphics &g, float x, float y, float height, float dashLength)
{
    const auto b = y + height;
    for (; y < height - dashLength; y += ((dashLength * 2.f) - 1.f))
    {
        g.fillRect(x, y, 1.f, dashLength);
    }

    if (y < b)
    {
        g.fillRect(x, y, 1.f, b - y);
    }
}

//===----------------------------------------------------------------------===//
// GlyphArrangement helpers to avoid using JUCE's GlyphArrangementCache
//===----------------------------------------------------------------------===//

void HelioTheme::drawText(Graphics &g,
    const String &text, Rectangle<float> area,
    Justification justificationType, bool useEllipsesIfTooBig)
{
    GlyphArrangement arrangement;

    arrangement.addCurtailedLineOfText(g.getCurrentFont(),
        text, 0.f, 0.f,
        area.getWidth(),
        useEllipsesIfTooBig);

    arrangement.justifyGlyphs(0,
        arrangement.getNumGlyphs(),
        area.getX(), area.getY(),
        area.getWidth(), area.getHeight(),
        justificationType);

    arrangement.draw(g);
}

void HelioTheme::drawFittedText(Graphics &g,
    const String &text, int x, int y, int width, int height,
    Justification justification,
    const int maximumNumberOfLines,
    const float minimumHorizontalScale)
{
    GlyphArrangement arrangement;

    arrangement.addFittedText(g.getCurrentFont(),
        text,
        float(x), float(y),
        float(width), float(height),
        justification,
        maximumNumberOfLines,
        minimumHorizontalScale);

    arrangement.draw(g);
}

Typeface::Ptr HelioTheme::getTypefaceForFont(const Font &font)
{
#if PLATFORM_DESKTOP
    if (font.getTypefaceName() == Font::getDefaultSansSerifFontName() ||
        font.getTypefaceName() == Font::getDefaultSerifFontName())
    {
        return this->textTypefaceCache;
    }
#endif

    return Font::getDefaultTypefaceForFont(font);
}

void HelioTheme::drawStretchableLayoutResizerBar(Graphics &g,
        int w, int h, bool isVerticalBar,
        bool isMouseOver, bool isMouseDragging) {}

//===----------------------------------------------------------------------===//
// Text Editor
//===----------------------------------------------------------------------===//

void HelioTheme::fillTextEditorBackground(Graphics &g, int w, int h, TextEditor &ed)
{
    g.setColour(this->findColour(TextEditor::backgroundColourId));
    g.fillRect(1, 1, w - 2, h - 2);
    g.setColour(this->findColour(TextEditor::outlineColourId));
    g.drawVerticalLine(0, 1.f, h - 1.f);
    g.drawVerticalLine(w - 1, 1.f, h - 1.f);
    g.drawHorizontalLine(0, 1.f, w - 1.f);
    g.drawHorizontalLine(h - 1, 1.f, w - 1.f);
}

void HelioTheme::drawPopupMenuBackground(Graphics& g, int width, int height)
{
    g.fillAll(findColour(PopupMenu::backgroundColourId));
}

//===----------------------------------------------------------------------===//
// Labels
//===----------------------------------------------------------------------===//

void HelioTheme::drawLabel(Graphics &g, Label &label)
{
    this->drawLabel(g, label, 0);
}

// Label rendering is one of the most time-consuming bottlenecks in the app.
// Calling this method too often should be avoided at any costs:
// relatively small labels that are always on the screen (like headline titles,
// annotations, key/time signatures, track headers) should be set cached to images,
// and they also should have fixed sizes (not dependent on a container component's size)
// so that resizing a parent won't force resizing, re-rendering and re-caching a label.
// Such labels should be re-rendered only when their content changes.

void HelioTheme::drawLabel(Graphics &g, Label &label, juce_wchar passwordCharacter)
{
    if (label.isBeingEdited())
    {
        return;
    }

    const auto font = this->getLabelFont(label);
    const auto textToDraw = (passwordCharacter != 0) ?
        String::repeatedString(String::charToString(passwordCharacter), label.getText().length()) :
        label.getText();

    // Try to guess the right max number of lines depending on label height and font height:
    const int maxLines = int(float(label.getHeight()) / font.getHeight());

    // using label.findColour, not findDefaultColour, as it is actually overridden in some places:
    const auto colour = label.findColour(Label::textColourId);
    const auto textArea = label.getBorderSize().subtractedFrom(label.getLocalBounds());

    // there is something wrong with how JUCE does vertical font centering
    // on both iOS and Android; my guess is that it might be incorrectly
    // handling ascents/descents/bounding boxes of the default fonts;
    // whatever it is, this dirty hack is here to aid it:
#if JUCE_IOS
    const int heightHack = 3;
#elif JUCE_ANDROID
    const int heightHack = 2;
#else
    const int heightHack = 0;
#endif

    g.setFont(font);
    g.setColour(colour);
    HelioTheme::drawFittedText(g,
        textToDraw,
        textArea.getX(),
        textArea.getY(),
        textArea.getWidth(),
        textArea.getHeight() + heightHack,
        label.getJustificationType(),
        maxLines,
        label.getMinimumHorizontalScale());
}

//===----------------------------------------------------------------------===//
// Buttons
//===----------------------------------------------------------------------===//

Font HelioTheme::getTextButtonFont(TextButton &button, int buttonHeight)
{
    return { Globals::UI::Fonts::L };
}

void HelioTheme::drawButtonText(Graphics &g, TextButton &button,
    bool isMouseOverButton, bool isButtonDown)
{
    const auto font = this->getTextButtonFont(button, button.getHeight());
    g.setFont(font);
    
    const int yIndent = jmin(4, button.proportionOfHeight(0.3f));
    const int yHeight = (button.getHeight() - (yIndent * 2));
    const int cornerSize = jmin(button.getHeight(), button.getWidth()) / 2;

    const int fontHeight = int(font.getHeight() * 0.5f);
    const int leftIndent = fontHeight;
    const int rightIndent = jmin(fontHeight, 2 + cornerSize / (button.isConnectedOnRight() ? 4 : 2));

    g.setColour(findDefaultColour(TextButton::textColourOnId)
        .withMultipliedAlpha(button.isEnabled() ? 1.0f : 0.5f));

    HelioTheme::drawFittedText(g,
        button.getButtonText(),
        leftIndent,
        yIndent,
        button.getWidth() - leftIndent - rightIndent,
        yHeight,
        Justification::centred, 4, 1.f);
}

void HelioTheme::drawButtonBackground(Graphics &g, Button &button,
    const Colour &backgroundColour,
    bool isMouseOverButton, bool isButtonDown)
{
    const auto width = float(button.getWidth());
    const auto height = float(button.getHeight());
    if (width == 0.f || height == 0.f)
    {
        jassertfalse;
        return;
    }

    constexpr float cornerSize = 1.f;
    const auto baseColour = backgroundColour
        .withMultipliedAlpha(button.isEnabled() ? 0.75f : 0.3f);

    Path outline;
    outline.addRoundedRectangle(0.f, 0.f,
        width, height, cornerSize, cornerSize,
        true, true, true, true);

    g.setColour(baseColour);
    g.fillPath(outline);

    if (isButtonDown || isMouseOverButton)
    {
        g.setColour(baseColour.brighter(isButtonDown ? 0.1f : 0.01f));
        g.fillPath(outline);
    }

    g.setColour(baseColour.withMultipliedAlpha(1.1f));
    g.strokePath(outline, PathStrokeType(1.f));
}

Path HelioTheme::getTickShape(float height)
{
    static const unsigned char pathData[] = {
        110,109,32,210,202,64,126,183,148,64,108,39,244,247,64,245,76,124,64,108,178,131,27,65,246,76,252,64,108,175,242,4,65,246,76,252,
        64,108,236,5,68,65,0,0,160,180,108,240,150,90,65,21,136,52,63,108,48,59,16,65,0,0,32,65,108,32,210,202,64,126,183,148,64, 99,101,0,0
    };

    Path path;
    path.loadPathFromData(pathData, sizeof(pathData));
    path.scaleToFit(0, 0, height * 2.0f, height, true);
    return path;
}

void HelioTheme::drawToggleButton(Graphics &g, ToggleButton &button,
    bool shouldDrawButtonAsHighlighted, bool shouldDrawButtonAsDown)
{
    constexpr auto fontSize = Globals::UI::Fonts::M;
    constexpr auto tickWidth = fontSize;

    const auto isRadioButton = button.getRadioGroupId() != 0;

    const auto tickBounds = Rectangle<float>(4.f, 
        (float(button.getHeight()) - tickWidth) * 0.5f,
        tickWidth, tickWidth).reduced(isRadioButton ? 1.5f : 0.f);

    g.setColour(button.findColour(ToggleButton::tickDisabledColourId));
    g.drawRoundedRectangle(tickBounds,
        isRadioButton ? (tickBounds.getWidth() / 2.f) : 4.f,
        isRadioButton ? 1.25f : 1.f);

    if (button.getToggleState())
    {
        g.setColour(button.findColour(ToggleButton::tickColourId));

        if (isRadioButton)
        {
            const auto checkBounds = tickBounds.reduced(4.5f);
            g.fillRoundedRectangle(checkBounds, (checkBounds.getWidth() / 2.f));
        }
        else
        {
            const auto checkShape = this->getTickShape(0.75f);
            g.fillPath(checkShape,
                checkShape.getTransformToScaleToFit(tickBounds.reduced(4.f, 5.f), false));
        }
    }

    const Font font(fontSize);
    g.setFont(font);
    g.setColour(this->findColour(ToggleButton::textColourId));

    if (!button.isEnabled())
    {
        g.setOpacity(0.5f);
    }

    const auto area = button.getLocalBounds()
        .withTrimmedLeft(roundToInt(tickWidth) + 10)
        .withTrimmedRight(2);

    HelioTheme::drawFittedText(g,
        button.getButtonText(),
        area.getX(), area.getY(),
        area.getWidth(), area.getHeight(),
        Justification::centredLeft,
        10, 1.f);
}

//===----------------------------------------------------------------------===//
// Tables
//===----------------------------------------------------------------------===//

void HelioTheme::drawTableHeaderBackground(Graphics &g, TableHeaderComponent &header) {}

void HelioTheme::drawTableHeaderColumn(Graphics &g,
    TableHeaderComponent &header, const String &columnName,
    int columnId, int width, int height, bool isMouseOver, bool isMouseDown, int columnFlags)
{
    const auto highlightColour = this->findColour(TableHeaderComponent::highlightColourId);

    if (isMouseDown)
    {
        g.fillAll(highlightColour);
    }
    else if (isMouseOver)
    {
        g.fillAll(highlightColour.withMultipliedAlpha(0.625f));
    }

    Rectangle<int> area(width, height);
    area.reduce(4, 0);

    g.setColour(findDefaultColour(TableHeaderComponent::textColourId));
    if ((columnFlags & (TableHeaderComponent::sortedForwards | TableHeaderComponent::sortedBackwards)) != 0)
    {
        Path sortArrow;
        sortArrow.addTriangle(0.f, 0.f,
            0.5f, (columnFlags & TableHeaderComponent::sortedForwards) != 0 ? -0.8f : 0.8f,
            1.f, 0.f);

        g.fillPath(sortArrow,
            sortArrow.getTransformToScaleToFit(area.removeFromRight(height / 2)
                .reduced(6).toFloat(), true));
    }

#if PLATFORM_DESKTOP
    g.setFont(Globals::UI::Fonts::M);
#elif PLATFORM_MOBILE
    g.setFont(Globals::UI::Fonts::S);
#endif

    HelioTheme::drawFittedText(g,
        columnName,
        area.getX(), area.getY(), area.getWidth(), area.getHeight(),
        columnId == 1 ? // a nasty hack, only used in AudioPluginsListComponent
            Justification::centredRight : Justification::centredLeft, 1);
}

//===----------------------------------------------------------------------===//
// Scrollbars
//===----------------------------------------------------------------------===//

int HelioTheme::getDefaultScrollbarWidth()
{
#if PLATFORM_DESKTOP
    return 2;
#elif PLATFORM_MOBILE
    return 20;
#endif
}

void HelioTheme::drawScrollbar(Graphics &g, ScrollBar &scrollbar,
    int x, int y, int width, int height,
    bool isScrollbarVertical, int thumbStartPosition, int thumbSize,
    bool isMouseOver, bool isMouseDown)
{
    Path thumbPath;

    if (thumbSize > 0)
    {
        const float thumbIndent = (isScrollbarVertical ? width : height) * 0.25f;
        const float thumbIndentx2 = thumbIndent * 2.0f;

        if (isScrollbarVertical)
        {
            thumbPath.addRoundedRectangle(x + thumbIndent, thumbStartPosition + thumbIndent,
                width - thumbIndentx2, thumbSize - thumbIndentx2, 1.f);
        }
        else
        {
            thumbPath.addRoundedRectangle(thumbStartPosition + thumbIndent, y + thumbIndent,
                thumbSize - thumbIndentx2, height - thumbIndentx2, 1.f);
        }
    }

#if PLATFORM_DESKTOP
    const auto thumbCol = this->findColour(ScrollBar::thumbColourId);
#elif PLATFORM_MOBILE
    const auto thumbCol = this->findColour(ScrollBar::thumbColourId).
        withMultipliedAlpha((isMouseOver || isMouseDown) ? 0.69f : 0.420f);
#endif

    g.setColour(thumbCol);
    g.fillPath(thumbPath);
}

//===----------------------------------------------------------------------===//
// Sliders
//===----------------------------------------------------------------------===//

void HelioTheme::drawRotarySlider(Graphics &g, int x, int y, int width, int height,
    float sliderPos, float rotaryStartAngle, float rotaryEndAngle, Slider &slider)
{
    const auto fill = findDefaultColour(Slider::rotarySliderFillColourId);
    const auto outline = findDefaultColour(Slider::rotarySliderOutlineColourId);
    
    const auto bounds = Rectangle<int>(x, y, width, height).toFloat().reduced(8);
    const auto radius = jmin(bounds.getWidth(), bounds.getHeight()) / 2.0f;
    const auto toAngle = rotaryStartAngle + sliderPos * (rotaryEndAngle - rotaryStartAngle);
    const auto lineW = jmin(8.0f, radius * 0.25f);
    const auto arcRadius = radius - lineW * 0.5f;

    if (slider.isEnabled())
    {
        Path fullArc;
        fullArc.addCentredArc(bounds.getCentreX(), bounds.getCentreY(),
            arcRadius, arcRadius, 0.0f, rotaryStartAngle, rotaryEndAngle, true);

        g.setColour(outline);
        g.strokePath(fullArc, PathStrokeType(lineW, PathStrokeType::curved, PathStrokeType::rounded));

        Path valueArc;
        valueArc.addCentredArc(bounds.getCentreX(), bounds.getCentreY(),
            arcRadius, arcRadius, 0.0f, rotaryStartAngle, toAngle, true);

        g.setColour(fill);
        g.strokePath(valueArc, PathStrokeType(lineW, PathStrokeType::curved, PathStrokeType::rounded));
    }
}

//===----------------------------------------------------------------------===//
// Window
//===----------------------------------------------------------------------===//

void HelioTheme::drawCornerResizer(Graphics& g, int w, int h,
    bool /*isMouseOver*/, bool /*isMouseDragging*/)
{
    const float lineThickness = jmin(w, h) * 0.05f;
    const auto colour1 = this->findColour(ResizableWindow::backgroundColourId).darker(0.25f);
    const auto colour2 = this->findColour(ResizableWindow::backgroundColourId).brighter(0.05f);

    for (float i = 0.8f; i > 0.2f; i -= 0.25f)
    {
        g.setColour(colour1);
        g.drawLine(w * i, h + 1.f, w + 1.f, h * i, lineThickness);

        g.setColour(colour2);
        g.drawLine(w * i + lineThickness, h + 1.f, w + 1.f, h * i + lineThickness, lineThickness);
    }
}

void HelioTheme::drawResizableFrame(Graphics &g, int w, int h, const BorderSize<int> &border)
{
    if (!border.isEmpty())
    {
        const Rectangle<int> fullSize(0, 0, w, h);
        const Rectangle<int> centreArea(border.subtractedFrom(fullSize));

        g.saveState();

        g.excludeClipRegion(centreArea);

        g.setColour(this->findColour(ResizableWindow::backgroundColourId));
        g.drawRect(fullSize, 5);

        g.restoreState();
    }
}

class HelioWindowButton final : public Button
{
public:

    explicit HelioWindowButton(const Path &shape) noexcept :
        Button({}),
        shape(shape) {}

    void paintButton(Graphics &g, bool isMouseOverButton, bool isButtonDown) override
    {
        float alpha = isMouseOverButton ? (isButtonDown ? 1.0f : 0.8f) : 0.6f;
        if (!this->isEnabled())
        {
            alpha *= 0.5f;
        }

        float x = 0, y = 0, diam;

        if (this->getWidth() < this->getHeight())
        {
            diam = (float)this->getWidth();
            y = (this->getHeight() - this->getWidth()) * 0.5f;
        }
        else
        {
            diam = (float)this->getHeight();
            y = (this->getWidth() - this->getHeight()) * 0.5f;
        }

        x += diam * 0.05f;
        y += diam * 0.05f;
        diam *= 0.9f;

        const Colour colour(findDefaultColour(TextButton::textColourOnId));

        if (isMouseOverButton)
        {
            g.setColour(colour.withAlpha(alpha * 0.05f));
            g.fillAll();
        }

        const AffineTransform t(this->shape.getTransformToScaleToFit(x + diam * 0.3f,
            y + diam * 0.3f, diam * 0.4f, diam * 0.4f, true));

        g.setColour(colour);
        g.fillPath(this->shape, t);
    }

private:

    const Path shape;

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(HelioWindowButton)
};

void HelioTheme::drawDocumentWindowTitleBar(DocumentWindow &window,
    Graphics &g, int w, int h,
    int titleSpaceX, int titleSpaceW,
    const Image *icon,
    bool drawTitleTextOnLeft)
{
#if PLATFORM_DESKTOP
    if (window.isUsingNativeTitleBar())
    {
        return;
    }

    const auto &theme = HelioTheme::getCurrentTheme();
    g.setFillType({ theme.getPageBackgroundA(), {} });
    g.fillRect(0, 0, w, h);

    g.setColour(findDefaultColour(ColourIDs::Common::borderLineLight));
    g.fillRect(0, h - 2, w, 1);
    g.setColour(findDefaultColour(ColourIDs::Common::borderLineDark));
    g.fillRect(0, h - 1, w, 1);
    g.setColour(findDefaultColour(ColourIDs::Common::borderLineLight).withMultipliedAlpha(0.35f));
    g.fillRect(0, 0, w, 1);

    const auto uiScaleFactor = App::Config().getUiFlags()->getUiScaleFactor();
    const Font font(Globals::UI::Fonts::S * uiScaleFactor, Font::plain);
    g.setFont(font);

    static const String title = "helio " + App::getAppReadableVersion();
    const auto textWidth = font.getStringWidth(title);
    g.setColour(findDefaultColour(Label::textColourId).withAlpha(0.25f));
    g.drawText(title, titleSpaceX, 0, w - textWidth - 24, h - 3, Justification::centredRight, true);
#endif
}

Button *HelioTheme::createDocumentWindowButton(int buttonType)
{
    Path shape;

    if (buttonType == DocumentWindow::closeButton)
    {
        shape.addLineSegment(Line<float>(0.0f, 0.0f, 1.0f, 1.0f), 0.05f);
        shape.addLineSegment(Line<float>(1.0f, 0.0f, 0.0f, 1.0f), 0.05f);
        return new HelioWindowButton(shape);
    }
    if (buttonType == DocumentWindow::minimiseButton)
    {
        shape.addLineSegment(Line<float>(0.0f, 0.0f, 0.00001f, 0.0f), 0.00001f);
        shape.addLineSegment(Line<float>(0.0f, 0.55f, 1.0f, 0.55f), 0.05f);
        return new HelioWindowButton(shape);
    }
    else if (buttonType == DocumentWindow::maximiseButton)
    {
        shape.addLineSegment(Line<float>(0.0f, 0.0f, 0.0f, 0.8f), 0.05f);
        shape.addLineSegment(Line<float>(0.0f, 0.8f, 1.0f, 0.8f), 0.05f);
        shape.addLineSegment(Line<float>(1.0f, 0.8f, 1.0f, 0.0f), 0.05f);
        shape.addLineSegment(Line<float>(1.0f, 0.0f, 0.0f, 0.0f), 0.05f);
        return new HelioWindowButton(shape);
    }

    jassertfalse;
    return nullptr;
}

void HelioTheme::positionDocumentWindowButtons(DocumentWindow &,
        int titleBarX, int titleBarY,
        int titleBarW, int titleBarH,
        Button *minimiseButton,
        Button *maximiseButton,
        Button *closeButton,
        bool positionTitleBarButtonsOnLeft)
{
    const int buttonSize = int(23 * App::Config().getUiFlags()->getUiScaleFactor());
    const int y = ((titleBarH - titleBarY) / 2) - (buttonSize / 2) - 1;
    int x = titleBarX + titleBarW - buttonSize - buttonSize / 4;

    if (closeButton != nullptr)
    {
        closeButton->setBounds(x, y, buttonSize, buttonSize);
        x += -(buttonSize + buttonSize / 4);
    }

    if (maximiseButton != nullptr)
    {
        maximiseButton->setBounds(x, y, buttonSize, buttonSize);
        x += -(buttonSize + buttonSize / 4);
    }

    if (minimiseButton != nullptr)
    {
        minimiseButton->setBounds(x, y, buttonSize, buttonSize);
    }
}

void HelioTheme::initResources() noexcept
{
    Icons::initBuiltInImages();

#if PLATFORM_DESKTOP

    if (App::Config().containsProperty(Serialization::Config::lastUsedFont))
    {
        const String lastUsedFontName = App::Config().getProperty(Serialization::Config::lastUsedFont);
        this->textTypefaceCache = Typeface::createSystemTypefaceFor(Font(lastUsedFontName, 0, Font::plain));
        return;
    }

#if DEBUG
    const auto scanStartTime = Time::getMillisecondCounter();
#endif

    // using Font::findAllTypefaceNames instead of the slower
    // Font::findFonts which seems to cause freezes on some systems
    const auto systemFonts = Font::findAllTypefaceNames();

    DBG("Found " + String(systemFonts.size()) + " fonts in " +
        String(Time::getMillisecondCounter() - scanStartTime) + " ms");

    const auto userLanguage = SystemStats::getUserLanguage().toLowerCase().substring(0, 2);
    auto preferredFontNames = StringArray("Noto Sans", "Noto Sans UI", "Source Han Sans");

    if (userLanguage == "kr")
    {
        preferredFontNames.add("Dotum");
    }
    else if (userLanguage == "zh" || userLanguage == "ja")
    {
        preferredFontNames.addArray({ "YeHei", "Hei", "Heiti SC" });
    }

#if JUCE_LINUX
    preferredFontNames.add("Ubuntu");
    preferredFontNames.add("Liberation Sans");
#endif

    String pickedFontName;
    const String perfectlyFineFontName = "Noto Sans CJK";
    for (const auto &fontName : systemFonts)
    {
        if (preferredFontNames.contains(fontName))
        {
            pickedFontName = fontName;
        }
        else if (fontName.startsWithIgnoreCase(perfectlyFineFontName))
        {
            pickedFontName = fontName;
            break;
        }
    }

    if (pickedFontName.isNotEmpty())
    {
        this->textTypefaceCache = Typeface::createSystemTypefaceFor({ pickedFontName, 0, Font::plain });
    }
    else
    {
        // Verdana on Windows, Bitstream Vera Sans or something on Linux, Lucida Grande on macOS:
        this->textTypefaceCache = Font::getDefaultTypefaceForFont({ Font::getDefaultSansSerifFontName(), 0, Font::plain });
    }

    DBG("Using font: " + this->textTypefaceCache->getName());
    App::Config().setProperty(Serialization::Config::lastUsedFont, this->textTypefaceCache->getName());

#endif
}

void HelioTheme::updateFont(const Font &font) noexcept
{
#if PLATFORM_DESKTOP

    Typeface::clearTypefaceCache();
    this->textTypefaceCache = Typeface::createSystemTypefaceFor(font);
    App::Config().setProperty(Serialization::Config::lastUsedFont, font.getTypefaceName());

#endif
}

void HelioTheme::initColours(const ::ColourScheme::Ptr s) noexcept
{
    const auto textColour = s->getTextColour();
    
    // bright text probably means dark theme:
    this->isDarkTheme = textColour.getPerceivedBrightness() > 0.5f;

    this->setColour(Slider::rotarySliderOutlineColourId, textColour.contrasting(0.9f).withMultipliedAlpha(0.5f));
    this->setColour(Slider::rotarySliderFillColourId, textColour);
    this->setColour(Slider::thumbColourId, textColour);
    this->setColour(Slider::trackColourId, textColour.withMultipliedAlpha(0.5f));

    this->setColour(Label::textColourId, textColour);
    this->setColour(Label::outlineColourId, textColour.contrasting());

    this->setColour(ResizableWindow::backgroundColourId, s->getPageFillColour().brighter(0.045f));
    this->setColour(ScrollBar::backgroundColourId, Colours::transparentBlack);
    this->setColour(ScrollBar::thumbColourId,
        s->getFrameBorderColour().withAlpha(this->isDarkTheme ? 0.25f : 0.35f));

    this->setColour(TextButton::buttonColourId,
        s->getDialogFillColour().brighter(2.5f).withMultipliedAlpha(this->isDarkTheme ? 0.0625f : 0.35f));
    this->setColour(TextButton::buttonOnColourId,
        s->getDialogFillColour().brighter(2.5f).withMultipliedAlpha(this->isDarkTheme ? 0.025f : 0.25f));
    this->setColour(TextButton::textColourOnId, textColour.withMultipliedAlpha(0.8f));
    this->setColour(TextButton::textColourOffId, textColour.withMultipliedAlpha(0.5f));

    this->setColour(TextEditor::textColourId, textColour);
    this->setColour(TextEditor::backgroundColourId, s->getPageFillColour().darker(0.055f));
    this->setColour(TextEditor::outlineColourId, s->getFrameBorderColour().withAlpha(0.075f));
    this->setColour(TextEditor::highlightedTextColourId, textColour);
    this->setColour(TextEditor::focusedOutlineColourId, textColour.contrasting().withAlpha(0.2f));
    this->setColour(TextEditor::shadowColourId, s->getPageFillColour().darker(0.05f));
    this->setColour(TextEditor::highlightColourId, textColour.withAlpha(this->isDarkTheme ? 0.035f : 0.07f));
    this->setColour(CaretComponent::caretColourId, textColour.withAlpha(0.35f));

    this->setColour(PopupMenu::backgroundColourId, s->getPageFillColour());
    this->setColour(PopupMenu::textColourId, textColour);
    this->setColour(PopupMenu::headerTextColourId, textColour);
    this->setColour(PopupMenu::highlightedBackgroundColourId, s->getPageFillColour().brighter(0.069f));
    this->setColour(PopupMenu::highlightedTextColourId, textColour);

    this->setColour(ListBox::textColourId, textColour);
    this->setColour(ListBox::backgroundColourId, Colours::transparentBlack);
    this->setColour(TableHeaderComponent::backgroundColourId, Colours::transparentBlack);
    this->setColour(TableHeaderComponent::outlineColourId, s->getFrameBorderColour().withAlpha(0.05f));
    this->setColour(TableHeaderComponent::highlightColourId, s->getPageFillColour().brighter(0.05f));
    this->setColour(TableHeaderComponent::textColourId, textColour.withMultipliedAlpha(0.8f));

    this->setColour(ToggleButton::textColourId, textColour);
    this->setColour(ToggleButton::tickColourId, textColour);
    this->setColour(ToggleButton::tickDisabledColourId, textColour.withMultipliedAlpha(0.65f));

    this->setColour(ColourIDs::SelectionComponent::fill, s->getLassoFillColour().withAlpha(0.2f));
    this->setColour(ColourIDs::SelectionComponent::outline, s->getLassoBorderColour().withAlpha(0.75f));

    this->setColour(ColourIDs::RollHeader::selection, s->getLassoBorderColour().withAlpha(0.8f));
    this->setColour(ColourIDs::RollHeader::soundProbe, s->getLassoBorderColour().withMultipliedBrightness(1.1f).withAlpha(0.75f));
    this->setColour(ColourIDs::RollHeader::timeDistance, textColour.withAlpha(0.420f));

    this->setColour(ColourIDs::Icons::fill, s->getIconBaseColour());
    this->setColour(ColourIDs::Icons::shadow, s->getIconShadowColour());

    this->setColour(ColourIDs::Panel::pageFillA, s->getPageFillColour());
    this->setColour(ColourIDs::Panel::pageFillB, s->getPageFillColour().darker(0.01f));
    this->setColour(ColourIDs::Panel::sidebarFill, s->getSidebarFillColour());
    this->setColour(ColourIDs::Panel::bottomPanelFill,
        s->getSidebarFillColour().withMultipliedLightness(1.025f));

    this->setColour(ColourIDs::Breadcrumbs::fill, s->getHeadlineFillColour());
    this->setColour(ColourIDs::Breadcrumbs::selectionMarker, this->isDarkTheme ?
        Colours::white.withAlpha(0.1f) : Colours::black.withAlpha(0.1f));

    this->setColour(ColourIDs::Dialog::fill, s->getDialogFillColour());
    this->setColour(ColourIDs::Dialog::header,
        textColour.withAlpha(this->isDarkTheme ? 0.420f : 0.69f));

    this->setColour(ColourIDs::Menu::fill, s->getHeadlineFillColour());
    this->setColour(ColourIDs::Menu::header,
        s->getHeadlineFillColour().brighter(this->isDarkTheme ? 0.1f : 0.5f));
    this->setColour(ColourIDs::Menu::cursorFill, s->getTextColour().withAlpha(0.55f));
    this->setColour(ColourIDs::Menu::cursorShade, this->isDarkTheme ?
        s->getWhiteKeyColour().darker(0.420f).withAlpha(0.69f) :
        s->getWhiteKeyColour().brighter(0.420f).withAlpha(0.69f));
    const auto cursorHighlight =
        s->getHeadlineFillColour().brighter(2.f).withAlpha(this->isDarkTheme ? 0.025f : 0.2f);
    this->setColour(ColourIDs::Menu::highlight, cursorHighlight);
    this->setColour(ColourIDs::Menu::toggleMarker, s->getIconBaseColour().withMultipliedAlpha(0.9f));
    this->setColour(ColourIDs::Menu::currentItemMarker, s->getTextColour().withAlpha(0.55f));
    this->setColour(ColourIDs::Menu::currentItemFill, cursorHighlight);

    this->setColour(ColourIDs::Arrow::lineStart, this->isDarkTheme ? Colour(0x33ffffff) : Colour(0x44ffffff));
    this->setColour(ColourIDs::Arrow::lineEnd, this->isDarkTheme ? Colour(0x17ffffff) : Colour(0x27ffffff));
    this->setColour(ColourIDs::Arrow::shadowStart, this->isDarkTheme ? Colour(0x77000000) : Colour(0x66000000));
    this->setColour(ColourIDs::Arrow::shadowEnd, this->isDarkTheme ? Colour(0x17000000) : Colour(0x27000000));

    const auto tooltipFill = this->isDarkTheme ?
        s->getPageFillColour().darker(0.5f) : s->getPageFillColour().brighter(0.25f);
    this->setColour(ColourIDs::Tooltip::messageFill, tooltipFill);
    this->setColour(ColourIDs::Tooltip::messageBorder, textColour.withMultipliedAlpha(0.03f));
    this->setColour(ColourIDs::Tooltip::messageText, textColour);
    this->setColour(ColourIDs::Tooltip::okIconFill, tooltipFill);
    this->setColour(ColourIDs::Tooltip::okIconForeground, textColour);
    this->setColour(ColourIDs::Tooltip::failIconFill, tooltipFill);
    this->setColour(ColourIDs::Tooltip::failIconForeground, textColour);

    this->setColour(ColourIDs::Panel::border,
        s->getFrameBorderColour().withAlpha(this->isDarkTheme ? 0.225f : 0.3f));

    this->setColour(ColourIDs::TrackScroller::borderLineDark,
        s->getPageFillColour().darker(this->isDarkTheme ? 0.5f : 0.3f));
    this->setColour(ColourIDs::TrackScroller::borderLineLight,
        Colours::white.withAlpha(this->isDarkTheme ? 0.055f : 0.1f));
    const auto screenRangeFill = this->isDarkTheme ? s->getIconBaseColour() :
        s->getIconBaseColour().interpolatedWith(s->getBlackKeyColour(), 0.65f);
    this->setColour(ColourIDs::TrackScroller::viewBeatRangeFill, screenRangeFill.withMultipliedAlpha(0.225f));
    this->setColour(ColourIDs::TrackScroller::viewBeatRangeBorder, screenRangeFill.withMultipliedAlpha(0.125f));
    this->setColour(ColourIDs::TrackScroller::viewRangeFill, screenRangeFill.withMultipliedAlpha(0.325f));
    this->setColour(ColourIDs::TrackScroller::projectOutRangeFill,
        Colours::black.withAlpha(this->isDarkTheme ? 0.125f : 0.05f));

    this->setColour(ColourIDs::Instrument::fill, Colour(0x55ffffff));
    this->setColour(ColourIDs::Instrument::outline, Colour(0x55000000));
    this->setColour(ColourIDs::Instrument::text, Colours::black.withAlpha(0.75f));
    this->setColour(ColourIDs::Instrument::midiNode, Colours::black.withAlpha(0.15f));
    this->setColour(ColourIDs::Instrument::audioNode, Colours::black.withAlpha(0.15f));
    this->setColour(ColourIDs::Instrument::midiConnector,
        Colours::white.withAlpha(this->isDarkTheme ? 0.35f : 0.7f));
    this->setColour(ColourIDs::Instrument::audioConnector,
        Colours::black.withAlpha(this->isDarkTheme ? 0.45f : 0.35f));
    this->setColour(ColourIDs::Instrument::pinShadow, Colours::black.withAlpha(0.15f));
    this->setColour(ColourIDs::Instrument::connectorShadow, Colours::black.withAlpha(0.3f));

    this->setColour(ColourIDs::Common::borderLineLight,
        Colours::white.withAlpha(this->isDarkTheme ? 0.066f : 0.2f));
    this->setColour(ColourIDs::Common::borderLineDark,
        Colours::black.withAlpha(this->isDarkTheme ? 0.33f : 0.25f));
    this->setColour(ColourIDs::Common::separatorLineLight,
        Colours::white.withAlpha(this->isDarkTheme ? 0.077f : 0.25f));
    this->setColour(ColourIDs::Common::separatorLineDark,
        Colours::black.withAlpha(this->isDarkTheme ? 0.55f : 0.2f));

    this->setColour(ColourIDs::ColourButton::outline, textColour);
    this->setColour(ColourIDs::ColourButton::highlight, textColour.withAlpha(0.025f));
    this->setColour(ColourIDs::ColourButton::pressed, textColour.withAlpha(0.05f));

    this->setColour(ColourIDs::Callout::fill, s->getDialogFillColour());
    this->setColour(ColourIDs::Callout::frame, Colours::black.withAlpha(0.75f));

    this->setColour(ColourIDs::Roll::blackKey, s->getBlackKeyColour());
    this->setColour(ColourIDs::Roll::whiteKey, s->getWhiteKeyColour());
    this->setColour(ColourIDs::Roll::rootKey,
        s->getWhiteKeyColour().brighter(this->isDarkTheme ? 0.125f : 0.525f));

    this->setColour(ColourIDs::Roll::rowLine, s->getRowColour());
    this->setColour(ColourIDs::Roll::barLine, s->getBarColour().withAlpha(0.8f));
    this->setColour(ColourIDs::Roll::barLineBevel,
        Colours::white.withAlpha(this->isDarkTheme ? 0.015f : 0.075f));
    this->setColour(ColourIDs::Roll::beatLine, s->getBarColour().withAlpha(0.4f));
    this->setColour(ColourIDs::Roll::snapLine, s->getBarColour().withAlpha(0.1f));

    const auto headerFill = s->getTimelineColour();
    this->setColour(ColourIDs::Roll::headerFill, headerFill);
    this->setColour(ColourIDs::Roll::headerBorder,
        Colours::white.withAlpha(this->isDarkTheme ? 0.045f : 0.15f));
    this->setColour(ColourIDs::Roll::headerSnaps,
        headerFill.contrasting().interpolatedWith(headerFill, 0.625f));
    this->setColour(ColourIDs::Roll::headerReprise,
        headerFill.contrasting().interpolatedWith(headerFill, this->isDarkTheme ? 0.45f : 0.3f));
    this->setColour(ColourIDs::Roll::headerRecording,
        headerFill.interpolatedWith(Colours::red, 0.55f));

    this->setColour(ColourIDs::Roll::playheadShade,
        s->getBlackKeyColour().darker(1.f).withAlpha(0.069f));
    this->setColour(ColourIDs::Roll::playheadPlayback, s->getLassoBorderColour().
        interpolatedWith(s->getBlackKeyColour(), this->isDarkTheme ? 0.2f : 0.1f).withAlpha(1.f));
    this->setColour(ColourIDs::Roll::playheadSmallPlayback, s->getLassoBorderColour().
        interpolatedWith(s->getWhiteKeyColour(), this->isDarkTheme ? 0.5f : 0.25f).withAlpha(1.f));
    this->setColour(ColourIDs::Roll::playheadRecording,
        s->getLassoBorderColour().interpolatedWith(Colours::red, 0.5f).withAlpha(1.f));

    this->setColour(ColourIDs::Roll::cursorFill, s->getLassoBorderColour().withAlpha(1.f));
    this->setColour(ColourIDs::Roll::cursorShade, this->isDarkTheme ?
        s->getBlackKeyColour().darker(1.f).withAlpha(0.35f) :
        s->getWhiteKeyColour().brighter(0.5f).withAlpha(0.35f));

    this->setColour(ColourIDs::Roll::patternRowFill, s->getBlackKeyColour().brighter(0.02f));
    this->setColour(ColourIDs::Roll::trackHeaderFill, s->getWhiteKeyColour());
    this->setColour(ColourIDs::Roll::trackHeaderBorderLight,
        Colours::white.withAlpha(this->isDarkTheme ? 0.085f : 0.15f));
    this->setColour(ColourIDs::Roll::trackHeaderShadow,
        Colours::black.withAlpha(this->isDarkTheme ? 0.1f : 0.025f));
    this->setColour(ColourIDs::Roll::trackHeaderBorderDark,
        Colours::black.withAlpha(this->isDarkTheme ? 0.1f : 0.07f));

    this->setColour(ColourIDs::Roll::clipFill, this->isDarkTheme ?
        s->getBlackKeyColour().darker(1.f).withAlpha(0.77f) :
        s->getWhiteKeyColour().brighter(0.11f).withAlpha(0.88f));
    this->setColour(ColourIDs::Roll::clipForeground, textColour);

    this->setColour(ColourIDs::Roll::noteFill, textColour.interpolatedWith(Colours::white, 0.5f));
    this->setColour(ColourIDs::Roll::noteNameFill, this->isDarkTheme ?
        s->getBlackKeyColour().darker(0.4f) : s->getWhiteKeyColour().brighter(0.15f));
    this->setColour(ColourIDs::Roll::noteNameBorder,
        textColour.withAlpha(this->isDarkTheme ? 0.4f : 0.2f));
    this->setColour(ColourIDs::Roll::noteNameShadow, textColour.withAlpha(0.1f));

    this->setColour(ColourIDs::Roll::noteCutMark, s->getBlackKeyColour().darker(this->isDarkTheme ? 1.f : 0.05f));
    this->setColour(ColourIDs::Roll::noteCutMarkOutline, Colours::white.withAlpha(0.2f));
    this->setColour(ColourIDs::Roll::cuttingGuide, s->getLassoBorderColour().withAlpha(0.9f));
    this->setColour(ColourIDs::Roll::cuttingGuideOutline, s->getLassoBorderColour().contrasting().withAlpha(0.1f));
    this->setColour(ColourIDs::Roll::draggingGuide, s->getLassoBorderColour().withAlpha(0.420f));
    this->setColour(ColourIDs::Roll::draggingGuideShadow, s->getBlackKeyColour().withMultipliedAlpha(0.69f));
    this->setColour(ColourIDs::Roll::resizingGuideFill, s->getSidebarFillColour());
    this->setColour(ColourIDs::Roll::resizingGuideOutline, s->getLassoBorderColour().withAlpha(0.55f));
    this->setColour(ColourIDs::Roll::resizingGuideShadow,
        s->getBlackKeyColour().withMultipliedAlpha(this->isDarkTheme ? 0.420f : 0.69f));

    this->setColour(ColourIDs::TransportControl::recordInactive, Colours::transparentBlack);
    this->setColour(ColourIDs::TransportControl::recordHighlight, Colours::red.withAlpha(0.35f));
    this->setColour(ColourIDs::TransportControl::recordActive,
        s->getTimelineColour().darker(0.05f).interpolatedWith(Colours::red, 0.5f));

    this->setColour(ColourIDs::TransportControl::playInactive, Colours::white.withAlpha(0.035f));
    this->setColour(ColourIDs::TransportControl::playHighlight, Colours::white.withAlpha(0.075f));
    this->setColour(ColourIDs::TransportControl::playActive, Colours::white.withAlpha(0.1f));

    this->setColour(ColourIDs::Logo::fill,
        textColour.withMultipliedAlpha(this->isDarkTheme ? 0.2f : 0.5f));

    this->setColour(ColourIDs::AudioMonitor::foreground,
        textColour.withAlpha(this->isDarkTheme ? 0.5f : 1.f));

    this->setColour(ColourIDs::VersionControl::revisionConnector, textColour.withAlpha(0.2f));
    this->setColour(ColourIDs::VersionControl::revisionOutline, textColour.withAlpha(0.4f));
    this->setColour(ColourIDs::VersionControl::revisionFill, s->getPageFillColour().darker(0.02f));
    this->setColour(ColourIDs::VersionControl::revisionHighlight, this->isDarkTheme ?
        s->getPageFillColour().darker(0.15f) : s->getPageFillColour().brighter(0.035f));
    this->setColour(ColourIDs::VersionControl::stageSelectionFill, this->isDarkTheme ?
        s->getPageFillColour().darker(1.f).withAlpha(0.1f) :
        s->getPageFillColour().brighter(1.f).withAlpha(0.15f));

    this->setColour(ColourIDs::RenderProgressBar::fill, s->getSidebarFillColour().darker(0.15f));
    this->setColour(ColourIDs::RenderProgressBar::outline, textColour.withAlpha(0.4f));
    this->setColour(ColourIDs::RenderProgressBar::progress, textColour.contrasting().withAlpha(0.4f));
    this->setColour(ColourIDs::RenderProgressBar::waveform, textColour);

    this->setColour(ColourIDs::TapTempoControl::fill, textColour.withAlpha(0.015f));
    this->setColour(ColourIDs::TapTempoControl::fillHighlighted, textColour.withAlpha(0.05f));
    this->setColour(ColourIDs::TapTempoControl::outline, textColour.withAlpha(0.25f));

    const auto shadowIntensity = this->isDarkTheme ? 0.88f : 0.420f;
    this->setColour(ColourIDs::Shadows::fillLight, Colours::black.withAlpha(shadowIntensity * 0.05f));
    this->setColour(ColourIDs::Shadows::borderLight, Colours::black.withAlpha(shadowIntensity * 0.15f));
    this->setColour(ColourIDs::Shadows::fillNormal, Colours::black.withAlpha(shadowIntensity * 0.075f));
    this->setColour(ColourIDs::Shadows::borderNormal, Colours::black.withAlpha(shadowIntensity * 0.2f));
    this->setColour(ColourIDs::Shadows::fillHard, Colours::black.withAlpha(shadowIntensity * 0.125f));
    this->setColour(ColourIDs::Shadows::borderHard, Colours::black.withAlpha(shadowIntensity * 0.25f));

    // Pre-rendered image backgrounds:
    constexpr int w = 256;
    constexpr int h = 256;

    {
        this->pageBackgroundA = Image(Image::ARGB, w, h, true);
        Graphics g(this->pageBackgroundA);
        g.setColour(this->findColour(ColourIDs::Panel::pageFillA));
        g.fillAll();
        this->drawNoise(g, 0.25f);
    }

    {
        this->pageBackgroundB = Image(Image::ARGB, w, h, true);
        Graphics g(this->pageBackgroundB);
        g.setColour(this->findColour(ColourIDs::Panel::pageFillB));
        g.fillAll();
        this->drawNoise(g, 0.25f);
    }

    {
        this->sidebarBackground = Image(Image::ARGB, w, h, true);
        Graphics g(this->sidebarBackground);
        g.setColour(this->findColour(ColourIDs::Panel::sidebarFill));
        g.fillAll();
        this->drawNoise(g);
    }

    {
        this->bottomPanelBackground = Image(Image::ARGB, w, h, true);
        Graphics g(this->bottomPanelBackground);
        g.setColour(this->findColour(ColourIDs::Panel::bottomPanelFill));
        g.fillAll();
        this->drawNoise(g);
    }

    {
        this->headlineBackground = Image(Image::ARGB, w, h, true);
        Graphics g(this->headlineBackground);
        g.setColour(this->findColour(ColourIDs::Breadcrumbs::fill));
        g.fillAll();
        this->drawNoise(g);
    }

    {
        this->dialogBackground = Image(Image::ARGB, w, h, true);
        Graphics g(this->dialogBackground);
        g.setColour(this->findColour(ColourIDs::Dialog::fill));
        g.fillAll();
        this->drawNoise(g, 0.25f);
    }

    Icons::clearPrerenderedCache();
}
