/*
    This file is part of Helio music sequencer.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#pragma once

class RollBase;
class ProjectNode;

#include "CommandPaletteActionsProvider.h"
#include "CommandPaletteModel.h"

class CommandPaletteTextEditor final : public TextEditor
{
public:
    bool keyPressed(const KeyPress &key) override;
};

class CommandPalette final : public Component,
                             public TextEditor::Listener,
                             public ListBoxModel
{
public:

    CommandPalette(ProjectNode *project,
        RollBase *roll, const String &defaultText = {});

    ~CommandPalette();

    void resized() override;
    void parentHierarchyChanged() override;
    void handleCommandMessage(int commandId) override;
    bool keyPressed(const KeyPress &key) override;
    void inputAttemptWhenModal() override;

    static Array<KeyPress> getAllCommandPaletteHotkeys();

private:

    static constexpr auto rowHeight = 28;

    //===------------------------------------------------------------------===//
    // TextEditor::Listener
    //===------------------------------------------------------------------===//

    void textEditorTextChanged(TextEditor &) override;
    void textEditorReturnKeyPressed(TextEditor &) override;
    void textEditorEscapeKeyPressed(TextEditor &) override;
    void textEditorFocusLost(TextEditor &) override;

    //===------------------------------------------------------------------===//
    // ListBoxModel
    //===------------------------------------------------------------------===//

    int getNumRows() override;
    void paintListBoxItem(int rowNumber, Graphics &g, int w, int h, bool rowIsSelected) override;
    void selectedRowsChanged(int lastRowSelected) override;
    void listBoxItemClicked(int row, const MouseEvent &) override;
    void moveRowSelectionBy(int offset);

    void applySelectedCommand();
    int getHeightToFitActions();
    int getNumVisibleRows() const noexcept;

    void dismiss();
    void updatePosition();

    SafePointer<RollBase> roll;

    Array<WeakReference<CommandPaletteActionsProvider>> actionsProviders;
    WeakReference<CommandPaletteActionsProvider> currentActionsProvider;
    WeakReference<CommandPaletteActionsProvider> rootActionsProvider;

    UniquePointer<Component> background;
    UniquePointer<Component> shadowDown;
    UniquePointer<Component> shadowLeft;
    UniquePointer<Component> shadowRight;
    UniquePointer<CommandPaletteTextEditor> textEditor;
    UniquePointer<ListBox> actionsList;

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(CommandPalette)
};
