/*
    This file is part of Helio music sequencer.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Common.h"
#include "InstrumentEditorPin.h"
#include "InstrumentEditor.h"
#include "ColourIDs.h"

InstrumentEditorPin::InstrumentEditorPin(AudioProcessorGraph::NodeID nodeId, int index, bool isInput) :
    nodeId(nodeId),
    index(index),
    isInput(isInput)
{
    this->setSize(18, 18);
    this->setWantsKeyboardFocus(false);
    this->setMouseCursor(MouseCursor::PointingHandCursor);
}

void InstrumentEditorPin::paint(Graphics &g)
{
    const auto w = float(this->getWidth());
    const auto h = float(this->getHeight());
    const bool isMidiChannel = (this->index == Instrument::midiChannelNumber);

    g.setColour(findDefaultColour(ColourIDs::Instrument::pinShadow));
    g.drawEllipse(3.f, 4.f, w - 6.f, h - 6.f, 4.f);

    const int colourId = isMidiChannel ?
        ColourIDs::Instrument::midiNode : ColourIDs::Instrument::audioNode;
    g.setColour(findDefaultColour(colourId));
    g.drawEllipse(3.f, 3.f, w - 6.f, h - 6.f, 4.f);
}

void InstrumentEditorPin::mouseDown(const MouseEvent &e)
{
    const AudioProcessorGraph::NodeID sourceId(this->isInput ? 0 : this->nodeId.uid);
    const AudioProcessorGraph::NodeID destinationId(this->isInput ? this->nodeId.uid : 0);
    this->getParentEditor()->beginConnectorDrag(sourceId, this->index, destinationId, this->index, e);
    this->setMouseCursor(MouseCursor::DraggingHandCursor);
}

void InstrumentEditorPin::mouseDrag(const MouseEvent &e)
{
    this->getParentEditor()->dragConnector(e);
}

void InstrumentEditorPin::mouseUp(const MouseEvent &e)
{
    this->setMouseCursor(MouseCursor::PointingHandCursor);
    this->getParentEditor()->endDraggingConnector(e);
}

InstrumentEditor *InstrumentEditorPin::getParentEditor() const noexcept
{
    return findParentComponentOfClass<InstrumentEditor>();
}
