/*
    This file is part of Helio music sequencer.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#pragma once

#include "DialogBase.h"
#include "Meter.h"
#include "TimeSignatureEvent.h"
#include "MobileComboBox.h"
#include "UndoStack.h"
#include "TimeSignaturesSequence.h"
#include "MidiTrack.h"

class ProjectNode;
class MetronomeEditor;

class TimeSignatureDialog final : public DialogBase
{
public:

    TimeSignatureDialog(ProjectNode &project,
        WeakReference<MidiTrack> targetTrack,
        WeakReference<TimeSignaturesSequence> targetSequence,
        const TimeSignatureEvent &editedEvent,
        bool shouldAddNewEvent) noexcept;

    ~TimeSignatureDialog();

    static UniquePointer<Component> editingDialog(ProjectNode &project,
        const TimeSignatureEvent &event);

    static UniquePointer<Component> addingDialog(ProjectNode &project,
        WeakReference<TimeSignaturesSequence> tsSequence,
        float targetBeat);

    void resized() override;
    void parentHierarchyChanged() override;
    void parentSizeChanged() override;
    void handleCommandMessage(int commandId) override;
    bool keyPressed(const KeyPress &key) override;
    bool keyStateChanged(bool isKeyDown) override;

private:

    void dialogCancelAction() override;
    void dialogApplyAction() override;
    void dialogDeleteAction() override;

    const WeakReference<UndoStack> undoStack;

    // either of them will be nullptr:
    const WeakReference<MidiTrack> targetTrack;
    const WeakReference<TimeSignaturesSequence> targetSequence;

    // keeping a copy of original event so we can send undo actions (from -> to):
    TimeSignatureEvent originalEvent;
    // all current edits are here so callbacks can just modify them:
    TimeSignatureEvent editedEvent;

    const Array<Meter::Ptr> defaultMeters;

    Component *getPrimaryFocusTarget() override;

    void updateOkButtonState();
    void updateSize();

    enum class Mode
    {
        EditTrackTimeSignature,
        EditTimelineTimeSignature,
        AddTimelineTimeSignature
    };

    const Mode mode;

    bool hasMadeChanges = false;

    void sendEventChange(const TimeSignatureEvent &newEvent);
    void removeTimeSignature();

    UniquePointer<MobileComboBox::Container> presetsCombo;
    UniquePointer<Label> messageLabel;
    UniquePointer<TextButton> removeEventButton;
    UniquePointer<TextButton> okButton;
    UniquePointer<TextEditor> textEditor;
    UniquePointer<MetronomeEditor> metronomeEditor;

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (TimeSignatureDialog)
};
