/*
    This file is part of Helio music sequencer.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#include "Common.h"
#include "Diff.h"

namespace VCS
{

Diff::Diff(TrackedItem &diffTarget)
{
    this->description = diffTarget.getVCSName();
    this->vcsUuid = diffTarget.getUuid();
    this->displayColur = diffTarget.getRevisionDisplayColour();
    this->logic.reset(DiffLogic::createLogicCopy(diffTarget, *this));
}

bool Diff::hasAnyChanges() const
{
    return (this->getNumDeltas() > 0);
}

void Diff::applyDeltas(Array<DeltaDiff> &&deltaDiffs)
{
    for (auto &fullDelta : deltaDiffs)
    {
        this->deltas.add(fullDelta.delta.release());
        this->deltasData.add(fullDelta.deltaData);
    }
}

void Diff::applyDelta(DeltaDiff &&deltaDiff)
{
    this->deltas.add(deltaDiff.delta.release());
    this->deltasData.add(deltaDiff.deltaData);
}

void Diff::applyDelta(Delta *newDelta, SerializedData data)
{
    this->deltas.add(newDelta);
    this->deltasData.add(data);
}

void Diff::clear()
{
    this->deltas.clear();
    this->deltasData.clear();
}

//===----------------------------------------------------------------------===//
// TrackedItem
//===----------------------------------------------------------------------===//

int Diff::getNumDeltas() const
{
    return this->deltas.size();
}

Delta *Diff::getDelta(int index) const
{
    return this->deltas[index];
}

SerializedData Diff::getDeltaData(int deltaIndex) const
{
    return this->deltasData[deltaIndex];
}

String Diff::getVCSName() const
{
    return this->description;
}

DiffLogic *Diff::getDiffLogic() const
{
    return this->logic.get();
}

Colour Diff::getRevisionDisplayColour() const
{
    return this->displayColur;
}

}
