/*
    This file is part of Helio music sequencer.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#pragma once

#include "MidiEvent.h"
#include "MidiTrack.h"
#include "Meter.h"

// Time signatures are a bit different from other MidiEvents
// in a way that not only they can belong to a TimeSignaturesSequence
// at the timeline, but they also can belong to a MidiTrack to override
// the timeline's time signatures when this track is selected.

class TimeSignatureEvent final : public MidiEvent
{
public:

    // makes an invalid time signature
    TimeSignatureEvent() noexcept;

    TimeSignatureEvent(const TimeSignatureEvent &other) noexcept = default;
    TimeSignatureEvent &operator= (const TimeSignatureEvent &other) = default;

    TimeSignatureEvent(TimeSignatureEvent &&other) noexcept = default;
    TimeSignatureEvent &operator= (TimeSignatureEvent &&other) = default;

    explicit TimeSignatureEvent(WeakReference<MidiTrack> owner) noexcept;

    TimeSignatureEvent(WeakReference<MidiSequence> owner,
        const TimeSignatureEvent &parametersToCopy) noexcept;

    explicit TimeSignatureEvent(WeakReference<MidiSequence> owner,
        float newBeat = 0.f,
        int newNumerator = Globals::Defaults::timeSignatureNumerator,
        int newDenominator = Globals::Defaults::timeSignatureDenominator) noexcept;
    
    void exportMessages(MidiMessageSequence &outSequence, const Clip &clip,
        const KeyboardMapping &keyMap, double timeFactor) const noexcept override;

    TimeSignatureEvent withDeltaBeat(float beatOffset) const noexcept;
    TimeSignatureEvent withBeat(float newBeat) const noexcept;
    TimeSignatureEvent withNumerator(const int newNumerator) const noexcept;
    TimeSignatureEvent withDenominator(const int newDenominator) const noexcept;
    TimeSignatureEvent withMeter(const Meter &meter) const noexcept;
    TimeSignatureEvent withMetronome(const MetronomeScheme &scheme) const noexcept;
    TimeSignatureEvent withParameters(const SerializedData &parameters) const noexcept;

    // a unique id can be generated by a sequence
    TimeSignatureEvent withNewId() const noexcept;
    // or it can be set by time signature aggregator:
    TimeSignatureEvent withId(MidiEvent::Id id) const noexcept;

    //===------------------------------------------------------------------===//
    // Accessors
    //===------------------------------------------------------------------===//

    bool isValid() const noexcept override;

    int getTrackControllerNumber() const noexcept override;
    int getTrackChannel() const noexcept override;
    Colour getTrackColour() const noexcept override;
    
    int getNumerator() const noexcept;
    int getDenominator() const noexcept;
    float getBarLengthInBeats() const noexcept;
    float getDenominatorInBeats() const noexcept;
    const Meter &getMeter() const noexcept;

    WeakReference<MidiTrack> getTrack() const noexcept;

    String toString() const noexcept;

    //===------------------------------------------------------------------===//
    // Serializable
    //===------------------------------------------------------------------===//

    SerializedData serialize() const override;
    void deserialize(const SerializedData &data) override;
    void reset() noexcept override;

    //===------------------------------------------------------------------===//
    // Helpers
    //===------------------------------------------------------------------===//

    void applyChanges(const TimeSignatureEvent &parameters) noexcept;

private:

    WeakReference<MidiTrack> track = nullptr;

    Meter meter;

    JUCE_LEAK_DETECTOR(TimeSignatureEvent)
};
